/* Copyright 2006 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest;

import java.util.ArrayList;
import java.util.List;

import jp.sourceforge.jlogtest.util.xom.SpecificChildElementIterator;

import nu.xom.Attribute;
import nu.xom.Element;
import nu.xom.ParsingException;

public class LogSequence implements ILoggable {
	
	public static final String SEQUENCE = "sequence";
	public static final String NAME = "name";
	
	private final List<ILog> seqElements;
	private final String name;
	private final TargetSequence targetSeq;
	
	public LogSequence(final TargetSequence targetSeq) {
		seqElements = new ArrayList<ILog>();
		// データが二重化しており、本当は好ましくない書き方
		this.name = targetSeq.getName(); 
		this.targetSeq = targetSeq;
	}
	
	//
	// XMLからオブジェクトを生成する機能
	//

	public LogSequence(final Element sequence) throws ParsingException {
		if (!SEQUENCE.equals(sequence.getLocalName()))
			throw new ParsingException("no " + SEQUENCE + " element");
		
		name = sequence.getAttributeValue(NAME);
		if (name == null)
			throw new ParsingException("no " + NAME + " attribute");
		
		seqElements = new ArrayList<ILog>();
		try {
			for (final Element child :
				new SpecificChildElementIterator(sequence, JclLogInfo.JCLLOG, LogInfo.LOG))
				seqElements.add(getLog(child));
		} catch (Exception e) {
			throw new ParsingException("parse failed", e);
		}
		// 特に深い意味はない
		// （このコンストラクタで初期化したとき、addLogメソッドは用いられないはずであるので、
		// 何を代入しても構わない）
		targetSeq = null;
	}

	private static ILog getLog(final Element child) throws ParsingException {
		assert child != null;
		
		final String name = child.getLocalName();
		if (JclLogInfo.JCLLOG.equals(name)) {
			return new JclLogInfo(child);
		} else if (LogInfo.LOG.equals(name)) {
			return new LogInfo(child);
		}
		assert false;
		throw new IllegalArgumentException();
	}
	
	//
	// ここまで
	//

	public void addLog(final ILog log) {
		if (targetSeq.contains(log))
			seqElements.add(log);
	}

	public ILog[] getLogs() {
		return seqElements.toArray(new ILog[]{});
	}

	public String getName() {
		return name;
	}
	
	//
	// XMLでログを出力する機能
	//

	public Element asXml() {
		return createLogsElement();
	}

	private Element createLogsElement() {
		final Element sequence = new Element(SEQUENCE);
		if (name != null)
			sequence.addAttribute(new Attribute(NAME, name));
		for (final ILog aLog : seqElements) {
			sequence.appendChild(aLog.asXml());
		}
		return sequence;
	}
	
	//
	// ここまで
	//
}
