/* Copyright 2006 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest;

import static jp.sourceforge.jlogtest.util.xom.NodeCreator.getPreserveSpaceAttribute;
import jp.sourceforge.jlogtest.util.java.HashGen;
import jp.sourceforge.jlogtest.util.xom.ChildSelector;
import nu.xom.Attribute;
import nu.xom.Element;
import nu.xom.Node;
import nu.xom.ParsingException;

public class LogInfo implements ILog {
	
	public static final String LOG = "log";
	private static final String NAME = "name";
	private static final String TEXT = "text";

	private final String text;
	private final String name;
	
	public LogInfo(final String text, final String name) {
		this.text = text;
		this.name = name;
	}
	
	public LogInfo(final Element log) throws ParsingException {
		if (!LOG.equals(log.getLocalName()))
			throw new ParsingException("no " + LOG + " element");
		
		try {
			text = getText(log);
			name = log.getAttributeValue(NAME);
			
			if (text == null || name == null)
				throw new ParsingException("parse failed");
		} catch (Exception e) {
			throw new ParsingException("parse failed", e);
		}
	}
	
	private static String getText(final Element log) throws ParsingException {
		assert LOG.equals(log.getLocalName());
		
		final ChildSelector selector = new ChildSelector(log);
		if (!selector.isUnique(TEXT))
			throw new ParsingException("no " + LOG + " element");
		
		final Element text = selector.getUniqueElement(TEXT);
		return text.getValue();
	}
	
	public Element asXml() {
		final Element log = new Element(LOG);
		log.addAttribute(new Attribute(NAME, name));
		log.appendChild(getXmlTextPart());
		return log;
	}

	private Node getXmlTextPart() {
		final Element text = new Element(TEXT);
		text.addAttribute(getPreserveSpaceAttribute());
		text.appendChild(this.text);
		return text;
	}

	public String asOutputString() {
		return String.format("name : %s\ntext : '%s'", name, text);
	}

	public String getText() {
		return text;
	}

	public String getName() {
		return name;
	}

	@Override
	public boolean equals(final Object obj) {
		if (!(obj instanceof LogInfo))
			return false;
		
		final LogInfo info = (LogInfo)obj;
		return text.equals(info.text) && name.equals(info.name);
	}

	@Override
	public int hashCode() {
		return HashGen.gen(text, name);
	}
}
