/* Copyright 2006 Harai Akihiro.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.jlogtest;

import static jp.sourceforge.jlogtest.util.xom.NodeCreator.getPreserveSpaceAttribute;
import jp.sourceforge.jlogtest.util.java.HashGen;
import jp.sourceforge.jlogtest.util.xom.ChildSelector;
import nu.xom.Attribute;
import nu.xom.Element;
import nu.xom.Node;
import nu.xom.ParsingException;

public class JclLogInfo implements ILog {
	
	public static final String JCLLOG = "jcllog";
	private static final String CLASS = "class";
	private static final String LEVEL = "level";
	private static final String TEXT = "text";
	
	private final String text;
	private final String className;
	private final JclLogLevel level;

	public JclLogInfo(final String text, final String className, final JclLogLevel level) {
		this.text = text;
		this.className = className;
		this.level = level;
	}
	
	public JclLogInfo(final Element jcllog) throws ParsingException {
		if (!JCLLOG.equals(jcllog.getLocalName()))
			throw new ParsingException("no " + JCLLOG + " element");
		
		try {
			text = getText(jcllog);
			className = jcllog.getAttributeValue(CLASS);
			level = JclLogLevel.valueOf(jcllog.getAttributeValue(LEVEL));
			
			if (text == null || className == null || level == null)
				throw new ParsingException("parse failed");
		} catch (Exception e) {
			throw new ParsingException("parse failed", e);
		}
	}
	
	private static String getText(final Element jcllog) throws ParsingException {
		assert JCLLOG.equals(jcllog.getLocalName());
		
		final ChildSelector selector = new ChildSelector(jcllog);
		if (!selector.isUnique(TEXT))
			throw new ParsingException("no " + JCLLOG + " element");
		
		final Element text = selector.getUniqueElement(TEXT);
		return text.getValue();
	}
	
	public String getText() {
		return text;
	}
	
	public String getClassName() {
		return className;
	}

	public JclLogLevel getJclLogLevel() {
		return level;
	}

	public Element asXml() {
		final Element log = new Element(JCLLOG);
		log.addAttribute(new Attribute(CLASS, className));
		log.addAttribute(new Attribute(LEVEL, level.toString()));
		log.appendChild(getXmlTextPart());
		return log;
	}

	private Node getXmlTextPart() {
		final Element text = new Element(TEXT);
		text.addAttribute(getPreserveSpaceAttribute());
		text.appendChild(this.text);
		return text;
	}
	
	@Override
	public boolean equals(final Object obj) {
		if (!(obj instanceof JclLogInfo))
			return false;
		
		final JclLogInfo info = (JclLogInfo)obj;
		return text.equals(info.text) &&
				className.equals(info.className) &&
				level.equals(info.level);
	}

	@Override
	public int hashCode() {
		return HashGen.gen(text, className, level);
	}

	public String asOutputString() {
		return String.format(
				"class : %s\nlevel : %s\ntext : '%s'", className, level, text);
	}
}
