/*
 * HTML sequence
 *
 * Copyright(c) 2009 olyutorskii
 * $Id: HtmlSequence.java 394 2009-02-13 15:29:26Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.net.URL;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;

/**
 * HTML本文。
 * 任意のCharSequenceをラップする。
 * 由来となるURLとエンコーディングエラーを含む。
 */
public class HtmlSequence implements CharSequence{

    /**
     * エンコードエラーのダンプ
     * @param errorInfo エラー情報
     */
    public static void dumpEncodeError(SortedMap<Long, byte[]> errorInfo){
        if(errorInfo.size() > 0){
            Set<Long> keySet = errorInfo.keySet();
            for(long pos : keySet){
                System.out.print(" " + pos);
                byte[] bytes = errorInfo.get(pos);
                for(byte b : bytes){
                    System.out.print(" " + b);
                }
            }
        }
        return;
    }

    private final URL url;
    private final CharSequence html;
    private final SortedMap<Long, byte[]> encodeError;

    /**
     * コンストラクタ
     * @param url 由来のURL
     * @param html HTML本文
     * @param encodeError エンコーディングエラー
     *         @see SjisInputStream#getEncodeError()
     * @throws java.lang.NullPointerException 引数がnull
     */
    public HtmlSequence(URL url,
                          CharSequence html,
                          SortedMap<Long, byte[]> encodeError )
        throws NullPointerException{
        if(url == null || html == null || encodeError == null){
            throw new NullPointerException();
        }
        this.url = url;
        this.html = html;
        this.encodeError = encodeError;
        return;
    }

    /**
     * URLを返す。
     * @return URL
     */
    public URL getURL(){
        return this.url;
    }

    /**
     * エンコードエラーを返す。
     * @return エンコードエラー @see SjisInputStream#getEncodeError()
     */
    protected SortedMap<Long, byte[]> getEncodeError(){
        return this.encodeError;
    }

    /**
     * エンコードエラーの部分ビューを返す。
     * エラー位置はfromKeyを起点に割り引かれる。
     * @param fromKey 範囲開始（これを含む）
     * @param toKey 範囲終了（これを含まない）
     * @return 部分ビュー
     */
    public SortedMap<Long, byte[]> subEncodeError(long fromKey, long toKey){
        SortedMap<Long, byte[]> result = new TreeMap<Long, byte[]>();

        Set<Long> keySet = this.encodeError.keySet();
        for(long pos : keySet){
            if(pos < fromKey) continue;
            if(toKey <= pos) break;
            byte[] value = this.encodeError.get(pos);
            byte[] newvalue = new byte[value.length];
            System.arraycopy(value, 0, newvalue, 0, value.length);
            result.put(pos - fromKey, newvalue);
        }

        return result;
    }

    /**
     * {@inheritDoc}
     * @param index {@inheritDoc}
     * @return {@inheritDoc}
     */
    public char charAt(int index){
        return this.html.charAt(index);
    }

    /**
     * {@inheritDoc}
     * @return {@inheritDoc}
     */
    public int length(){
        return this.html.length();
    }

    /**
     * {@inheritDoc}
     * @param start {@inheritDoc}
     * @param end {@inheritDoc}
     * @return {@inheritDoc}
     */
    public CharSequence subSequence(int start, int end){
        return this.html.subSequence(start, end);
    }

    /**
     * {@inheritDoc}
     * @return {@inheritDoc}
     */
    @Override
    public String toString(){
        return this.html.toString();
    }
}
