/*
 * MVC controller
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: Controller.java 407 2009-02-27 13:26:30Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Font;
import java.awt.Frame;
import java.awt.LayoutManager;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.MouseAdapter;
import java.awt.font.FontRenderContext;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.regex.Pattern;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JToolBar;
import javax.swing.JTree;
import javax.swing.LookAndFeel;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.event.TreeWillExpandListener;
import javax.swing.tree.TreePath;

/**
 * いわゆるMVCでいうとこのコントローラ
 */
public class Controller
        implements ActionListener,
                   TreeWillExpandListener,
                   TreeSelectionListener,
                   ChangeListener,
                   AnchorHitListener {

    private final ActionManager actionManager;
    private final TopView topView;
    private final LandsModel model;

    private final FilterPanel filterFrame;
    private final LogFrame showlogFrame;
    private final FontChooser fontChooser;
    private final FindPanel findPanel;
    private final TalkPreview talkPreview;
    private JFrame helpFrame;
    private AccountPanel accountFrame;
    private DaySummary daySummaryPanel;
    private Map<Window, Boolean> windowMap = new HashMap<Window, Boolean>();

    private volatile boolean isBusyNow;

    private JFrame topFrame = null;

    /**
     * コントローラの生成。
     * @param actionManager アクション管理
     * @param topView 最上位ビュー
     * @param model 最上位データモデル
     */
    public Controller(ActionManager actionManager,
                       TopView topView,
                       LandsModel model){
        super();

        this.actionManager = actionManager;
        this.topView = topView;
        this.model = model;

        JToolBar toolbar = this.actionManager.getBrowseToolBar();
        this.topView.setBrowseToolBar(toolbar);

        this.actionManager.addActionListener(this);

        JTree treeView = this.topView.getTreeView();
        treeView.setModel(this.model);
        treeView.addTreeWillExpandListener(this);
        treeView.addTreeSelectionListener(this);

        this.topView.getTabBrowser().addChangeListener(this);
        this.topView.getTabBrowser().addActionListener(this);
        this.topView.getTabBrowser().addAnchorHitListener(this);

        JButton reloadVillageListButton = this.topView
                                         .getLandsTree()
                                         .getReloadVillageListButton();
        reloadVillageListButton.addActionListener(this);
        reloadVillageListButton.setEnabled(false);

        this.filterFrame = new FilterPanel(this.topFrame);
        this.filterFrame.addChangeListener(this);
        this.filterFrame.pack();
        this.filterFrame.setVisible(false);

        this.showlogFrame = new LogFrame(this.topFrame);
        this.showlogFrame.pack();
        this.showlogFrame.setSize(600,250);
        this.showlogFrame.setLocationByPlatform(true);
        this.showlogFrame.setVisible(false);
        if(Jindolf.hasLoggingPermission()){
            Handler newHandler = this.showlogFrame.getHandler();
            Jindolf.logger.addHandler(newHandler);
            Handler[] handlers = Jindolf.logger.getHandlers();
            for(Handler handler : handlers){
                if( ! (handler instanceof PileHandler) ) continue;
                PileHandler pile = (PileHandler) handler;
                pile.delegate(newHandler);
                pile.close();
            }
        }

        this.talkPreview = new TalkPreview(this.topFrame);
        this.talkPreview.pack();
        this.talkPreview.setSize(700, 500);
        this.talkPreview.setVisible(false);

        this.fontChooser = new FontChooser(this.topFrame);
        this.fontChooser.pack();
        this.fontChooser.setSize(450, 450);
        this.fontChooser.setVisible(false);
        initFontChooser();

        this.findPanel = new FindPanel(this.topFrame);
        this.findPanel.pack();
        this.findPanel.setVisible(false);

        this.windowMap.put(this.filterFrame,  true);
        this.windowMap.put(this.showlogFrame, false);
        this.windowMap.put(this.talkPreview,  false);
        this.windowMap.put(this.fontChooser,  false);
        this.windowMap.put(this.findPanel,    true);

        this.topView
            .getTabBrowser()
            .setFontInfo(this.fontChooser.getSelectedFont(),
                         this.fontChooser.getFontRenderContext());
        this.talkPreview.setTextFont(this.fontChooser.getSelectedFont());

        return;
    }

    /**
     * トップフレームを生成する。
     * @return トップフレーム
     */
    @SuppressWarnings("serial")
    public JFrame createTopFrame(){
        this.topFrame = new JFrame();

        Container content = this.topFrame.getContentPane();
        LayoutManager layout = new BorderLayout();
        content.setLayout(layout);
        content.add(this.topView, BorderLayout.CENTER);

        Component glassPane = new JComponent() {};
        glassPane.addMouseListener(new MouseAdapter() {});
        glassPane.addKeyListener(new KeyAdapter() {});
        this.topFrame.setGlassPane(glassPane);

        this.topFrame.setJMenuBar(this.actionManager.getMenuBar());
        setFrameTitle(null);

        this.windowMap.put(this.topFrame, false);

        return this.topFrame;
    }

    /**
     * フォント選択パネルの初期化
     */
    private void initFontChooser(){
        FontRenderContext oldContext = this.fontChooser.getFontRenderContext();
        boolean isAntiAliased         = oldContext.isAntiAliased();
        boolean usesFractionalMetrics = oldContext.usesFractionalMetrics();
        if(AppSetting.getInitfont() != null){
            Font newFont = Font.decode(AppSetting.getInitfont());
            if( ! newFont.equals(this.fontChooser.getSelectedFont()) ){
                this.fontChooser.setSelectedFont(newFont);
            }
            if(FontChooser.guessBitmapFont(newFont)){
                isAntiAliased         = false;
                usesFractionalMetrics = false;
            }else{
                isAntiAliased         = true;
                usesFractionalMetrics = true;
            }
        }
        if(AppSetting.useAntialias() != null){
            isAntiAliased = AppSetting.useAntialias().booleanValue();
        }
        if(AppSetting.useFractional() != null){
            usesFractionalMetrics = AppSetting.useFractional().booleanValue();
        }
        FontRenderContext renderContext =
                new FontRenderContext(oldContext.getTransform(),
                                      isAntiAliased,
                                      usesFractionalMetrics );
        this.fontChooser.setFontRenderContext(renderContext);

        return;
    }

    /**
     * About画面を表示する。
     */
    private void actionAbout(){
        String message =
                Jindolf.TITLE
                + "   Version " + Jindolf.VERSION + "\n"
                + Jindolf.COPYRIGHT + "\n"
                + "ライセンス: " + Jindolf.LICENSE + "\n"
                + "連絡先: " + Jindolf.CONTACT;

        JOptionPane pane = new JOptionPane(message,
                                           JOptionPane.INFORMATION_MESSAGE,
                                           JOptionPane.DEFAULT_OPTION,
                                           GUIUtils.getLogoIcon());

        JDialog dialog = pane.createDialog(this.topFrame,
                                           Jindolf.TITLE + "について");

        dialog.pack();
        dialog.setVisible(true);
        dialog.dispose();

        return;
    }

    /**
     * アプリ終了。
     */
    private void actionExit(){
        Jindolf.exit(0);
    }

    /**
     * Help画面を表示する。
     */
    private void actionHelp(){
        if(this.helpFrame != null){                 // show Toggle
            toggleWindow(this.helpFrame);
            return;
        }

        this.helpFrame = new HelpFrame();
        this.helpFrame.pack();
        this.helpFrame.setSize(450, 450);

        this.windowMap.put(this.helpFrame, false);

        this.helpFrame.setVisible(true);

        return;
    }

    /**
     * 村をWebブラウザで表示する。
     */
    private void actionShowWebVillage(){
        TabBrowser browser = this.topView.getTabBrowser();
        Village village = browser.getVillage();
        if(village == null) return;

        Land land = village.getParentLand();
        ServerAccess server = land.getServerAccess();

        URL url = server.getVillageURL(village);

        String urlText = url.toString();
        if(village.getState() != Village.State.GAMEOVER){
            urlText += "#bottom";
        }

        WebIPCDialog.showDialog(this.topFrame, urlText);

        return;
    }

    /**
     * 村に対応するまとめサイトをWebブラウザで表示する。
     */
    private void actionShowWebWiki(){
        TabBrowser browser = this.topView.getTabBrowser();
        Village village = browser.getVillage();
        if(village == null) return;

        String villageName = village.getVillageName();

        String url = "http://wolfbbs.jp/" + villageName + "%C2%BC.html";

        WebIPCDialog.showDialog(this.topFrame, url.toString());

        return;
    }

    /**
     * 村に対応するキャスト紹介表ジェネレーターをWebブラウザで表示する。
     */
    private void actionShowWebCast(){
        TabBrowser browser = this.topView.getTabBrowser();
        Village village = browser.getVillage();
        if(village == null) return;

        Land land = village.getParentLand();
        ServerAccess server = land.getServerAccess();

        URL villageUrl = server.getVillageURL(village);

        String url = "http://hon5.com/jinro/";

        try{
            url += "?u=" + URLEncoder.encode(villageUrl.toString(), "UTF-8");
        }catch(UnsupportedEncodingException e){
            return;
        }

        url += "&s=1";

        WebIPCDialog.showDialog(this.topFrame, url.toString());

        return;
    }

    /**
     * 日(Period)をWebブラウザで表示する。
     */
    private void actionShowWebDay(){
        PeriodView periodView = currentPeriodView();
        if(periodView == null) return;

        Period period = periodView.getPeriod();
        if(period == null) return;

        TabBrowser browser = this.topView.getTabBrowser();
        Village village = browser.getVillage();
        if(village == null) return;

        Land land = village.getParentLand();
        ServerAccess server = land.getServerAccess();

        URL url = server.getPeriodURL(period);

        String urlText = url.toString();
        if(period.isHot()) urlText += "#bottom";

        WebIPCDialog.showDialog(this.topFrame, urlText);

        return;
    }

    /**
     * 個別の発言をWebブラウザで表示する。
     */
    private void actionShowWebTalk(){
        TabBrowser browser = this.topView.getTabBrowser();
        Village village = browser.getVillage();
        if(village == null) return;

        PeriodView periodView = currentPeriodView();
        if(periodView == null) return;

        Discussion discussion = periodView.getDiscussion();
        Talk talk = discussion.getPopupedTalk();
        if(talk == null) return;

        Period period = periodView.getPeriod();
        if(period == null) return;

        Land land = village.getParentLand();
        ServerAccess server = land.getServerAccess();

        URL url = server.getPeriodURL(period);

        String urlText = url.toString();
        urlText += "#" + talk.getMessageID();
        WebIPCDialog.showDialog(this.topFrame, urlText);

        return;
    }

    /**
     * ポータルサイトをWebブラウザで表示する。
     */
    private void actionShowPortal(){
        WebIPCDialog.showDialog(this.topFrame, Jindolf.CONTACT);
        return;
    }

    /**
     * L&Fの変更を行う。
     */
    // TODO Nimbus対応
    private void actionChangeLaF(){
        String className = this.actionManager.getSelectedLookAndFeel();

        LookAndFeel lnf;
        try{
            Class<?> lnfClass = Class.forName(className);
            lnf = (LookAndFeel)( lnfClass.newInstance() );
        }catch(Exception e){
            String message = "このLook&Feel["
                            + className
                            + "]を読み込む事ができません。";
            Jindolf.logger.log(Level.WARNING, message, e);
            JOptionPane.showMessageDialog(
                this.topFrame,
                message,
                "Look&Feel - " + Jindolf.TITLE,
                JOptionPane.WARNING_MESSAGE );
            return;
        }

        try{
            UIManager.setLookAndFeel(lnf);
        }catch(UnsupportedLookAndFeelException e){
            String message = "このLook&Feel["
                            + lnf.getName()
                            + "]はサポートされていません。";
            Jindolf.logger.log(Level.WARNING, message, e);
            JOptionPane.showMessageDialog(
                this.topFrame,
                message,
                "Look&Feel - " + Jindolf.TITLE,
                JOptionPane.WARNING_MESSAGE );
            return;
        }

        Jindolf.logger.info(
                "Look&Feelが["
                +lnf.getName()
                +"]に変更されました。");

        final Runnable updateUITask = new Runnable(){
            public void run(){
                Set<Window> windows = Controller.this.windowMap.keySet();
                for(Window window : windows){
                    SwingUtilities.updateComponentTreeUI(window);
                    window.validate();
                    boolean needPack = Controller.this.windowMap.get(window);
                    if(needPack){
                        window.pack();
                    }
                }

                return;
            }
        };

        Executor executor = Executors.newCachedThreadPool();
        executor.execute(new Runnable(){
            public void run(){
                setBusy(true);
                updateStatusBar("Look&Feelを更新中…");
                try{
                    SwingUtilities.invokeAndWait(updateUITask);
                }catch(Exception e){
                    Jindolf.logger.log(
                            Level.WARNING,
                            "Look&Feelの更新に失敗しました。",
                            e );
                }finally{
                    updateStatusBar("Look&Feelが更新されました");
                    setBusy(false);
                }
                return;
            }
        });

        return;
    }

    /**
     * 発言フィルタ画面を表示する。
     */
    private void actionShowFilter(){
        toggleWindow(this.filterFrame);
        return;
    }

    /**
     * アカウント管理画面を表示する。
     */
    private void actionShowAccount(){
        if(this.accountFrame != null){                 // show Toggle
            toggleWindow(this.accountFrame);
            return;
        }

        this.accountFrame = new AccountPanel(this.topFrame, this.model);
        this.accountFrame.pack();
        this.accountFrame.setVisible(true);

        this.windowMap.put(this.accountFrame, true);

        return;
    }

    /**
     * ログ表示画面を表示する。
     */
    private void actionShowLog(){
        toggleWindow(this.showlogFrame);
        return;
    }

    /**
     * 発言エディタを表示する。
     */
    private void actionTalkPreview(){
        toggleWindow(this.talkPreview);
        return;
    }

    /**
     * 発言表示フォント選択画面を表示する。
     */
    private void actionFontSelect(){
        this.fontChooser.setVisible(true);
        if(this.fontChooser.isCanceled()) return;

        final Font newFont = this.fontChooser.getSelectedFont();
        final FontRenderContext newContext =
                this.fontChooser.getFontRenderContext();

        Executor executor = Executors.newCachedThreadPool();
        executor.execute(new Runnable(){
            public void run(){
                setBusy(true);
                updateStatusBar("発言表示フォントを変更中…");
                try{
                    TabBrowser tabBrowser =
                            Controller.this.topView.getTabBrowser();
                    tabBrowser.setFontInfo(newFont, newContext);
                    Controller.this.talkPreview.setTextFont(newFont);
                }catch(Exception e){
                    Jindolf.logger.log(
                            Level.WARNING,
                            "発言表示フォントの変更に失敗しました。",
                            e );
                }finally{
                    updateStatusBar(
                            "発言表示フォントが変更されました");
                    setBusy(false);
                }
                return;
            }
        });

        return;
    }

    /**
     * 検索パネルを表示する。
     */
    private void actionShowFind(){
        this.findPanel.setVisible(true);
        if(this.findPanel.isCanceled()){
            updateFindPanel();
            return;
        }
        if(this.findPanel.isBulkSearch()){
            bulkSearch();
        }else{
            regexSearch();
        }
        return;
    }

    /**
     * 検索処理。
     */
    private void regexSearch(){
        Discussion discussion = currentDiscussion();
        if(discussion == null) return;

        RegexPattern regPattern = this.findPanel.getRegexPattern();
        int hits = discussion.setRegexPattern(regPattern);

        String hitMessage = "［" + hits + "］件ヒットしました";
        updateStatusBar(hitMessage);

        String loginfo = "";
        if(regPattern != null){
            Pattern pattern = regPattern.getPattern();
            if(pattern != null){
                loginfo = "正規表現 " + pattern.pattern() + " に";
            }
        }
        loginfo += hitMessage;
        Jindolf.logger.info(loginfo);

        return;
    }

    /**
     * 一括検索処理
     */
    private void bulkSearch(){
        Executor executor = Executors.newCachedThreadPool();
        executor.execute(new Runnable(){
            public void run(){
                taskBulkSearch();
                return;
            }
        });
    }

    /**
     * 一括検索処理。ヘビータスク版
     */
    private void taskBulkSearch(){
        taskLoadAllPeriod();
        int totalhits = 0;
        RegexPattern regPattern = this.findPanel.getRegexPattern();
        StringBuilder hitDesc = new StringBuilder();
        TabBrowser browser = this.topView.getTabBrowser();
        for(PeriodView periodView : browser.getPeriodViewList()){
            Discussion discussion = periodView.getDiscussion();
            int hits = discussion.setRegexPattern(regPattern);
            totalhits += hits;

            if(hits > 0){
                Period period = discussion.getPeriod();
                hitDesc.append(' ').append(period.getDay()).append("d:");
                hitDesc.append(hits).append("件");
            }
        }
        String hitMessage =
                  "［" + totalhits + "］件ヒットしました。"
                + hitDesc.toString();
        updateStatusBar(hitMessage);

        String loginfo = "";
        if(regPattern != null){
            Pattern pattern = regPattern.getPattern();
            if(pattern != null){
                loginfo = "正規表現 " + pattern.pattern() + " に";
            }
        }
        loginfo += hitMessage;
        Jindolf.logger.info(loginfo);

        return;
    }

    /**
     * 検索パネルに現在選択中のPeriodを反映させる。
     */
    private void updateFindPanel(){
        Discussion discussion = currentDiscussion();
        if(discussion == null) return;
        RegexPattern pattern = discussion.getRegexPattern();
        this.findPanel.setRegexPattern(pattern);
        return;
    }

    /**
     * 発言集計パネルを表示
     */
    private void actionDaySummary(){
        PeriodView periodView = currentPeriodView();
        if(periodView == null) return;

        Period period = periodView.getPeriod();
        if(period == null) return;

        if(this.daySummaryPanel == null){
            this.daySummaryPanel = new DaySummary(this.topFrame);
            this.daySummaryPanel.pack();
            this.daySummaryPanel.setSize(400, 500);
        }

        this.daySummaryPanel.summaryPeriod(period);
        this.daySummaryPanel.setVisible(true);

        this.windowMap.put(this.daySummaryPanel, false);

        return;
    }

    /**
     * 表示中PeriodをCSVファイルへエクスポートする。
     */
    private void actionDayExportCsv(){
        PeriodView periodView = currentPeriodView();
        if(periodView == null) return;

        Period period = periodView.getPeriod();
        if(period == null) return;

        File file = CsvExporter.exportPeriod(period, this.filterFrame);
        if(file != null){
            String message = "CSVファイル("
                            +file.getName()
                            +")へのエクスポートが完了しました";
            updateStatusBar(message);
        }

        // TODO 長そうなジョブなら別スレッドにした方がいいか？

        return;
    }

    /**
     * 検索結果の次候補へジャンプ
     */
    private void actionSearchNext(){
        Discussion discussion = currentDiscussion();
        if(discussion == null) return;

        discussion.nextHotTarget();

        return;
    }

    /**
     * 検索結果の全候補へジャンプ
     */
    private void actionSearchPrev(){
        Discussion discussion = currentDiscussion();
        if(discussion == null) return;

        discussion.prevHotTarget();

        return;
    }

    /**
     * Period表示の強制再更新処理。
     */
    private void actionReloadPeriod(){
        updatePeriod(true);
        return;
    }

    /**
     * 全日程の一括ロード。
     */
    private void actionLoadAllPeriod(){
        Executor executor = Executors.newCachedThreadPool();
        executor.execute(new Runnable(){
            public void run(){
                taskLoadAllPeriod();
                return;
            }
        });

        return;
    }

    /**
     * 全日程の一括ロード。ヘビータスク版
     */
    private void taskLoadAllPeriod(){
        setBusy(true);
        updateStatusBar("一括読み込み開始");
        try{
            TabBrowser browser = this.topView.getTabBrowser();
            Village village = browser.getVillage();
            if(village == null) return;
            for(PeriodView periodView : browser.getPeriodViewList()){
                Period period = periodView.getPeriod();
                if(period == null) continue;
                String message =
                        "" + period.getDay() + "日目の"
                        +"データを読み込んでいます";
                updateStatusBar(message);
                try{
                    period.updatePeriod();
                }catch(IOException e){
                    showNetworkError(village, e);
                    return;
                }
                periodView.showTopics();
            }
        }finally{
            updateStatusBar("一括読み込み完了");
            setBusy(false);
        }
        return;
    }

    /**
     * 村一覧の再読み込み
     */
    private void actionReloadVillageList(){
        JTree tree = this.topView.getTreeView();
        TreePath path = tree.getSelectionPath();
        if(path == null) return;

        Land land = null;
        for(int ct=0; ct<path.getPathCount(); ct++){
            Object obj = path.getPathComponent(ct);
            if(obj instanceof Land){
                land = (Land) obj;
                break;
            }
        }
        if(land == null) return;

        this.topView.showInitPanel();

        execReloadVillageList(land);

        return;
    }

    /**
     * 選択文字列をクリップボードにコピーする。
     */
    private void actionCopySelected(){
        Discussion discussion = currentDiscussion();
        if(discussion == null) return;

        CharSequence copied = discussion.copySelected();
        if(copied == null) return;

        copied = StringUtils.suppressString(copied);
        updateStatusBar(
                "[" + copied + "]をクリップボードにコピーしました");
        return;
    }

    /**
     * 一発言のみクリップボードにコピーする。
     */
    private void actionCopyTalk(){
        Discussion discussion = currentDiscussion();
        if(discussion == null) return;

        CharSequence copied = discussion.copyTalk();
        if(copied == null) return;

        copied = StringUtils.suppressString(copied);
        updateStatusBar(
                "[" + copied + "]をクリップボードにコピーしました");
        return;
    }

    /**
     * アンカーにジャンプする。
     */
    private void actionJumpAnchor(){
        PeriodView periodView = currentPeriodView();
        if(periodView == null) return;
        Discussion discussion = periodView.getDiscussion();

        final TabBrowser browser = this.topView.getTabBrowser();
        final Village village = browser.getVillage();
        final Anchor anchor = discussion.getPopupedAnchor();
        if(anchor == null) return;

        final int targetDay = anchor.getDay();
        final int tabIndex = browser.periodDaysToTabIndex(targetDay);

        Executor executor = Executors.newCachedThreadPool();
        executor.execute(new Runnable(){
            public void run(){
                setBusy(true);
                updateStatusBar("ジャンプ先の読み込み中…");

                try{
                    final Period period = village.getPeriod(targetDay);
                    if(period == null){
                        updateStatusBar(
                                  "アンカーのジャンプ先["
                                + anchor.toString()
                                + "]が見つかりません");
                        return;
                    }
                    period.updatePeriod();

                    Talk matchedTalk = null;
                    for(Topic topic : period.getTopicList()){
                        if( ! (topic instanceof Talk) ) continue;
                        Talk talk = (Talk) topic;
                        if(   talk.getHour()   == anchor.getHour()
                           && talk.getMinute() == anchor.getMinute()){
                            matchedTalk = talk;
                            break;
                        }
                    }

                    if(matchedTalk == null){
                        updateStatusBar(
                                  "アンカーのジャンプ先["
                                + anchor.toString()
                                + "]が見つかりません");
                        return;
                    }
                    final Talk targetTalk = matchedTalk;

                    final PeriodView target = browser.getPeriodView(tabIndex);

                    SwingUtilities.invokeLater(new Runnable(){
                        public void run(){
                            browser.setSelectedIndex(tabIndex);
                            target.setPeriod(period);
                            target.scrollToTalk(targetTalk);
                            return;
                        }
                    });
                    updateStatusBar(
                              "アンカー["
                            + anchor.toString()
                            + "]にジャンプしました");
                }catch(IOException e){
                    updateStatusBar(
                            "アンカーの展開中にエラーが起きました");
                }finally{
                    setBusy(false);
                }

                return;
            }
        });

        return;
    }

    /**
     * 指定した国の村一覧を読み込む。
     * @param land 国
     */
    private void execReloadVillageList(final Land land){
        final LandsTree treePanel = this.topView.getLandsTree();
        Executor executor = Executors.newCachedThreadPool();
        executor.execute(new Runnable(){
            public void run(){
                setBusy(true);
                updateStatusBar("村一覧を読み込み中…");
                try{
                    try{
                        Controller.this.model.loadVillageList(land);
                    }catch(IOException e){
                        showNetworkError(land, e);
                    }
                    treePanel.expandLand(land);
                }finally{
                    updateStatusBar("村一覧の読み込み完了");
                    setBusy(false);
                }
                return;
            }
        });
        return;
    }

    /**
     * Period表示の更新処理。
     * @param force trueならPeriodデータを強制再読み込み。
     */
    private void updatePeriod(final boolean force){
        final TabBrowser tabBrowser = this.topView.getTabBrowser();
        final Village village = tabBrowser.getVillage();
        if(village == null) return;
        setFrameTitle(village.getVillageFullName());

        final PeriodView periodView = currentPeriodView();
        Discussion discussion = currentDiscussion();
        if(discussion == null) return;
        discussion.setTopicFilter(this.filterFrame);
        final Period period = discussion.getPeriod();
        if(period == null) return;

        Executor executor = Executors.newCachedThreadPool();
        executor.execute(new Runnable(){
            public void run(){
                setBusy(true);
                try{
                    boolean wasHot = loadPeriod();

                    if(wasHot && ! period.isHot() ){
                        if(updatePeriodList() != true) return;
                    }

                    renderBrowser();
                }finally{
                    setBusy(false);
                }
                return;
            }

            private boolean loadPeriod(){
                updateStatusBar("1日分のデータを読み込んでいます…");
                boolean wasHot;
                try{
                    wasHot = period.isHot();
                    try{
                        period.loadPeriod(force);
                    }catch(IOException e){
                        showNetworkError(village, e);
                    }
                }finally{
                    updateStatusBar("1日分のデータを読み終わりました");
                }
                return wasHot;
            }

            private boolean updatePeriodList(){
                updateStatusBar("村情報を読み直しています…");
                try{
                    village.updatePeriodList();
                }catch(IOException e){
                    showNetworkError(village, e);
                    return false;
                }
                try{
                    SwingUtilities.invokeAndWait(new Runnable(){
                        public void run(){
                            tabBrowser.setVillage(village);
                            return;
                        }
                    });
                }catch(Exception e){
                    Jindolf.logger.log(Level.SEVERE,
                                       "タブ操作で致命的な障害が発生しました",
                                       e);
                }
                updateStatusBar("村情報を読み直しました…");
                return true;
            }

            private void renderBrowser(){
                updateStatusBar("レンダリング中…");
                try{
                    final int lastPos = periodView.getVerticalPosition();
                    try{
                        SwingUtilities.invokeAndWait(new Runnable(){
                            public void run(){
                                periodView.showTopics();
                                return;
                            }
                        });
                    }catch(Exception e){
                        Jindolf.logger
                               .log(Level.SEVERE,
                                    "ブラウザ表示で致命的な障害が発生しました",
                                    e);
                    }
                    SwingUtilities.invokeLater(new Runnable(){
                        public void run(){
                            periodView.setVerticalPosition(lastPos);
                        }
                    });
                }finally{
                    updateStatusBar("レンダリング完了");
                }
                return;
            }
        });

        return;
    }

    /**
     * 発言フィルタの操作による更新処理。
     */
    private void filterChanged(){
        final Discussion discussion = currentDiscussion();
        if(discussion == null) return;
        discussion.setTopicFilter(this.filterFrame);

        Executor executor = Executors.newCachedThreadPool();
        executor.execute(new Runnable(){
            public void run(){
                setBusy(true);
                updateStatusBar("フィルタリング中…");
                try{
                    discussion.filtering();
                }finally{
                    updateStatusBar("フィルタリング完了");
                    setBusy(false);
                }
                return;
            }
        });

        return;
    }

    /**
     * 現在選択中のPeriodを内包するPeriodViewを返す。
     * @return PeriodView
     */
    private PeriodView currentPeriodView(){
        TabBrowser tb = this.topView.getTabBrowser();
        PeriodView result = tb.currentPeriodView();
        return result;
    }

    /**
     * 現在選択中のPeriodを内包するDiscussionを返す。
     * @return Discussion
     */
    private Discussion currentDiscussion(){
        PeriodView periodView = currentPeriodView();
        if(periodView == null) return null;
        Discussion result = periodView.getDiscussion();
        return result;
    }

    /**
     * フレーム表示のトグル処理
     * @param window フレーム
     */
    private void toggleWindow(Window window){
        if(window == null) return;

        if(window instanceof Frame){
            Frame frame = (Frame) window;
            int winState = frame.getExtendedState();
            boolean isIconified = (winState & Frame.ICONIFIED) != 0;
            if(isIconified){
                winState &= ~(Frame.ICONIFIED);
                frame.setExtendedState(winState);
                frame.setVisible(true);
                return;
            }
        }

        if(window.isVisible()){
            window.setVisible(false);
            window.dispose();
        }else{
            window.setVisible(true);
        }
        return;
    }

    /**
     * ネットワークエラーを通知するモーダルダイアログを表示する。
     * OKボタンを押すまでこのメソッドは戻ってこない。
     * @param village 村
     * @param e ネットワークエラー
     */
    public void showNetworkError(Village village, IOException e){
        Land land = village.getParentLand();
        showNetworkError(land, e);
        return;
    }

    /**
     * ネットワークエラーを通知するモーダルダイアログを表示する。
     * OKボタンを押すまでこのメソッドは戻ってこない。
     * @param land 国
     * @param e ネットワークエラー
     */
    public void showNetworkError(Land land, IOException e){
        Jindolf.logger.log(Level.WARNING,
                           "ネットワークで障害が発生しました", e);

        ServerAccess server = land.getServerAccess();
        String message =
                land.getLandName()
                +"を運営するサーバとの間の通信で"
                +"何らかのトラブルが発生しました。\n"
                +"相手サーバのURLは [ " + server.getBaseURL() + " ] だよ。\n"
                +"Webブラウザでも遊べないか確認してみてね!\n";

        JOptionPane pane = new JOptionPane(message,
                                           JOptionPane.WARNING_MESSAGE,
                                           JOptionPane.DEFAULT_OPTION );

        JDialog dialog = pane.createDialog(this.topFrame,
                                           "通信異常発生 - " + Jindolf.TITLE);

        dialog.pack();
        dialog.setVisible(true);
        dialog.dispose();

        return;
    }

    /**
     * {@inheritDoc}
     * ツリーリストで何らかの要素（国、村）がクリックされたときの処理。
     * @param event イベント {@inheritDoc}
     */
    public void valueChanged(TreeSelectionEvent event){
        TreePath path = event.getNewLeadSelectionPath();
        if(path == null) return;

        Object selObj = path.getLastPathComponent();

        if( selObj instanceof Land ){
            Land land = (Land)selObj;
            setFrameTitle(land.getLandName());
            this.topView.showLandInfo(land);
            this.actionManager.appearVillage(false);
            this.actionManager.appearPeriod(false);
        }else if( selObj instanceof Village ){
            final Village village = (Village)selObj;

            Executor executor = Executors.newCachedThreadPool();
            executor.execute(new Runnable(){
                public void run(){
                    setBusy(true);
                    updateStatusBar("村情報を読み込み中…");

                    try{
                        village.updatePeriodList();
                    }catch(IOException e){
                        showNetworkError(village, e);
                        return;
                    }finally{
                        updateStatusBar("村情報の読み込み完了");
                        setBusy(false);
                    }

                    Controller.this.actionManager.appearVillage(true);
                    setFrameTitle(village.getVillageFullName());
                    Controller.this.topView.showVillageInfo(village);

                    return;
                }
            });
        }

        return;
    }

    /**
     * {@inheritDoc}
     * Periodがタブ選択されたときもしくは発言フィルタが操作されたときの処理。
     * @param event イベント {@inheritDoc}
     */
    public void stateChanged(ChangeEvent event){
        Object source = event.getSource();

        if(source == this.filterFrame){
            filterChanged();
        }else if(source instanceof TabBrowser){
            updateFindPanel();
            updatePeriod(false);
            PeriodView periodView = currentPeriodView();
            if(periodView == null) this.actionManager.appearPeriod(false);
            else                   this.actionManager.appearPeriod(true);
        }
        return;
    }

    /**
     * {@inheritDoc}
     * 主にメニュー選択やボタン押下など。
     * @param e イベント {@inheritDoc}
     */
    public void actionPerformed(ActionEvent e){
        if(this.isBusyNow) return;

        String cmd = e.getActionCommand();
        if(cmd.equals(ActionManager.CMD_ACCOUNT)){
            actionShowAccount();
        }else if(cmd.equals(ActionManager.CMD_EXIT)){
            actionExit();
        }else if(cmd.equals(ActionManager.CMD_COPY)){
            actionCopySelected();
        }else if(cmd.equals(ActionManager.CMD_SHOWFIND)){
            actionShowFind();
        }else if(cmd.equals(ActionManager.CMD_SEARCHNEXT)){
            actionSearchNext();
        }else if(cmd.equals(ActionManager.CMD_SEARCHPREV)){
            actionSearchPrev();
        }else if(cmd.equals(ActionManager.CMD_ALLPERIOD)){
            actionLoadAllPeriod();
        }else if(cmd.equals(ActionManager.CMD_WEBVILL)){
            actionShowWebVillage();
        }else if(cmd.equals(ActionManager.CMD_WEBWIKI)){
            actionShowWebWiki();
        }else if(cmd.equals(ActionManager.CMD_WEBCAST)){
            actionShowWebCast();
        }else if(cmd.equals(ActionManager.CMD_RELOAD)){
            actionReloadPeriod();
        }else if(cmd.equals(ActionManager.CMD_DAYSUMMARY)){
            actionDaySummary();
        }else if(cmd.equals(ActionManager.CMD_DAYEXPCSV)){
            actionDayExportCsv();
        }else if(cmd.equals(ActionManager.CMD_WEBDAY)){
            actionShowWebDay();
        }else if(cmd.equals(ActionManager.CMD_FONTSEL)){
            actionFontSelect();
        }else if(cmd.equals(ActionManager.CMD_LANDF)){
            actionChangeLaF();
        }else if(cmd.equals(ActionManager.CMD_SHOWFILT)){
            actionShowFilter();
        }else if(cmd.equals(ActionManager.CMD_SHOWEDIT)){
            actionTalkPreview();
        }else if(cmd.equals(ActionManager.CMD_SHOWLOG)){
            actionShowLog();
        }else if(cmd.equals(ActionManager.CMD_HELPDOC)){
            actionHelp();
        }else if(cmd.equals(ActionManager.CMD_SHOWPORTAL)){
            actionShowPortal();
        }else if(cmd.equals(ActionManager.CMD_ABOUT)){
            actionAbout();
        }else if(cmd.equals(ActionManager.CMD_VILLAGELIST)){
            actionReloadVillageList();
        }else if(cmd.equals(ActionManager.CMD_COPYTALK)){
            actionCopyTalk();
        }else if(cmd.equals(ActionManager.CMD_JUMPANCHOR)){
            actionJumpAnchor();
        }else if(cmd.equals(ActionManager.CMD_WEBTALK)){
            actionShowWebTalk();
        }
        return;
    }

    /**
     * {@inheritDoc}
     * 村選択ツリーリストが畳まれるとき呼ばれる。
     * @param event ツリーイベント {@inheritDoc}
     */
    public void treeWillCollapse(TreeExpansionEvent event){
        return;
    }

    /**
     * {@inheritDoc}
     * 村選択ツリーリストが展開されるとき呼ばれる。
     * @param event ツリーイベント {@inheritDoc}
     */
    public void treeWillExpand(TreeExpansionEvent event){
        if(!(event.getSource() instanceof JTree)){
            return;
        }

        TreePath path = event.getPath();
        Object lastObj = path.getLastPathComponent();
        if(!(lastObj instanceof Land)){
            return;
        }
        final Land land = (Land) lastObj;
        if(land.getVillageCount() > 0){
            return;
        }

        execReloadVillageList(land);

        return;
    }

    /**
     * {@inheritDoc}
     * @param event {@inheritDoc}
     */
    public void anchorHitted(AnchorHitEvent event){
        PeriodView periodView = currentPeriodView();
        if(periodView == null) return;
        Period period = periodView.getPeriod();
        if(period == null) return;
        final Village village = period.getVillage();

        final TalkDraw talkDraw = event.getTalkDraw();
        final Anchor anchor = event.getAnchor();
        final Discussion discussion = periodView.getDiscussion();

        Executor executor = Executors.newCachedThreadPool();
        executor.execute(new Runnable(){
            public void run(){
                setBusy(true);
                updateStatusBar("アンカーの展開中…");

                final List<Talk> talkList;
                try{
                    talkList = village.getTalkListFromAnchor(anchor);
                    if(talkList == null || talkList.size() <= 0){
                        updateStatusBar(
                                  "アンカーの展開先["
                                + anchor.toString()
                                + "]が見つかりません");
                        return;
                    }
                    SwingUtilities.invokeLater(new Runnable(){
                        public void run(){
                            talkDraw.showAnchorTalks(anchor, talkList);
                            discussion.layoutRows();
                            return;
                        }
                    });
                    updateStatusBar(
                              "アンカー["
                            + anchor.toString()
                            + "]の展開完了");
                }catch(IOException e){
                    updateStatusBar(
                            "アンカーの展開中にエラーが起きました");
                }finally{
                    setBusy(false);
                }

                return;
            }
        });

        return;
    }

    /**
     * ヘビーなタスク実行をアピール。
     * プログレスバーとカーソルの設定を行う。
     * @param isBusy trueならプログレスバーのアニメ開始&WAITカーソル。
     *                falseなら停止&通常カーソル。
     */
    private void setBusy(final boolean isBusy){
        this.isBusyNow = isBusy;

        Runnable microJob = new Runnable(){
            public void run(){
                Cursor cursor;
                if(isBusy){
                    cursor = Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR);
                }else{
                    cursor = Cursor.getDefaultCursor();
                }

                Component glass = Controller.this.topFrame.getGlassPane();
                glass.setCursor(cursor);
                glass.setVisible(isBusy == true);
                Controller.this.topView.setBusy(isBusy);

                return;
            }
        };

        if(SwingUtilities.isEventDispatchThread()){
            microJob.run();
        }else{
            try{
                SwingUtilities.invokeAndWait(microJob);
            }catch(Exception e){
                Jindolf.logger.log(Level.SEVERE, "ビジー処理で失敗", e);
            }
        }

        return;
    }

    /**
     * ステータスバーを更新する
     * @param message メッセージ
     */
    private void updateStatusBar(String message){
        this.topView.updateSysMessage(message);
    }

    /**
     * トップフレームのタイトルを設定する。
     * タイトルは指定された国or村名 + " - Jindolf"
     * @param name 国or村名
     */
    private void setFrameTitle(CharSequence name){
        String title = Jindolf.TITLE;

        if(name != null && name.length() > 0){
            title = name + " - " + title;
        }

        this.topFrame.setTitle(title);

        return;
    }
}
