/*
 * model of lands for JTree view
 * 
 * Copyright(c) 2008 olyutorskii
 * $Id: LandsModel.java 116 2008-07-17 14:10:22Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import javax.swing.event.EventListenerList;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

/**
 * 国の集合。あらゆるデータモデルの大元。
 * 国一覧と村一覧を管理。
 * JTreeのモデルも兼用。
 */
public class LandsModel implements TreeModel{ // ComboBoxModelも付けるか？

    /**
     * 村IDで範囲指定した、村のセクション集合。国-村間の中間ツリー。
     * @see javax.swing.tree.TreeModel
     */
    private static class VillageSection{

        /**
         * 与えられた国の全ての村を、指定されたinterval間隔でセクション化する。
         * @param land 国
         * @param interval セクションの間隔
         * @return セクションのリスト
         * @throws java.lang.IllegalArgumentException intervalが正でない
         */
        private static List<VillageSection> getSectionList(Land land,
                                                             int interval )
                throws IllegalArgumentException{
            if(interval <= 0){
                throw new IllegalArgumentException();
            }
            
            List<Village> villageList = land.getVillageList();
            Village village1st = villageList.get(0);
            Village villageLast = villageList.get(villageList.size() -1);
            
            int startID = village1st.getVillageIDNum();
            int endID = villageLast.getVillageIDNum();
            
            List<VillageSection> result = new LinkedList<VillageSection>();
            
            int fixedStart = startID / interval * interval;
            for(int ct=fixedStart; ct <= endID; ct += interval){
                VillageSection section =
                        new VillageSection(land, ct, ct +interval -1);
                result.add(section);
            }
            
            return Collections.unmodifiableList(result);
        }

        private int startID;
        private int endID;
        private String prefix;
        private List<Village> villageList = new LinkedList<Village>();
        
        /**
         * セクション集合を生成する。
         * @param land 国
         * @param startID 開始村ID
         * @param endID 終了村ID
         * @throws java.lang.IndexOutOfBoundsException IDの範囲指定が変
         */
        private VillageSection(Land land, int startID, int endID)
                throws IndexOutOfBoundsException{
            super();

            if(startID < 0 || startID > endID){
                throw new IndexOutOfBoundsException();
            }

            this.startID = startID;
            this.endID = endID;
            this.prefix = land.getLandPrefix();

            for(Village village : land.getVillageList()){
                int id = village.getVillageIDNum();
                if(startID <= id && id <= endID){
                    this.villageList.add(village);
                }
            }
            
            return;
        }
        
        /**
         * セクションに含まれる村の総数を返す。
         * @return 村の総数
         */
        private int getVillageCount(){
            return this.villageList.size();
        }
        
        /**
         * セクションに含まれるindex番目の村を返す。
         * @param index インデックス
         * @return index番目の村
         */
        private Village getVillage(int index){
            return this.villageList.get(index);
        }
 
        /**
         * セクションにおける、指定された子（村）のインデックス位置を返す。
         * @param child 子
         * @return インデックス位置
         */
        private int getIndexOfVillage(Object child){
            return this.villageList.indexOf(child);
        }
        
        /**
         * セクションの文字列表記。
         * JTree描画に反映される。
         * @return 文字列表記
         */
        @Override
        public String toString(){
            StringBuilder result = new StringBuilder();
            result.append(this.prefix).append(this.startID);
            result.append(" ～ ");
            result.append(this.prefix).append(this.endID);
            return result.toString();
        }
    }

    private static final String LANDS_DEF = "resources/land.properties";
    private static final String KEY_PREFIX = "land.order.";
    private static final String ROOT = "ROOT";
    private static final int SECTION_INTERVAL = 100;
    
    private final List<Land> landList = new LinkedList<Land>();
    private final List<Land> unmodList =
            Collections.unmodifiableList(this.landList);
    private final Map<Land, List<VillageSection>> sectionMap =
            new HashMap<Land, List<VillageSection>>();
    private boolean isLandListLoaded = false;
    
    private final EventListenerList listeners = new EventListenerList();

    private boolean ascending = false;

    /**
     * コンストラクタ。
     * この時点ではまだ国一覧が読み込まれない。
     */
    public LandsModel(){
        super();
        return;
    }
    
    /**
     * 指定した国の村一覧を読み込む。
     * @param land
     */
    public void loadVillageList(Land land) throws IOException{
        land.updateVillageList();
        
        List<VillageSection> sectionList =
                VillageSection.getSectionList(land, SECTION_INTERVAL);
        sectionMap.put(land, sectionList); 
        
        int[] childIndices = new int[sectionList.size()];
        for(int ct=0; ct < childIndices.length; ct++){
            childIndices[ct] = ct;
        }
        Object[] children = sectionList.toArray();
        
        Object[] path = {ROOT, land};
        TreePath treePath = new TreePath(path);
        TreeModelEvent event = new TreeModelEvent(this,
                                                  treePath,
                                                  childIndices,
                                                  children     );
        fireTreeStructureChanged(event);

        return;
    }
    
    /**
     * リソース上のプロパティファイルから国一覧を読み込む。
     */
    public void loadLandList() throws IOException{
        if(this.isLandListLoaded) return;

        InputStream is = Jindolf.getResourceAsStream(LANDS_DEF);
        loadLandList(is);

        return;
    }
    
    /**
     * InputStreamから国一覧を読み込む。
     */
    public void loadLandList(InputStream is) throws IOException{
        if(this.isLandListLoaded) return;

        Properties properties = new Properties();
        properties.load(is);
        is.close();
        
        loadLandList(properties);
        
        return;
    }
    
    /**
     * プロパティから国一覧を読み込む。
     */
    public void loadLandList(Properties properties){
        if(this.isLandListLoaded) return;
        
        String codeCheck = properties.getProperty("codeCheck");
        if(   codeCheck == null
           || codeCheck.length() != 1
           || codeCheck.charAt(0) != '\u72fc'){  // 「狼」
            Jindolf.logger.severe("国定義一覧プロパティファイルの"
                    +"文字コードがおかしいようです。"
                    +"native2ascii は正しく適用しましたか？");
            Jindolf.exit(1);
            return;
        }
        
        Set<Object> keySet = properties.keySet();
             
        SortedSet<Integer> orderSet = new TreeSet<Integer>();
        for(Object keyObj : keySet){
            if(keyObj == null) continue;
            String key = keyObj.toString();
            if( ! key.startsWith(KEY_PREFIX) ) continue;
            key = key.replace(KEY_PREFIX, "");
            Integer order;
            try{
                order = new Integer(key);
            }catch(NumberFormatException e){
                continue;
            }
            orderSet.add(order);
        }
        
        this.landList.clear();
        for(Integer order : orderSet){
            if(order == null) continue;
            String key = KEY_PREFIX + order.toString();
            String landId = properties.getProperty(key);
            if(landId == null || landId.length() <= 0) continue;

            Land land = new Land(landId, properties);
            this.landList.add(land);
        }
        isLandListLoaded = true;

        fireLandListChanged();

        return;
    }

    /**
     * ツリー内容が更新された事をリスナーに通知する。
     */
    private void fireLandListChanged(){
        int size = this.landList.size();
        int[] childIndices = new int[size];
        for(int ct=0; ct < size; ct++){
            int index = ct;
            childIndices[ct] = index;
        }
        
        Object[] children = this.landList.toArray();
            
        TreePath treePath = new TreePath(ROOT);
        TreeModelEvent event = new TreeModelEvent(this,
                                                  treePath,
                                                  childIndices,
                                                  children     );
        fireTreeStructureChanged(event);

        return;
    }
    
    /**
     * ツリーの並び順を設定する。
     * 場合によってはTreeModelEventが発生する。
     * @param ascending trueなら昇順
     */
    public void setAscending(boolean ascending){
        if(this.ascending == ascending) return;
        
        this.ascending = ascending;
        fireLandListChanged();
        
        return;
    }
    
    /**
     * リスナー登録
     * @param l リスナー
     */
    public void addTreeModelListener(TreeModelListener l){
        listeners.add(TreeModelListener.class, l);
        return;
    }

    /**
     * リスナー削除
     * @param l リスナー
     */
    public void removeTreeModelListener(TreeModelListener l){
        listeners.remove(TreeModelListener.class, l);
        return;
    }

    /**
     * 登録中のリスナーのリストを得る。
     * @return リスナーのリスト
     */
    private TreeModelListener[] getTreeModelListeners(){
        return listeners.getListeners(TreeModelListener.class);
    }

    /**
     * 全リスナーにイベントを送出する。
     * @param event ツリーイベント
     */
    protected void fireTreeStructureChanged(TreeModelEvent event){
        for(TreeModelListener listener : getTreeModelListeners()){
            listener.treeStructureChanged(event);
        }
        return;
    }

    /**
     * 国リストを得る。
     * @return 国のリスト
     */
    public List<Land> getLandList(){
        return this.unmodList;
    }

    /**
     * 親の子を返す。
     * @param parent 親
     * @param index 子のインデックス
     * @return 子
     */
    public Object getChild(Object parent, int index){
        if(index < 0)                      return null;
        if(index >= getChildCount(parent)) return null;

        if(parent == ROOT){
            List<Land> list = getLandList();
            if( ! this.ascending) index = list.size() -index -1;
            Land land = list.get(index);
            return land;
        }
        if(parent instanceof Land){
            Land land = (Land)parent;
            List<VillageSection> sectionList = this.sectionMap.get(land);
            if( ! this.ascending) index = sectionList.size() -index -1;
            VillageSection section = sectionList.get(index);
            return section;
        }
        if(parent instanceof VillageSection){
            VillageSection section = (VillageSection)parent;
            if( ! this.ascending) index = section.getVillageCount() -index -1;
            Village village = section.getVillage(index);
            return village;
        }
        return null;
    }

    /**
     * 子の数を返す。
     * @param parent 親
     * @return 子の数
     */
    public int getChildCount(Object parent){
        if(parent == ROOT){
            return getLandList().size();
        }
        if(parent instanceof Land){
            Land land = (Land)parent;
            List<VillageSection> sectionList = this.sectionMap.get(land);
            if(sectionList == null) return 0;
            return sectionList.size();
        }
        if(parent instanceof VillageSection){
            VillageSection section = (VillageSection)parent;
            return section.getVillageCount();
        }
        return 0;
    }

    /**
     * 親の中で子が占める位置のインデックス値を返す。
     * @param parent 親
     * @param child 子
     * @return インデックス値
     */
    public int getIndexOfChild(Object parent, Object child){
        if(child == null) return -1;
        if(parent == ROOT){
            List<Land> list = getLandList();
            int index = list.indexOf(child);
            if( ! this.ascending) index = list.size() -index -1;
            return index;
        }
        if(parent instanceof Land){
            Land land = (Land)parent;
            List<VillageSection> sectionList = this.sectionMap.get(land);
            int index = sectionList.indexOf(child);
            if( ! this.ascending) index = sectionList.size() -index -1;
            return index;
        }
        if(parent instanceof VillageSection){
            VillageSection section = (VillageSection)parent;
            int index = section.getIndexOfVillage(child);
            if( ! this.ascending) index = section.getVillageCount() -index -1;
            return index;
        }
        return -1;
    }

    /**
     * Treeの根元を返す。
     * @return 根本
     */
    public Object getRoot(){
        return ROOT;
    }

    /**
     * 葉要素か否か判定する。
     * @param node 要素
     * @return 葉ならtrue
     */
    public boolean isLeaf(Object node){
        if(node == ROOT)                   return false;
        if(node instanceof Land)           return false;
        if(node instanceof VillageSection) return false;
        if(node instanceof Village)        return true;
        return true;
    }

    /**
     * Tree項目が変更されたことを通知。
     * ※ たぶん使わないので必ず失敗させている。
     * @param path Treeパス
     * @param newValue 新しい値
     */
    public void valueForPathChanged(TreePath path, Object newValue){
        throw new UnsupportedOperationException("Not supported yet.");
    }
}
