/*
 * Find panel
 *
 * Copyright(c) 2008 olyutorskii
 * $Id: FindPanel.java 198 2008-09-26 09:23:39Z olyutorskii $
 */

package jp.sourceforge.jindolf;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.border.Border;

/**
 * 検索パネルGUI
 */
@SuppressWarnings("serial")
public class FindPanel extends JDialog
        implements ActionListener, WindowListener{

    private static final String FRAMETITLE = "発言検索 - " + Jindolf.TITLE;

    private final JComboBox findBox = new JComboBox();
    private final JButton searchButton = new JButton("検索");
    private final JCheckBox capitalSwitch =
            new JCheckBox("大文字/小文字を区別する");
    private final JCheckBox dotallSwitch =
            new JCheckBox("正規表現 \".\" が行末記号にもマッチする");
    private final JCheckBox multilineSwitch =
            new JCheckBox("正規表現 \"^\" や \"$\" が行末記号の前後に反応する");
    private final JButton closeButton = new JButton("キャンセル");
    private boolean isCanceled = false;

    /**
     * 検索パネルを生成する。
     * @param owner 親フレーム。
     */
    public FindPanel(Frame owner){
        super(owner, FRAMETITLE, true);

        setResizable(true);
        Toolkit kit = getToolkit();
        kit.setDynamicLayout(false);
//        setIconImage(GUIUtils.getWindowIconImage());
        setLocationByPlatform(true);
        setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
        addWindowListener(this);

        this.findBox.setToolTipText("検索文字列を正規表現で入力してね");
        this.findBox.setEditable(true);
        this.findBox.addItem("");
        this.findBox.addItem("【[^】]*】");
        this.findBox.addItem("[■●▼★□○▽☆〇◯∇]");
        this.findBox.addItem("(J|j)indolf");
        this.searchButton.setToolTipText("発言内容を検索する");
        
        this.capitalSwitch.setSelected(true);
        this.dotallSwitch.setSelected(true);
        this.multilineSwitch.setSelected(false);
        
        this.searchButton.addActionListener(this);
        this.closeButton.addActionListener(this);

        design();
        
        return;
    }
    
    /**
     * デザイン・レイアウトを行う。
     */
    private void design(){
        Container content = getContentPane();

        GridBagLayout layout = new GridBagLayout();
        GridBagConstraints constraints = new GridBagConstraints();

        content.setLayout(layout);

        constraints.insets = new Insets(2, 2, 2, 2);
        
        constraints.weightx = 1.0;
        constraints.fill = GridBagConstraints.HORIZONTAL;
        Border border =
                BorderFactory.createTitledBorder("正規表現を入力してください");
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        panel.add(this.findBox, BorderLayout.CENTER);
        panel.setBorder(border);
        content.add(panel, constraints);
        
        constraints.weightx = 0.0;
        constraints.fill = GridBagConstraints.NONE;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        content.add(this.searchButton, constraints);
        
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.anchor = GridBagConstraints.WEST;
        content.add(this.capitalSwitch, constraints);
        
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.anchor = GridBagConstraints.WEST;
        content.add(this.dotallSwitch, constraints);
        
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.anchor = GridBagConstraints.WEST;
        content.add(this.multilineSwitch, constraints);
        
        constraints.weightx = 1.0;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.fill = GridBagConstraints.HORIZONTAL;
        content.add(new JSeparator(), constraints);

        constraints.weightx = 0.0;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
        constraints.anchor = GridBagConstraints.EAST;
        constraints.fill = GridBagConstraints.NONE;
        content.add(this.closeButton, constraints);
        
        return;
    }

    /**
     * ダイアログが閉じられた原因を判定する。
     * @return キャンセルもしくはクローズボタンでダイアログが閉じられたらtrue
     */
    public boolean isCanceled(){
        return this.isCanceled;
    }
    
    /**
     * キャンセルボタン押下処理。
     * このモーダルダイアログを閉じる。
     */
    private void actionCancel(){
        isCanceled = true;
        setVisible(false);
        dispose();
        return;
    }
    
    /**
     * 検索ボタン押下処理。
     * このモーダルダイアログを閉じる。
     */
    private void actionSubmit(){
        isCanceled = false;
        setVisible(false);
        dispose();
        return;
    }
    
    /**
     * 現在選択中のの検索パターンを得る。
     * @return 検索パターン
     */
    public Pattern currentPattern(){
        Object selected = this.findBox.getSelectedItem();
        if(selected == null) return null;
        String searchString = selected.toString();
        if(searchString.length() <= 0) return null;

        int flag = 0x00000000;
        
        if( ! this.capitalSwitch.isSelected() ){
            flag |= Pattern.UNICODE_CASE;
            flag |= Pattern.CASE_INSENSITIVE;
        }
        if(this.dotallSwitch.isSelected())      flag |= Pattern.DOTALL;
        if(this.multilineSwitch.isSelected())   flag |= Pattern.MULTILINE;
        
        Pattern pattern;
        try{
            pattern= Pattern.compile(searchString, flag);
        }catch(PatternSyntaxException e){
            return null;
        }

        return pattern;
    }

    /**
     * ウィンドウリスナ
     * @param event ウィンドウ変化イベント
     */
    public void windowActivated(WindowEvent event){
        return;
    }

    /**
     * ウィンドウリスナ
     * @param event ウィンドウ変化イベント
     */
    public void windowDeactivated(WindowEvent event){
        return;
    }

    /**
     * ウィンドウリスナ
     * @param event ウィンドウ変化イベント
     */
    public void windowIconified(WindowEvent event){
        return;
    }

    /**
     * ウィンドウリスナ
     * @param event ウィンドウ変化イベント
     */
    public void windowDeiconified(WindowEvent event){
        return;
    }

    /**
     * ウィンドウリスナ
     * @param event ウィンドウ変化イベント
     */
    public void windowOpened(WindowEvent event){
        return;
    }

    /**
     * ウィンドウリスナ。
     * ダイアログのクローズボタン押下処理を行う。
     * @param event ウィンドウ変化イベント
     */
    public void windowClosing(WindowEvent event){
        actionCancel();
        return;
    }

    /**
     * ウィンドウリスナ
     * @param event ウィンドウ変化イベント
     */
    public void windowClosed(WindowEvent event){
        return;
    }
    
    /**
     * ボタン操作時にリスナとして呼ばれる。
     * @param event イベント
     */
    public void actionPerformed(ActionEvent event){
        Object source = event.getSource();
        if(source == this.closeButton){
            actionCancel();
        }else if(source == this.searchButton){
            actionSubmit();
        }
        return;
    }
    
    // TODO 村一括検索
    // TODO 正規表現以外のフリーワード指定
    // TODO 大小文字同一視などの検索オプション
    // TODO OR検索
    // TODO ブックマーク機能との統合
    // TODO クリック指定した箇所からの検索ナビゲーション
    // TODO F3やCtrl-Fなどのキーアサイン
    // TODO 検索履歴
}
