/*
 * Copyright 2000-2001,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.jetspeed.test;

// Turbine imports
import org.apache.turbine.modules.actions.sessionvalidator.SessionValidator;
import org.apache.turbine.modules.ActionLoader;
import org.apache.turbine.modules.PageLoader;
import org.apache.turbine.services.resources.TurbineResources;
import org.apache.turbine.services.template.TurbineTemplate;
import org.apache.turbine.util.RunData;

/**
 * This class is designed to aid in the testing of Jetspeed
 * page generation.  It includes methods that mimic what
 * is done in turbine.
 *
 * As a minimum you test case should look like:
 * <CODE>
 * {
 *  setupRunData(rundata);
 *  generatePage(rundata);
 *  outputPage(rundata);
 * }
 * </CODE>
 *
 * @author <a href="paulsp@apache.org">Paul Spencer</a>
 * @version $Id: TurbineTestUtilities.java,v 1.1 2004/04/07 22:02:41 jford Exp $
*/
public abstract class TurbineTestUtilities
{
   /**
     * Do all of the initialization and setup of RunData.  This includes
     * setting up the session, users.
     *
     * Note: This code is modeled after Turbine's org.apache.turbine.Turbine
     *
     * @param rundata Rundata to setup
     * @throws Exception General exceptions
     */    
    public static void setupRunData(RunData rundata) throws Exception
    {
        if (rundata == null)
            throw new NullPointerException("rundata is null");
        
        // Get the instance of the Session Validator.
        SessionValidator sessionValidator = (SessionValidator)ActionLoader
             .getInstance().getInstance(TurbineResources.getString(
                 "action.sessionvalidator"));
        if (sessionValidator == null)
            throw new NullPointerException("Failed to get a SessonValidator");

        // Fill in the screen and action variables.
        rundata.setScreen( rundata.getParameters().getString("screen") );
        rundata.setAction( rundata.getParameters().getString("action") );
        
        // Login or out if requested
        if ( rundata.hasAction()
           && rundata.getAction().equalsIgnoreCase(TurbineResources
              .getString("action.login"))
           || rundata.getAction().equalsIgnoreCase(TurbineResources
              .getString("action.logout")))
        {
            if (rundata.getAction().equalsIgnoreCase(TurbineResources
              .getString("action.login")))
            {
                String[] names = rundata.getSession().getValueNames();
                if (names != null)
                {
                    for (int i=0; i< names.length; i++)
                    {
                        rundata.getSession().removeValue(names[i]);
                    }
                }
            }
            ActionLoader.getInstance().exec( rundata, rundata.getAction() );
            rundata.setAction(null);
        }

        // Do the session validation
        ActionLoader.getInstance().exec(
           rundata,TurbineResources.getString("action.sessionvalidator") );
        
        // Put the access control list (ACL) into rundata.
        ActionLoader.getInstance().exec(
           rundata,TurbineResources.getString("action.accesscontroller"));
        
    }
    
    /**
     * Generate the page/content defined by rundata
     * @param rundata Rundata to setup
     * @throws Exception General exceptions
     */    
    public static void generatePage(RunData rundata) throws Exception
    {
        if (rundata == null)
            throw new NullPointerException("rundata is null");
        
        String defaultPage = TurbineTemplate.getDefaultPageName(rundata);
        PageLoader.getInstance().exec(rundata, defaultPage);
    }

    /**
     * Instuct turbine, via rundata, to output the page.
     *
     * Note: This code is modeled after Turbine's org.apache.turbine.Turbine
     *
     * @param rundata Rundata to setup
     * @throws Exception General exceptions
     */    
    public static void outputPage(RunData rundata) throws Exception
    {
        if (rundata == null)
            throw new NullPointerException("rundata is null");
        
        rundata.getResponse().setLocale( rundata.getLocale() );
        rundata.getResponse().setContentType( rundata.getContentType() );
        rundata.getResponse().setStatus( rundata.getStatusCode() );
        rundata.getPage().output(rundata.getOut());
        
        try
        {
            rundata.getOut().close();
        }
        catch (Exception e)
        {
            // Ignore.
        }
    }

}