/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.webpage;

// java.io
import java.io.IOException;

// javax.servlet
import javax.servlet.http.*;

// java.net
import java.net.URLConnection;

/**
 * Standard interface for all proxied sessions.
 * Handles the communication and session state between the webpage service and a single site
 *
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor</a>
 * @version $Id: SiteSession.java,v 1.2 2003/03/04 00:05:14 sgala Exp $ 
 */
public interface SiteSession
{    

    /**
     * Given a site URL, proxies the content of that site.  
     * The actual rules on rewriting the proxied resource are dependent on implementation 
     * and configuration parameters. For example, all HTTP hyperlinks(HREFs) could be
     * rewritten as proxied hyperlinks back to this Proxy.
     * Or all relative references to web resources (images, stylesheets, ...) could be
     * rewritten as absolute references, but are not proxied.
     *
     * @param site the proxied resource address.
     * @param data the request specific rundata.
     *
     * @exception IOException a servlet exception.
     */    
    public void proxy(String site, ProxyRunData data)
                    throws IOException;

    /**
     * Gets the HTML content from the URL Connection stream and returns it as a Stream
     *
     * @param con The URLConnection to read from.
     * @param data the request specific rundata.
     * @return The HTML Content from the stream.
     *
     * @deprecate
     * @exception IOException a servlet exception.
     */
    public String getContentAsString(URLConnection con,
                                     ProxyRunData data,
                                     String url) 
                    throws IOException;

    
    /**
     * Retrieves the content from the URL Connection stream and writes it to servlet response
     *
     * @param con The URLConnection to read from.
     *
     * @exception IOException a servlet exception.
     */
    public void drainContent(URLConnection con,
                             HttpServletResponse response) throws IOException;

    /**
      * Given a cookie, it first checks to see if that cookie is already
      * managed in this session. If it is, it means that the session has
      * timed out and that the network element has now created a new session.
      * In that case, replace the cookie, and re-establish the session (logon)
      * If its a new cookie, we will still need to logon, and and the cookie to
      * the managed cookies collection for this session.
      *
      * @param cookie new cookie returned from target server.
      * @return true when a new cookie added, false when updated.
      *
      */
    public boolean addCookieToSession(Cookie cookie);

    /**
     * Logs on to the target host
     *
     * @param data the request specific rundata.     
     *
     * @exception IOException a servlet exception.
     */    
    public boolean logon(ProxyRunData data)
                      throws IOException;


    /**
     * Logs out of the target host
     *
     * @param data the request specific rundata.     
     *
     * @exception IOException a servlet exception.
     */    
    public boolean logout(ProxyRunData data)
                      throws IOException;


    /**
     * Reads stream for proxied host, runs a rewriter against that stream,
     * rewriting relevant links, and writes the parsed stream back to the client.
     *
     * @param request Servlet request.
     * @param con the URLConnection with proxied host.
     * @param contentType the contentType of the request.
     *
     * @exception IOException a servlet exception.
     */    
    public void rewriteContent(ProxyRunData data,
                               URLConnection con,
                               int contentType,
                               String url) throws IOException;

    /**
     * Gets the hitcount for this session.
     *
     * @return the hitcount for this session.
     */
    public int getHitCount();

    /**
     * Increments the hitcount for this session.
     *
     */
    public void incHitCount();

    /**
     * Gets the cache count for this session.
     *
     * @return the cache count for this session.
     */
    public int getCacheCount();

    /**
     * Increments the hitcount for this session.
     *
     */
    public void incCacheCount();

}

