/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.security.turbine;


// Jetspeed imports
import org.apache.jetspeed.test.JetspeedTestCase;
import org.apache.jetspeed.om.profile.Entry;
import org.apache.jetspeed.om.profile.psml.PsmlEntry;
import org.apache.jetspeed.om.security.JetspeedUser;
import org.apache.jetspeed.om.registry.base.BaseSecurity;
import org.apache.jetspeed.om.registry.base.BasePortletEntry;
import org.apache.jetspeed.om.registry.base.BaseParameter;
import org.apache.jetspeed.om.registry.Parameter;
import org.apache.jetspeed.om.registry.PortletEntry;
import org.apache.jetspeed.om.registry.Security;

import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.services.JetspeedPortalAccessController;
import org.apache.jetspeed.services.resources.JetspeedResources;
import org.apache.jetspeed.services.security.PortalResource;

// Turbine imports
import org.apache.turbine.util.TurbineConfig;
import org.apache.turbine.util.StringUtils;

// Junit imports
import junit.awtui.TestRunner;
import junit.framework.Test;
import junit.framework.TestSuite;

/**
 * TestAccessController
 *
 * @author <a href="paulsp@apache.org">Paul Spencer</a>
 * @version $Id: TestAccessController.java,v 1.5 2003/08/08 18:28:03 morciuch Exp $
 */
public class TestAccessController extends JetspeedTestCase
{
    private static String ADMIN_PORTLET = "GlobalAdminPortlet"; // Portlet accessable by Admin user, role = admin
    private static String ALL_PORTLET = "HelloVelocity";  // Portlet accessable by Anonymous user
    private static String TEST_GROUP = "Jetspeed";
    private static String TEST_SECURITY_PAGE = "SecurityTest";
    private static String USER_PORTLET = "SkinBrowser"; // Portlet accessable by general user, role = user

    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestAccessController( String name )
    {
        super( name );
    }
    
    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[])
    {
        TestRunner.main( new String[]
        { TestAccessController.class.getName() } );
    }
    
    public void setup()
    {
        System.out.println("Setup: Testing categories of Profiler Service");
    }
    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite()
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestAccessController.class );
    }
    
    public void testVerifyEnvironment() throws Exception
    {
        assertEquals( "Using TurbineAccessController",
        "org.apache.jetspeed.services.security.turbine.TurbineAccessController",
        JetspeedResources.getString("services.PortalAccessController.classname"));
        
        assertNotNull( "Getting admin user", JetspeedSecurity.getUser("admin"));
        assertNotNull( "Getting turbine user", JetspeedSecurity.getUser("turbine"));
        assertNotNull( "Getting anonymous user", JetspeedSecurity.getAnonymousUser());
    }
        
    public void testRequiredActions() throws Exception
    {
        JetspeedUser adminUser = (JetspeedUser) JetspeedSecurity.getUser("admin");
        assertNotNull( "Getting admin user", adminUser);
        adminUser.setHasLoggedIn(Boolean.TRUE);
        
        JetspeedUser turbineUser = (JetspeedUser) JetspeedSecurity.getUser("turbine");
        assertNotNull( "Getting turbine user", turbineUser);
        turbineUser.setHasLoggedIn(Boolean.TRUE);
        
        JetspeedUser anonymousUser =  (JetspeedUser) JetspeedSecurity.getAnonymousUser();
        assertNotNull( "Getting anonymous user", anonymousUser);
        Entry adminEntry = createEntry( ADMIN_PORTLET, "ST_01.admin");
        Entry userEntry = createEntry(USER_PORTLET, "ST_01.user");
        Entry allEntry = createEntry(ALL_PORTLET, "ST_01.all");
        
        assertEquals( "Admin user has view access to " + ADMIN_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, adminEntry, "view"));
//        assertEquals( "Turbine user DOES NOT have view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( turbineUser, adminEntry, "view"));

//        assertEquals( "Anonymous user DOES NOT have view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( anonymousUser, adminEntry, "view"));
//        assertEquals( "null user DOES NOT have view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( (JetspeedUser) null, adminEntry, "view"));
        
        assertEquals( "Admin user has view access to " + USER_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, userEntry, "view"));
        assertEquals( "Turbine user has view access to " + USER_PORTLET, true, JetspeedPortalAccessController.checkPermission( turbineUser, userEntry, "view"));
 //       assertEquals( "Anonymous user DOES NOT have view access to " + USER_PORTLET, false, JetspeedPortalAccessController.checkPermission( anonymousUser, userEntry, "view"));
//        assertEquals( "null user DOES NOT have view access to " + USER_PORTLET, false, JetspeedPortalAccessController.checkPermission( (JetspeedUser) null, userEntry, "view"));
        
        assertEquals( "Admin user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, allEntry, "view"));
        assertEquals( "Turbine user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( turbineUser, allEntry, "view"));
        assertEquals( "Anonymous user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( anonymousUser, allEntry, "view"));
        assertEquals( "null user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( (JetspeedUser) null, allEntry, "view"));
    }

    public void testRegistryActions() throws Exception
    {
        JetspeedUser adminUser = (JetspeedUser) JetspeedSecurity.getUser("admin");
        assertNotNull( "Getting admin user", adminUser);
        adminUser.setHasLoggedIn(Boolean.TRUE);
        
        JetspeedUser turbineUser = (JetspeedUser) JetspeedSecurity.getUser("turbine");
        assertNotNull( "Getting turbine user", turbineUser);
        turbineUser.setHasLoggedIn(Boolean.TRUE);
        
        JetspeedUser anonymousUser =  (JetspeedUser) JetspeedSecurity.getAnonymousUser();
        assertNotNull( "Getting anonymous user", anonymousUser);
        // Create security objects
        Security adminSecurity = new BaseSecurity("admin");
        assertNotNull( "Have admin security", adminSecurity);
        Security userSecurity = new BaseSecurity("user");
        assertNotNull( "Have user security", userSecurity);

        PortletEntry userPortletEntry = new BasePortletEntry();
        assertNotNull( "Have userPortletEntry", userPortletEntry);
        userPortletEntry.setName( USER_PORTLET);
        userPortletEntry.setSecurity( userSecurity);
        Parameter adminParam = new BaseParameter();
        assertNotNull( "Have adminParameter", adminParam);
        adminParam.setName("AdminParam");
        adminParam.setValue("adminValue");
        adminParam.setSecurity(adminSecurity);
        userPortletEntry.addParameter(adminParam);

        Parameter userParam = new BaseParameter();
        assertNotNull( "Have userParameter", userParam);
        userParam.setName("UserParam");
        userParam.setValue("userValue");
        userParam.setSecurity(userSecurity);
        userPortletEntry.addParameter(userParam);
        assertEquals( "Admin user customize access to admin parameter", true, JetspeedPortalAccessController.checkPermission( adminUser, new PortalResource( userPortletEntry, adminParam), JetspeedSecurity.PERMISSION_CUSTOMIZE));
        assertEquals( "Turbine user customize access to admin parameter", false, JetspeedPortalAccessController.checkPermission( turbineUser, new PortalResource( userPortletEntry, adminParam), JetspeedSecurity.PERMISSION_CUSTOMIZE));
        assertEquals( "Admin user customize access to admin parameter", true, JetspeedPortalAccessController.checkPermission( adminUser, new PortalResource( userPortletEntry, userParam), JetspeedSecurity.PERMISSION_CUSTOMIZE));
        assertEquals( "Turbine user customize access to admin parameter", true, JetspeedPortalAccessController.checkPermission( turbineUser, new PortalResource( userPortletEntry, userParam), JetspeedSecurity.PERMISSION_CUSTOMIZE));
        
/*
        RegistryEntry adminEntry = createRegistryEntry( ADMIN_PORTLET, "ST_01.admin");
        RegistryEntry userEntry = createRegistryEntry( USER_PORTLET, "ST_01.user");
        RegistryEntry allEntry = createRegistryEntry( ALL_PORTLET, "ST_01.all");
        
        assertEquals( "Admin user has view access to " + ADMIN_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, adminEntry, "view"));
        assertEquals( "Turbine user DOES NOT have view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( turbineUser, adminEntry, "view"));
        assertEquals( "Anonymous user DOES NOT have view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( anonymousUser, adminEntry, "view"));
        assertEquals( "null user DOES NOT have view access to " + ADMIN_PORTLET, false, JetspeedPortalAccessController.checkPermission( (JetspeedUser) null, adminEntry, "view"));
        
        assertEquals( "Admin user has view access to " + USER_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, userEntry, "view"));
        assertEquals( "Turbine user has view access to " + USER_PORTLET, true, JetspeedPortalAccessController.checkPermission( turbineUser, userEntry, "view"));
        assertEquals( "Anonymous user DOES NOT have view access to " + USER_PORTLET, false, JetspeedPortalAccessController.checkPermission( anonymousUser, userEntry, "view"));
        assertEquals( "null user DOES NOT have view access to " + USER_PORTLET, false, JetspeedPortalAccessController.checkPermission( (JetspeedUser) null, userEntry, "view"));
        
        assertEquals( "Admin user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( adminUser, allEntry, "view"));
        assertEquals( "Turbine user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( turbineUser, allEntry, "view"));
        assertEquals( "Anonymous user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( anonymousUser, allEntry, "view"));
        assertEquals( "null user has view access to " + ALL_PORTLET, true, JetspeedPortalAccessController.checkPermission( (JetspeedUser) null, allEntry, "view"));
*/
    }
    /*
     * Setup Turbine environment
     */
    
    /*
     * Configuration object to run Turbine outside a servlet container
     * ( uses turbine.properties )
     */
    private static TurbineConfig config = null;
    
    /*
     * Sets up TurbineConfig using the system property:
     * <pre>turbine.properties</pre>
     */
    static
    {
        try
        {
            config = new TurbineConfig( "../webapp", "/WEB-INF/conf/TurbineResources.properties");
            config.init();
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
    }
    
    private PsmlEntry createEntry(java.lang.String parent, java.lang.String id)
    {
        PsmlEntry entry = new PsmlEntry();
        entry.setParent( parent);
        if (id != null)
            entry.setId( id);
        return entry;
    }
}
