package org.apache.jetspeed.services.logging;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *    "Apache Turbine" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    "Apache Turbine", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

// Java classes
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;

// Jetspeed classes
import org.apache.jetspeed.services.resources.JetspeedResources;

// Log4J classes
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;
import org.apache.log4j.xml.DOMConfigurator;

// Turbine classes
import org.apache.turbine.Turbine;
import org.apache.turbine.services.InitializationException;
import org.apache.turbine.services.TurbineBaseService;

/**
 * The default implementation of the logging service in Jetspeed.
 *
 * This service initializes the underlying logging implementation, 
 * and acts as a factory for loggers.
 * The current implementation uses Log4J.
 *
 * @see org.apache.log4j.LogManager
 * @see org.apache.log4j.Logger
 * @author <a href="mailto:harald@ommang.com">Harald Ommang</a>
 */
public class JetspeedLogFactoryService extends TurbineBaseService
{

    public String SERVICE_NAME = "JetspeedLogFactoryService"; 
    private static final String CONFIG_LOG4J_PROPERTIES = "log4j.properties";
    private static final String CONFIG_LOG4J_PROPERTIES_DEFAULT = "/WEB-INF/conf/log4j.properties";
    private static final String CONFIG_LOG4J_AND_WATCH = "log4j.configureAndWatch";
    private static final boolean CONFIG_LOG4J_AND_WATCH_DEFAULT = true;
    private static final String CONFIG_LOG4J_WATCHINTERVAL = "log4j.watchInterval";
    private static final long CONFIG_LOG4J_WATCHINTERVAL_DEFAULT = 60000L;
    private ServletContext context;
    /**
     * Flag to check for initilization. Needed to make time of init more robust.
     * Also, cannot access the init in parent class from static method
     */        
    private static boolean initDone = false;
        
    /**
     * Default constructor
     */    
    public JetspeedLogFactoryService()
    {
        context = null;
    }    
   
    /**
     * Initializes the service by getting the servlet configuration from Turbine
     *
     * @throws InitializationException Initialization failed
     */ 
    public void init() throws InitializationException
    {
        ServletConfig conf = Turbine.getTurbineServletConfig();
        if(conf != null)
        {
            init(conf);
        }
    }

    /**
     * Initializes the service with the given configuration
     * Initializes the underlying logging implementation, Log4J
     *
     * @param config The ServletConfiguration from Turbine
     *
     * @throws InitializationException Initialization failed
     */    
    public void init(ServletConfig config) throws InitializationException
    {
        context = config.getServletContext();
        String log4jProperties = JetspeedResources.getString(CONFIG_LOG4J_PROPERTIES, CONFIG_LOG4J_PROPERTIES_DEFAULT);
        if(log4jProperties != null)
        {
            try
            {
                String fileName = Turbine.getRealPath(log4jProperties);
                boolean watch = JetspeedResources.getBoolean(CONFIG_LOG4J_AND_WATCH, CONFIG_LOG4J_AND_WATCH_DEFAULT);
                long watchInterval = JetspeedResources.getLong(CONFIG_LOG4J_WATCHINTERVAL, CONFIG_LOG4J_WATCHINTERVAL_DEFAULT);
                System.setProperty("webappRoot", context.getRealPath("/"));
                
                // Check to see if property or XML configuration is to be used.
                if(fileName.endsWith(".properties"))
                {
                    if(watch)
                    {
                        // Configure with a property file and watch for changes
                        PropertyConfigurator.configureAndWatch(fileName, watchInterval);
                    } 
                    else
                    {
                        PropertyConfigurator.configure(fileName);
                    }
                } 
                else
                {
                    if(watch)
                    {
                        // Configure with an XML file and watch for changes
                        DOMConfigurator.configureAndWatch(fileName, watchInterval);
                    } 
                    else
                    {
                        DOMConfigurator.configure(fileName);
                    }
                }
            }
            catch(Exception e)
            {
                throw new InitializationException("Failed to load " + log4jProperties + " - " + e.toString());
            }
        }
        setInit(true);
        initDone = true;
    } // init

    /**
     * The actual Factory method that gets the appropriate logger from Log4j and
     * wraps it in a JetspeedLogger
     */        
    public static JetspeedLogger getLogger(String loggerName)
    {
        // This test needed to ensure correct init sequence between this and services that log.
        if (!initDone)
        {
            synchronized (JetspeedLogFactoryService.class)
            {
                 if (!initDone)
                { 
                    try 
                    {
                        new JetspeedLogFactoryService().init();
                    }
                    catch(Exception e)
                    {
                        System.err.println("Init failed no logging available" + e.getMessage());
                        e.printStackTrace();
                    }
                 }
            }
        }
        Logger newLog = LogManager.getLogger(loggerName);
        JetspeedLogger newLogger = new JetspeedLogger(newLog);
        return newLogger;
    }
} // class JetspeedLogFactoryService


