/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.jsp.tags;

// java classes
import java.util.Hashtable;
import java.util.StringTokenizer;

// jsp api
import javax.servlet.jsp.PageContext;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.tagext.BodyTagSupport;

// Turbine Classes 
import org.apache.turbine.util.RunData;
import org.apache.turbine.services.jsp.JspService;

// jetspeed
import org.apache.jetspeed.modules.ParameterLoader;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.jetspeed.services.security.PortalResource;
import org.apache.jetspeed.om.registry.Parameter;
import org.apache.jetspeed.om.registry.PortletEntry;
import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.om.security.JetspeedUser;
import org.apache.jetspeed.services.Registry;

/**
 * Supporting class for the parameter style tag.
 * Sends a parameter rendered using specific style to the output stream.
 * 
 * The following tag attributes are supported:
 * 
 * <UL>
 * <LI><code>name</code>: parameter name (required).</li>
 * <LI><code>style</code>: parameter style name (required)</LI>
 * <LI><code>value</code>: parameter current value</LI>
 * <LI><code>portlet</code>: portlet name to check security against</LI>
 * </UL>
 * <p>Note: tag body may also contain parameter style options in format: option1=value1;optionN=valueN. Check
 * documentation for individual parameter style to see what options are supported</p>
 * <p>Note: Use care when specifying style options in the tag body - the body is not cleansed to remove
 * embedded carriage returns and tabs.</p>
 * Examples:
 * <UL>
 * <LI><code>&lt;jetspeed:parameterStyle name="portlet-list" style="RegistryEntryListBox"/&gt;</CODE>
 * <LI><code>&lt;jetspeed:parameterStyle name="skin-list" style="RegistryEntryListBox"&gt;registry=Skin&lt;/jetspeed:parameterStyle/&gt;</CODE>
 * <LI><code>&lt;jetspeed:parameterStyle name="control-list" style="RegistryEntryListBox" value="TabControl"&gt;registry=PortletControl&lt;/jetspeed:parameterStyle/&gt;</CODE>
 * </UL>
 * 
 * @author <a href="mailto:morciuch@apache.org">Mark Orciuch</a>
 * @version $Id: JetspeedParameterStyleTag.java,v 1.3 2003/07/23 19:50:19 morciuch Exp $
 */
public class JetspeedParameterStyleTag extends BodyTagSupport
{
    /**
     * Static initialization of the logger for this class
     */    
    private static final JetspeedLogger logger = JetspeedLogFactoryService.getLogger(JetspeedParameterStyleTag.class.getName());
    
    /**
     * name parameter defines parameter name
     */
    private String name = null;

    /**
     * name parameter defines parameter style
     */
    private String style = null;

    /**
     * name parameter defines current parameter style value
     */
    private String value = null;

    /**
     * name parameter defines portlet name to check security against
     */
    private String portlet = null;

    /**
     * The setter for name parameter
     * 
     * @param value
     */
    public void setName(String value)
    {
        this.name = value;
    }

    /**
     * The setter for value parameter
     * 
     * @param value
     */
    public void setValue(String value)
    {
        this.value = value;
    }

    /**
     * The setter for syle parameter
     * 
     * @param value
     */
    public void setStyle(String value)
    {
        this.style = value;
    }

    /**
     * The setter for value parameter
     * 
     * @param value
     */
    public void setPortlet(String value)
    {
        this.portlet = value;
    }

    /**
     * 
     * @return code
     * @exception JspException
     */
    public int doStartTag() throws JspException 
    {
        return EVAL_BODY_TAG;
    }

    /**
     * 
     * @return code
     * @exception JspException
     */
    public int doEndTag() throws JspException 
    {
        RunData data = (RunData) pageContext.getAttribute(JspService.RUNDATA, PageContext.REQUEST_SCOPE);    
        String result = null;

        try
        {

            // See if body contains any parameter options
            String body = this.getBodyContent() == null ? null : this.getBodyContent().getString();
            Hashtable options = new Hashtable();

            if (body != null && !body.trim().equalsIgnoreCase(""))
            {
                StringTokenizer st = new StringTokenizer(body, ";");
                String prefix = this.name + ".style.";
                while (st.hasMoreTokens())
                {
                    StringTokenizer pair = new StringTokenizer(st.nextToken(), "=");
                    if (pair.countTokens() == 2)
                    {
                        options.put(prefix + pair.nextToken().trim(), pair.nextToken().trim());
                    }
                }
            }

            boolean canAccess = true;

            // If portlet name is specified, it will be used to check security for the parameter
            if (this.portlet != null)
            {
                // Retrieve registry entry and its parameter
                PortletEntry entry = (PortletEntry) Registry.getEntry(Registry.PORTLET, this.portlet);
                Parameter param = entry.getParameter(this.name);

                // Verify security for the parameter
                canAccess = JetspeedSecurity.checkPermission((JetspeedUser) data.getUser(), 
                                                             new PortalResource(entry, param), 
                                                             JetspeedSecurity.PERMISSION_CUSTOMIZE);
            }

            if (canAccess)
            {
                result = ParameterLoader.getInstance().eval(data, 
                                                            this.style, 
                                                            this.name, 
                                                            this.value, 
                                                            options);                
            }

            pageContext.getOut().print(result);


        }
        catch (Exception e)
        {
            result = "<input type=\"text\" name=\"" + this.name + "\" value=\"" + this.value + "\"";

            String message = "Error processing portlet (PortletTag): [" + name + "]";
            logger.error(message, e);
            try
            {
                pageContext.getOut().print(result);
            }
            catch (java.io.IOException ioe)
            {
            }
        }

        return EVAL_PAGE;
    }

}