/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.forward;

import java.util.Collection;
import java.util.Map;

import org.apache.turbine.services.Service;
import org.apache.turbine.util.RunData;
import org.apache.turbine.util.DynamicURI;

import org.apache.jetspeed.services.forward.configuration.Forward;
import org.apache.jetspeed.services.forward.configuration.PortletForward;


/**
 * <P>This is the interface to the Jetspeed Forward services.
 *    The interface defines methods for forwarding navigation to 
 *    other pages or panes in the portal. The Forward service
 *    provides an abstraction, by removing the hard-coding of
 *    portal resources in your actions. Instead, all forward targets
 *    are defined in a centralized configuration file. By using the 
 *    forward service, you use logical forward names in your java code.</P>
 *
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a>
 * @version $Id: ForwardService.java,v 1.4 2003/04/14 19:01:45 taylor Exp $
 */

public interface ForwardService extends Service
{

    /** The name of this service */
    public String SERVICE_NAME = "ForwardService";


    /**
     *  Forward to a specific forward by name.
     *  All parameters are resolved statically (via the forward definition)
     *
     * @param rundata The turbine rundata context for this request.     
     * @param forwardName Forward to this abstract forward name.
     * @return DynamicURI the full link to the referenced page
     */
    public DynamicURI forward(RunData rundata, String forwardName);


    /**
     *  For the given portlet and given action, forward to the target
     *  defined in the forward configuration for the portlet + action.
     *  All parameters are resolved statically (via the forward definition)     
     *
     * @param portlet The name of the portlet for which we are forwarding.
     * @param target A logical target name. Portlets can have 1 or more targets.
     * @return DynamicURI the full link to the referenced page
     */
    public DynamicURI forward(RunData rundata, String portlet, String target);


    /**
     *  Forward to a specific forward by name.
     *  Parameters are resolved both statically and dynamically, with the 
     *  dynamic parameter overriding the static parameter definitions.
     *
     * @param rundata The turbine rundata context for this request.     
     * @param forwardName Forward to this abstract forward name.
     * @param parameters The dynamic Validation Parameters used in creating validation forwards
     * @return DynamicURI the full link to the referenced page
     */
    public DynamicURI forwardDynamic(RunData rundata, String forwardName, Map parameters);


    /**
     *  For the given portlet and given action, forward to the target
     *  defined in the forward configuration for the portlet + action.
     *  Parameters are resolved both statically and dynamically, with the 
     *  dynamic parameter overriding the static parameter definitions.     
     *
     * @param portlet The name of the portlet for which we are forwarding.
     * @param target A logical target name. Portlets can have 1 or more targets.
     * @param parameters The dynamic Validation Parameters used in creating validation forwards     
     * @return DynamicURI the full link to the referenced page
     */
    public DynamicURI forwardDynamic(RunData rundata, 
                                 String portlet, 
                                 String target,
                                 Map parameters);

    
    /**
     * Get a collection of all forwards in the system.
     *
     * @return Collection of all forward definitions
     */
    public Collection getForwards();

    /**
     * Get a collection of all portlet forwards in the system.
     *
     * @return Collection of all portlet forward definitions
     */
    public Collection getPortletForwards();

    /**
     * Lookup a single forward definition by forward name
     *
     * @param  forwardName The name of the Forward to find
     * @return Forward The found forward definition or null if not found
     */
    public Forward getForward(String forwardName);

    /**
     * Lookup a single portlet forward definition by portlet name + target name
     *
     * @param  portlet The name of the portlet in the Portlet Forward to find
     * @param  target The name of the target in the Portlet Forward to find     
     * @return Forward The found portlet forward definition or null if not found
     */
    public PortletForward getPortletForward(String portlet, String target);

}
