package org.apache.jetspeed.services.customlocalization;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

import java.util.Locale;
import java.util.ResourceBundle;

import org.apache.turbine.services.localization.TurbineLocalizationService;
import org.apache.turbine.services.resources.TurbineResources;
import org.apache.turbine.util.RunData;

import org.apache.jetspeed.om.security.JetspeedUser;

/**
 * <p>This class is the single point of access to all localization
 * resources.  It caches different ResourceBundles for different
 * Locales.</p>
 * 
 * Work in the same way of turbine except for getLocale(RunData data),
 * turbine read the accept-language header in a http request,
 * instead this method read the user.getPerm("language")
 * from the RunData to obtain the language choice by the user
 * without the browser language rule.
 * If a user not change the language with a ChangeLanguagePortlet,
 * and a user.getPerm("language")are not set,
 * the "Accept-Language" header are read.
 * 
 * @author <a href="mailto:desmax74@yahoo.it">Dess Massimiliano</a>
 * @version $Id: JetspeedLocalizationService.java,v 1.6 2003/05/22 17:45:18 morciuch Exp $
 */
public class JetspeedLocalizationService extends TurbineLocalizationService implements CustomLocalizationService
{

    /**
     * Initialize list of default bundle names.
     * 
     * @param ignored
     */
    protected void initBundleNames(String ignored[])
    {
        bundleNames = TurbineResources.getStringArray("locale.default.bundles");
        String name = TurbineResources.getString("locale.default.bundle");
        if (name != null && name.length() > 0)
        {
            if (bundleNames == null || bundleNames.length <= 0)
            {
                bundleNames = (new String[] {name});
            }
            else
            {
                String array[] = new String[bundleNames.length + 1];
                array[0] = name;
                System.arraycopy(bundleNames, 0, array, 1, bundleNames.length);
                bundleNames = array;
            }
        }
        if (bundleNames == null)
        {
            bundleNames = new String[0];
        }
    }

    /**
     * Call getDefaultBundleName() of turbine
     * 
     * @return 
     */
    public String getDefaultBundleName()
    {
        return bundleNames.length > 0 ? bundleNames[0] : "";
    }


    /**
     * This method read if a user has set getPerm("language")
     * to use another language or not.
     * If not set , accept-language of the request are returned.
     * 
     * @param data
     * @return 
     */
    public final Locale getLocale(RunData data)
    {
        JetspeedUser user = (JetspeedUser) data.getUser();
        if (user == null)
        {
            return getLocale(data.getRequest().getHeader(CustomLocalizationService.ACCEPT_LANGUAGE));
        }
        else
        {
            String lang = "null";

            try
            {
                if (user.getPerm("language") == null)
                {
                    return getLocale(data.getRequest().getHeader(CustomLocalizationService.ACCEPT_LANGUAGE));
                }
                else
                {
                    lang = user.getPerm("language").toString();
                    Locale locale = new Locale(lang, "");
                    return locale;
                }
            }
            catch (Exception use)
            {
                return getLocale(data.getRequest().getHeader(CustomLocalizationService.ACCEPT_LANGUAGE));
            }
        }
    }

    /**
     * Call searchKey(Locale locale, String key) to search the key in the Bundles
     * 
     * @param bundleName
     * @param locale
     * @param key
     * @return 
     */
    public String getString(String bundleName, Locale locale, String key)
    {
        return searchKey(locale,key);
    }

    /**
     * Search the key in the first bundle, if is not found
     * search in the list of bundles
     * 
     * @param locale
     * @param key
     * @return 
     */
    private  String searchKey(Locale locale, String key)
    {
        String keyTemp=null;
        int i=0;
        boolean find=false;
        ResourceBundle rb ;

        while ((null==keyTemp)&&(!find)&&(i<bundleNames.length))
        {
            rb = getBundle(bundleNames[i], locale);
            keyTemp=super.getStringOrNull(rb,key);
            if (keyTemp!=null)
            {
                find=true;
            }
            else i++;
        }
        return keyTemp;
    }


    private String bundleNames[];

}
