/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.modules.actions.portlets;

import java.util.Iterator;
import java.util.Map;

import org.apache.jetspeed.modules.actions.portlets.security.SecurityConstants;
import org.apache.jetspeed.om.registry.RegistryEntry;
import org.apache.jetspeed.om.registry.SkinEntry;
import org.apache.jetspeed.portal.portlets.VelocityPortlet;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.jetspeed.services.Registry;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

/**
 * This action enables to update the skin entries
 *
 * @author <a href="mailto:caius1440@hotmail.com">Jeremy Ford</a>
 * @version $Id: SkinUpdateAction.java,v 1.5 2003/07/23 19:50:12 morciuch Exp $
 */
public class SkinUpdateAction extends RegistryUpdateAction
{

    private static final String PARAMETER = "parameter.";
    private static final String SKIN_UPDATE_PANE = "SkinForm";
    
    /**
     * Static initialization of the logger for this class
     */    
    private static final JetspeedLogger logger = JetspeedLogFactoryService.getLogger(SkinUpdateAction.class.getName());     
    
    public SkinUpdateAction()
    {
        registryEntryName = "skinname";
        registry = Registry.SKIN;
        pane = SKIN_UPDATE_PANE;
    }

    /**
     * Subclasses must override this method to provide default behavior
     * for the portlet action
     */
    /**
     * Build the normal state content for this portlet.
     *
     * @param portlet The velocity-based portlet that is being built.
     * @param context The velocity context for this request.
     * @param rundata The turbine rundata context for this request.
     */
    protected void buildNormalContext(
        VelocityPortlet portlet,
        Context context,
        RunData rundata)
        throws Exception
    {
        super.buildNormalContext(portlet, context, rundata);

        String mode =
            rundata.getParameters().getString(SecurityConstants.PARAM_MODE);

        if (mode != null
            && (mode.equals(SecurityConstants.PARAM_MODE_DELETE)
                || mode.equals(SecurityConstants.PARAM_MODE_UPDATE)))
        {
            String skinName = rundata.getParameters().getString("skinname");
            SkinEntry skinEntry =
                (SkinEntry) Registry.getEntry(Registry.SKIN, skinName);
            context.put("entry", skinEntry);
        }
    }

    /**
     * @see org.apache.jetspeed.modules.actions.portlets.RegistryUpdateAction#updateRegistryEntry(org.apache.turbine.util.RunData, org.apache.jetspeed.om.registry.RegistryEntry)
     */
    protected void updateRegistryEntry(
        RunData rundata,
        RegistryEntry registryEntry) throws Exception
    {
        super.updateRegistryEntry(rundata, registryEntry);
        updateParameters(rundata, (SkinEntry) registryEntry);

    }

    /**
     * Populates the user's temp storage with form data
     * @param rundata The turbine rundata context for this request.
     */
    protected void resetForm(RunData rundata)
    {
        super.resetForm(rundata);

        Object[] keys = rundata.getParameters().getKeys();
        if (keys != null)
        {
            for (int i = 0; i < keys.length; i++)
            {
                String key = (String) keys[i];

                if (key.startsWith(PARAMETER))
                {
                    String parameterValue =
                        rundata.getParameters().getString(key);

                    if (parameterValue != null && parameterValue.length() > 0)
                    {
                        rundata.getUser().setTemp(key, parameterValue);
                    }
                }
            }
        }
    }

    /**
     * Adds parameters to a skin entry
     * @param rundata The turbine rundata context for this request.
     * @param skinEntry
     */
    private void updateParameters(RunData rundata, SkinEntry skinEntry)
    {
        Object[] keys = rundata.getParameters().getKeys();
        if (keys != null)
        {
            for (int i = 0; i < keys.length; i++)
            {
                String key = (String) keys[i];

                if (key.startsWith(PARAMETER))
                {
                    String parameterValue =
                        rundata.getParameters().getString(key);

                    if (parameterValue != null && parameterValue.length() > 0)
                    {
                        String parameterName =
                            key.substring(PARAMETER.length());
                        skinEntry.removeParameter(parameterName);
                        skinEntry.addParameter(parameterName, parameterValue);
                    }
                }
            }
        }
    }

    /**
     * Clears the temporary storage of any data that was used
     * @param rundata
     */
    protected void clearUserData(RunData rundata)
    {
        try
        {
            super.clearUserData(rundata);

            Map tempStorage = rundata.getUser().getTempStorage();
            if (tempStorage != null)
            {
                Iterator keyIter = tempStorage.keySet().iterator();
                while (keyIter.hasNext())
                {
                    Object keyObj = keyIter.next();
                    if (keyObj instanceof String)
                    {
                        String key = (String) keyObj;
                        if (key.startsWith(PARAMETER))
                        {
                            keyIter.remove();
                        }
                    }
                }
            }
        }
        catch (Exception e)
        {
            if (logger.isDebugEnabled())
            {
                logger.debug("SkinUpdateAction: Failed to clear user data");
            }
        }
    }
}
