/*
  iSlide.js
  
  iSlideMaker
    http://sourceforge.jp/projects/islidemaker/simple/

  Copyright(c) 2010, Isao Hara, isao-hara@users.sourceforge.jp

  All rights reserved. This program is made available under the terms of the
  Eclipse Public License v1.0 which accompanies this distribution, and is
  available at http://www.eclipse.org/legal/epl-v10.html

  Contributors:  Isao Hara.

*/

////////////////
var Slide;
var touchX;
var touchY;
var touchMX;
var touchMY;
var orientation;

///////////////

function iSlide_init(){
  Slide = new iSlide();
  Slide.showSlideByIndex(0);
}

function setSlideIndex(n){
  Slide.showSlideByIndex(n);
}

function hasClass(element, className) {
  if (!element.className) return false;
  return (element.className.search('(^|\\s)' + className + '(\\s|$)') != -1);
}

function hasValue(element, value) {
  if (!element) return false;
  return (element.search('(^|\\s)' + value + '(\\s|$)') != -1);
}

function removeClass(element,className) {
  if (!element) return;
  element.className = element.className.replace(new RegExp('(^|\\s)'+className+'(\\s|$)'), RegExp.$1+RegExp.$2);
}

function addClass(element,className) {
  if (!element || hasClass(element, className)) return;
  if (element.className) {
    element.className += ' '+className;
  } else {
    element.className = className;
  }
}

function GetElementsWithClassName(top, elementName,className) {
  
  var allElements = top.getElementsByTagName(elementName);
  var elemColl = new Array();
  for (var i = 0; i< allElements.length; i++) {
    if (hasClass(allElements[i], className)) {
      elemColl[elemColl.length] = allElements[i];
    }
  }
  return elemColl;
}

function isParentOrSelf(element, id) {
  if (element == null || element.nodeName=='BODY') return false;
  else if (element.id == id) return true;
  else return isParentOrSelf(element.parentNode, id);
}

function getSlideId(element) {
  if (element == null || element.nodeName=='BODY') return null;
  else if (element.id != null && element.id.substr(0,5) == 'slide') return element.id;
  else return getSlideId(element.parentNode);
}

function nodeValue(node) {
  var result = "";
  if (node.nodeType == 1) {
    var children = node.childNodes;
    for (var i = 0; i < children.length; ++i) {
      result += nodeValue(children[i]);
    }		
  }
  else if (node.nodeType == 3) {
    result = node.nodeValue;
  }
  return(result);
}

////////  iSlide Class
var iSlide = function(){
  if(window.innerWidth>window.innerHeight){
    orientation = 'landscape';
    this.WinWidth=window.innerWidth;
    this.WinHeight=window.innerHeight;
    this.scale = 1;
  }else{
    orientation = 'portrate';
    this.WinWidth=window.innerHeight;
    this.WinHeight=window.innerWidth;
    this.scale = window.innerWidth/ window.innerHeight;
  }


  this.current_incs=0;
  this.thumbMaxX=5;
  this.current_slide_index = -1;
  this.current_slide_id = null;
  this.mode;
  this.slides=null;
  this.incrementals=null;
  this.header=null;
  this.footer=null;
  this.num_slides=0;

  this.initSlide();
}

iSlide.prototype.initSlide=function(){
  var top = document.getElementById('presentation');
  if(!top){
    alert('No presentation dvi...');
    return;
  }

  this.slides = GetElementsWithClassName(top, 'div', 'slide');
  this.num_slides=this.slides.length;
  this.incrementals = this.createIncrementals();

  this.createControls();
  this.slideLabel();

  this.header = document.getElementById('header');
  this.header.setAttribute('class','header');
  this.footer = document.getElementById('footer');
  this.footer.setAttribute('class','footer');
  this.num_slides=this.slides.length;

  for(var i=0;i<this.slides.length; i++){
    var header = this.header.cloneNode(true);
    var footer = this.footer.cloneNode(true);
    var obj = this.slides[i];

    obj.setAttribute('id', 'slide'+i);
    header.setAttribute('id','');
    footer.setAttribute('id','');

    obj.insertBefore(header, obj.childNodes[0]);
    obj.appendChild(footer);

    if(navigator.userAgent.match("iPad")){
    
      obj.addEventListener('touchstart', selectSlide, false);
      obj.addEventListener('touchmove', moveEventHandle, false);
      obj.addEventListener('touchend', touchSlide, false);
      
    }else{
      document.onmousedown=selectSlideWithMouse;
      document.onmousemove=moveEventHandleWithMouse;
      document.onmouseup=touchSlideWithMouse;
    }

    obj.style.width=this.WinWidth+'px';
    obj.style.height=this.WinHeight+'px';
    obj.style.position='absolute';
    obj.style.display='none';
  }
}

iSlide.prototype.updateOrientation=function() {
  if(window.innerWidth > window.innerHeight){
    orientation = 'landscape';
    this.WinWidth=window.innerWidth;
    this.WinHeight=window.innerHeight;
    this.scale = 1;
  }else{
    orientation = 'portrate';
    this.WinWidth=window.innerHeight;
    this.WinHeight=window.innerWidth;
    this.scale = window.innerWidth/ window.innerHeight;
  }
}

iSlide.prototype.slideLabel=function() {
  var list = document.getElementById('jumplist');
  smax = this.num_slides;
  for (var n = 0; n < smax; n++) {
    var obj = this.slides[n];

    var did = 'slide' + n.toString();
    obj.setAttribute('id',did);

    var otext = '';
    var menu = obj.firstChild;
    if (!menu) continue; 
    while (menu && menu.nodeType == 3) {
      menu = menu.nextSibling;
    }
    if (!menu) continue;
    
    var menunodes = menu.childNodes;
    for (var o = 0; o < menunodes.length; o++) {
      otext += nodeValue(menunodes[o]);
    }

    list.options[list.length] = new Option(n + ' : '  + otext, n);
  }
}

iSlide.prototype.createControls = function() {
  var controlsDiv = document.getElementById("controls");
  var hideDiv, hideList = '';

  if (!controlsDiv) return;

   controlsDiv.innerHTML = '<div id="navLinks">' +
	'<button onClick="hideAllSlides();fileSelector();">End<\/button>'+
	'<button onClick="hideAllSlides();editCurrentSlide();">Edit<\/button>'+
	'<button onClick="showThumbnail();">Thumb<\/button>'+
	'<select id="jumplist" onchange="jumpSlide();"><\/select>' +
	'<\/div>';

  controlsDiv.style.display='none'; 
}

iSlide.prototype.displaySlide = function(idx, scale, x, y, frame){
  var obj = this.slides[idx];
  var x0=this.WinWidth*scale;
  var y0=this.WinHeight*scale;
  var xx = (this.WinWidth/2) * scale - this.WinWidth/2 + 5 + (5+x0)*x;
  var yy  = (this.WinHeight/2) * scale - this.WinHeight/2 + 5+ (5+y0)*y;

  obj.style.webkitTransform='scale('+scale+')';

  if (frame){
    obj.style.border="1px solid black";
  }else{
    xx=xx-5;
    yy=yy-5;
    obj.style.border="1px none black";
  }

  obj.style.left = xx+'px';
  obj.style.top = yy+'px';

  obj.style.display='block';
}

iSlide.prototype.showThumbnail = function(){
  var maxX = this.thumbMaxX;
  if(this.slides == null) return;

  this.mode='thumbnail';
  this.current_slide_index = -1;

  scale = (this.WinWidth - 10)/maxX/this.WinWidth;

  var y = 0;
  var x = 0;

  for(var i=0;i<this.num_slides;i++){
    y = Math.floor(i/maxX);
    x = i % maxX;
    this.displaySlide(i, scale, x, y, true);
  }

  var svg_objs = document.getElementsByTagName('object');
  for(var i=0;i<svg_objs.length;i++){
    svg_objs[i].style.display='none';
  }
}

iSlide.prototype.currentSlide = function() {
  var cs = document.getElementById('currentSlide');
  cs.innerHTML = '<span id="csHere">' + this.current_slide_index + '<\/span> ' + 
		'<span id="csSep">\/<\/span> ' + 
		'<span id="csTotal">' + (this.num_slides-1) + '<\/span>';
  if (this.current_slide_index == 0) { cs.style.visibility = 'hidden'; }
   else { cs.style.visibility = 'visible'; }
}

iSlide.prototype.getIncrementals=function(obj) {
  var incrs = new Array();

  if (!obj) return incrs;

  var children = obj.childNodes;

  for (var i = 0; i < children.length; i++) {
    var child = children[i];
    if (hasClass(child, 'incremental')) {
      if (child.nodeName == 'OL' || child.nodeName == 'UL') {
        removeClass(child, 'incremental');
        for (var j = 0; j < child.childNodes.length; j++) {
          if (child.childNodes[j].nodeType == 1) {
            addClass(child.childNodes[j], 'incremental');
          }
        }
      } else {
        incrs[incrs.length] = child;
        removeClass(child,'incremental');
      }
    }
    if (hasClass(child, 'show-first')) {
      if (child.nodeName == 'OL' || child.nodeName == 'UL') {
        removeClass(child, 'show-first');
        if (child.childNodes[isGe].nodeType == 1) {
          removeClass(child.childNodes[isGe], 'incremental');
        }
      } else {
        incrs[incrs.length] = child;
      }
    }
    incrs = incrs.concat(this.getIncrementals(child));
  }
  return incrs;
}

iSlide.prototype.createIncrementals=function() {
  this.incrementals = new Array();
  for (var i = 0; i < this.num_slides; i++) {
    this.incrementals[i] = this.getIncrementals(this.slides[i]);
  }
  return this.incrementals;
}

iSlide.prototype.jumpSlide=function(){
  var jl = document.getElementById('jumplist');
  var pos  = parseInt(jl.value);
  this.showSlideByIndex(pos);
}

iSlide.prototype.showSlideByIndex=function(idx){
  if(!this.slides) return false;

  var incs = this.incrementals[idx];

  for(var i=0;i < incs.length;i++){
    removeClass(incs[i], "current");
    addClass(incs[i], "incremental");
  }
  this.current_incs = 0;
  this.showSlide(idx);
}

iSlide.prototype.showSlide=function(idx){
  var svg_objs = document.getElementsByTagName('object');
  for(var i=0;i<svg_objs.length;i++){ svg_objs[i].style.display='block'; }

  for(var i=0; i<this.num_slides; i++){
    if(i != idx) this.slides[i].style.display='none';
  }
  hideControls();
  this.current_slide_index = idx;
  this.displaySlide(idx, this.scale, 0, 0, false);
  this.currentSlide();
//  drawCanvas(this.slides[idx]);
  this.mode='presentation';
}

iSlide.prototype.hideAllSlides=function(){
  for(i in this.slides){
    this.slides[i].style.display='none';
  }
  hideControls();
}

iSlide.prototype.nextSlide=function(){
  if(this.current_slide_index < 0  || this.current_slide_index == this.num_slides -1 ) return false;
  this.current_incs = 0;
  this.showSlideByIndex(this.current_slide_index + 1);
}

iSlide.prototype.prevSlide=function(){
  if(this.current_slide_index <= 0 ) return false;
  this.current_incs = 0;
  this.showSlideByIndex(this.current_slide_index -1);
}

iSlide.prototype.modeEditFile=function(){
    this.mode = 'editfile';
}

iSlide.prototype.showCurrentSlide=function(){
  if(this.current_slide_index >= 0 && this.current_slide_index < this.num_slides)
      this.showSlideByIndex(this.current_slide_index);
}

iSlide.prototype.nextStep=function(){
  var incs = this.incrementals[this.current_slide_index];

  if(incs.length  == 0 || incs.length  < this.current_incs){
    this.nextSlide();
  }else if(incs.length  > Slide.current_incs){
    if(this.current_incs > 0) removeClass(incs[this.current_incs-1], "current");
    removeClass(incs[this.current_incs], "incremental");
    addClass(incs[this.current_incs], "current");
    this.current_incs += 1;
  }else if(incs.length  == this.current_incs){
    if(this.current_incs > 0) removeClass(incs[this.current_incs-1], "current");
    this.current_incs += 1;
  }
}

////// 
function hideControls(){
  var controlsDiv = document.getElementById("controls");
  controlsDiv.style.display='none'; 
}

function hideAllSlides(){
  Slide.hideAllSlides();
}

function jumpSlide(){
  Slide.jumpSlide();
}
function modeEditFile(){
 Slide.modeEditFile();
}

function showThumbnail(){
  Slide.showThumbnail();
  hideControls();
}

function editCurrentSlide(){
  if(Slide.current_slide_index < 0) return;
  editSlideByIndex(Slide.current_slide_index);
  Slide.mode = 'EditSlide';
}

//// Event Handler
function selectSlide(e){
  if(e.touches.length != 1) return false;

  e.preventDefault();

  touchMX=touchX=e.touches[0].clientX;
  touchMY=touchY=e.touches[0].clientY;

  if(Slide.mode != 'thumbnail') return false;

  var current_slide_id = getSlideId(e.touches[0].target);

  if( current_slide_id !==  null){
    Slide.current_slide_index = parseInt(current_slide_id.substr(5));
  }else {
    Slide.current_slide_index = -1;
  }

  return false;
}


function touchSlide(e){
  e.preventDefault();

  var controlsDiv = document.getElementById("controls");

  if(controlsDiv.style.display != 'none'){
    controlsDiv.style.display='none';
    return false;
  }

  if(Slide.mode == 'thumbnail') {
    Slide.showCurrentSlide();
    return false;
  }

  if(Slide.mode != 'presentation' && Slide.mode != 'editfile') return false;

  if(touchX < Slide.scale*Slide.WinWidth*0.1 && touchY > Slide.scale*Slide.WinHeight*0.9 ){
    if(Slide.mode == 'editfile'){
      hideItemById('presentation');
      showItemById('menuDiv');
      showItemById('editDiv');
      return false;
    }else{
      showThumbnail();
      return false;
    }
  }

  var deltaX = touchMX  - touchX;
  var deltaY = touchMY  - touchY;

  if(deltaX > Slide.scale*Slide.WinWidth*0.1){
    Slide.nextSlide();
  }else if(deltaX < -Slide.scale*Slide.WinWidth*0.1){
    Slide.prevSlide();
  }else if(touchY > Slide.scale*Slide.WinHeight - 20){
    if(Slide.mode == 'editfile'){
      hideItemById('presentation');
      showItemById('menuDiv');
      showItemById('editDiv');
   }else{
      controlsDiv.style.display='block';
   }
  }else { 
    Slide.nextStep();
  }

}
function moveEventHandle(e){
  if(e.touches.length != 1) return false;
  e.preventDefault();

  touchMX=e.touches[0].clientX;
  touchMY=e.touches[0].clientY;
  return false;
}

function selectSlideWithMouse(e){
  touchMX=touchX=e.pageX;
  touchMY=touchY=e.pageY;

  if(Slide.mode != 'thumbnail') return false;

  var current_slide_id = getSlideId(e.target);

  if( current_slide_id !==  null){
    Slide.current_slide_index = parseInt(current_slide_id.substr(5));
  }else {
    current_slide_index = -1;
  }

  return false;
}

function touchSlideWithMouse(e){
  var controlsDiv = document.getElementById("controls");

  if(controlsDiv.style.display != 'none'){
    controlsDiv.style.display='none';
    return false;
  }

  if(Slide.mode == 'thumbnail') {
    Slide.showCurrentSlide();
    return false;
  }

  if(Slide.mode != 'presentation' && Slide.mode != 'editfile') return false;

  if(touchX < Slide.scale*Slide.WinWidth*0.1 && touchY > Slide.scale*Slide.WinHeight*0.9 ){
    if(Slide.mode == 'editfile'){
      hideItemById('presentation');
      showItemById('menuDiv');
      showItemById('editDiv');
      return false;
    }else{
      showThumbnail();
      return false;
    }
  }

  var deltaX = touchMX  - touchX;
  var deltaY = touchMY  - touchY;

  if(deltaX > Slide.scale*Slide.WinWidth*0.1){
    Slide.nextSlide();
  }else if(deltaX < -Slide.scale*Slide.WinWidth*0.1){
    Slide.prevSlide();
  }else if(touchY > Slide.scale*Slide.WinHeight - 20){
    if(Slide.mode == 'editfile'){
      hideItemById('presentation');
      showItemById('menuDiv');
      showItemById('editDiv');
   }else{
      controlsDiv.style.display='block';
   }
  }else { 
    Slide.nextStep();
  }

}

function moveEventHandleWithMouse(e){
  touchMX=e.pageX;
  touchMY=e.pageY;
  return false;
}


////////
function updateOrientation(){
  Slide.updateOrientation();

  if(Slide.mode=='thumbnail'){
    showThumbnail();
  }else if(Slide.mode=='presentation'){
    Slide.showCurrentSlide();
  }
}

function toSVGElement(str, w, h){
  var xmlsvg = "xmlns:svg=\"http://www.w3.org/2000/svg\""
  var parser = new DOMParser();
  var header = "<svg:svg width=\""+w+"\" height=\""+h+"\" "+xmlsvg+">"
  var footer = "</svg:svg>";
  var xmlDoc = parser.parseFromString(header+str+footer, "text/xml");
  var ele =  document.importNode(xmlDoc.documentElement,true);
  return ele;
}

///////
window.addEventListener('orientationchange', updateOrientation, false);
