/*
 * $Id: ImageConvertTask.java,v 1.2 2002/05/03 14:33:28 ymakise Exp $
 */

/*
 * Copyright (c) 2002, MAKISE Yoshitaro
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. Neither the name of the iModoki nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.imodoki.ant;

import java.io.*;
import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;
import java.awt.image.RenderedImage;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;

/*
  TODO:
   - more encoding option (?)
 */

public class ImageConvertTask extends FileConvertTask {
    private ImageFormatConverter m_converter;

    public ImageConvertTask() {
        m_converter = new ImageIOConverter();
        if (m_converter.available())
            return;
        m_converter = new JAIConverter();
        if (m_converter.available())
            return;
        System.out.println("Warning: No image format converter available!");
        System.out.println("         Image format conversion disabled.");
        m_converter = new CopyConverter();

        /* default output format */
        m_converter.setOutputFormat("PNG");
    }

    public void setOutputFormat(String outputFormat) {
        m_converter.setOutputFormat(outputFormat);
    }

    protected void convert(File srcFile, File destFile) {
        if (srcFile.equals(destFile)) {
            throw new BuildException("file " + srcFile 
                                     + " would overwrite its self");
        }

        log("converting " + srcFile + " to " + destFile, Project.MSG_VERBOSE);

        try {
            m_converter.convert(srcFile, destFile);
        } catch (Exception e) {
            throw new BuildException("convertion failed: " + e.toString());
        }
    }

    private static interface ImageFormatConverter {
        public boolean available();
        public void setOutputFormat(String format);
        public void convert(File srcFile, File destFile) throws Exception;
    }

    /*
     * ХʥΤߤۤƤư褦ˡޤJAI  JDK1.4 
     * 줫ʤĶǤ⥳ѥǽˤ뤿ˡե쥯
     * Ȥ
     *
     * JDK1.4 ڤޤȤʥɤ˺ؤ롣
     */

    /** Converter which uses JAI (Java Advanced Imaging). */
    private static class JAIConverter implements ImageFormatConverter {
        private String m_outputFormat;
        private Class m_classJAI;
        private Method m_methodJAI_create_1;
        private Method m_methodJAI_create_2;

        public JAIConverter() {
            try {
                m_classJAI = Class.forName("javax.media.jai.JAI");
                m_methodJAI_create_1 =
                    m_classJAI.getMethod("create",
                                         new Class[] { String.class,
                                                       Object.class });
                m_methodJAI_create_2 =
                    m_classJAI.getMethod("create",
                                         new Class[] { String.class,
                                                       RenderedImage.class,
                                                       Object.class,
                                                       Object.class,
                                                       Object.class });
            } catch (Exception e) {
                m_classJAI = null;
            }
        }

        public boolean available() {
            return (m_classJAI != null);
        }

        public void setOutputFormat(String format) {
            m_outputFormat = format;
        }

        public void convert(File srcFile, File destFile) throws Exception {
//             RenderedOp img = JAI.create("fileload", srcFile.getPath());
//             JAI.create("filestore", img, destFile.getPath(), outputFormat, null);

            try {
                RenderedImage img = (RenderedImage)
                    m_methodJAI_create_1.invoke(null,
                                                new Object[] {
                                                    "fileload",
                                                    srcFile.getPath()
                                                });
                m_methodJAI_create_2.invoke(null,
                                            new Object[] {
                                                "filestore",
                                                img,
                                                destFile.getPath(),
                                                m_outputFormat,
                                                null
                                            });
            } catch (IllegalAccessException iae) {
                /* Should not happen */
                throw new RuntimeException(iae.toString());
            } catch (InvocationTargetException ite) {
                Throwable te = ite.getTargetException();
                if (te instanceof Exception) {
                    throw (Exception)te;
                } else {
                    throw (Error)te;
                }
            }
        }
    }

    /** Converter which uses ImageIO of J2SE 1.4. */
    private static class ImageIOConverter implements ImageFormatConverter {
        private String m_outputFormat;
        private Class m_classImageIO;
        private Method m_methodImageIO_read;
        private Method m_methodImageIO_write;

        public ImageIOConverter() {
            try {
                m_classImageIO = Class.forName("javax.imageio.ImageIO");
                m_methodImageIO_read =
                    m_classImageIO.getMethod("read",
                                             new Class[] { File.class });
                m_methodImageIO_write =
                    m_classImageIO.getMethod("write",
                                             new Class[] { RenderedImage.class,
                                                           String.class,
                                                           File.class });
            } catch (Exception e) {
                System.out.println("!!! exception: " + e);
                m_classImageIO = null;
            }
        }

        public boolean available() {
            return (m_classImageIO != null);
        }

        public void setOutputFormat(String format) {
            m_outputFormat = format;
        }

        public void convert(File srcFile, File destFile) throws Exception {
//             RenderedImage img = ImageIO.read(srcFile);
//             ImageIO.write(img, outputFormat, new File(destFile));
        
            try {
                RenderedImage img = (RenderedImage)
                    m_methodImageIO_read.invoke(null,
                                                new Object[] { srcFile });
                m_methodImageIO_write.invoke(null,
                                             new Object[] { img,
                                                            m_outputFormat,
                                                            destFile });
            } catch (IllegalAccessException iae) {
                /* Should not happen */
                throw new RuntimeException(iae.toString());
            } catch (InvocationTargetException ite) {
                Throwable te = ite.getTargetException();
                if (te instanceof Exception) {
                    throw (Exception)te;
                } else {
                    throw (Error)te;
                }
            }
        }
    }

    /** Converter which does not convert image format but only copy it. */
    private static class CopyConverter implements ImageFormatConverter {
        private String m_outputFormat;

        public boolean available() { return true; }

        public void setOutputFormat(String format) {
            m_outputFormat = format;
        }

        public void convert(File srcFile, File destFile) throws Exception {
            InputStream in = null;
            OutputStream out = null;

            try {
                in = new FileInputStream(srcFile);
                out = new FileOutputStream(destFile);

                byte[] buf = new byte[4096];
                int len;

                while ((len = in.read(buf)) != -1)
                    out.write(buf, 0, len);
                
            } finally {
                if (in != null)
                    in.close();
                if (out != null)
                    out.close();
            }
        }
    }
}
