#/*
# *  Copyright 2007 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: App.pm 188 2007-02-06 12:36:44Z hikarin $
#

package Zeromin::App;

use strict;

BEGIN {
    my $pkg = __PACKAGE__;
    for my $mtd (qw(bbs key user kernel request)) {
        my $attr = '__' . $mtd;
        no strict 'refs';
        *{"${pkg}::${mtd}"} = sub { $_[0]->{$attr} }
    }
}

sub new {
    my ( $zClass, $iKernel, @args ) = @_;
    bless {
        __kernel  => $iKernel,
        __request => Img0ch::Request->new(@args),
    }, $zClass;
}

sub run {
    my ($zApp)   = @_;
    my $iConfig  = $zApp->kernel()->get_config();
    my $iRequest = $zApp->{__request};
    my $content;
    $iRequest->init($iConfig);

    my $charset = $zApp->encoding();
    my $header  = {};

    $zApp->proxy( \$content, $iRequest );
    if ( $iConfig->get('ZerominUseGzip') ) {
        my $compress = sub {
            require Compress::Zlib;
            my ($content_ref) = @_;
            $$content_ref = Compress::Zlib::memGzip($$content_ref);
            return;
        };
        my $accept = $iRequest->get_header('accept-encoding');
        if ( index( $accept, 'x-gzip' ) >= 0 ) {
            $header->{'content-encoding'} = 'x-gzip';
            $compress->( \$content );
        }
        elsif ( index( $accept, 'gzip' ) >= 0 ) {
            $header->{'content-encoding'} = 'gzip';
            $compress->( \$content );
        }
    }
    $iRequest->send_http_header( $zApp->{__content_type}, $charset, $header );
    print $content;
}

sub proxy {
    my ( $zApp, $content, $iRequest ) = @_;
    $iRequest ||= $zApp->{__request};
    my $class  = $iRequest->param('C');
    my $method = $iRequest->param('M');
    my @args   = $iRequest->param('argv');
    if ( $class and $method ) {
        $class = 'Zeromin::App::' . lc $class;
    }
    elsif ( $iRequest->param('uv') ) {
        $class  = 'Zeromin::App';
        $method = 'null';
    }
    else {
        $zApp->_html( $content, $iRequest );
        return 1;
    }

    eval {
        my $mtd = _load( $class, $method )
            or die "${class}::${method}() not found\n";
        $iRequest->init();
        if ( my $bbs = $iRequest->bbs() ) {
            require Img0ch::BBS;
            $zApp->{__bbs}
                = Img0ch::BBS->new( $zApp->{__kernel}, { id => $bbs, } );
            $zApp->{__key} = $iRequest->key();
        }
        my $ok = $zApp->_valid( ( $zApp->{__bbs} || $zApp->{__kernel} ),
            $iRequest );
        if ( $iRequest->param('html') ) {
            $zApp->{__content_type} = 'text/html';
            ${$content} = $mtd->( $zApp, @args );
        }
        else {
            $zApp->{__content_type} = 'text/javascript';
            $$content = ${
                $iRequest->get_json(
                    {   ok      => 1,
                        loginOK => $ok,
                        data    => $mtd->( $zApp, @args ),
                    }
                )
                };
        }
        return 1;
    };
    if ( my $e = $@ ) {
        $zApp->{__content_type} = 'text/javascript';
        $$content
            = ${ $iRequest->get_json( { ok => 0, loginOK => 0, data => $e } )
            };
        return 0;
    }
    else {
        return 1;
    }
}

sub null {undef}

sub logger {
    my ( $zApp, $success_or_fail, $reason, $params ) = @_;
    my $class = (caller)[0];
    my $zUser = $zApp->user();
    my $user  = $zUser ? $zUser->get_current_user->{id} : 0;
    my $ip    = $zApp->request()->ip_int();

    require Zeromin::Log::Action;
    my $zLogAct = Zeromin::Log::Action->new( $zApp->kernel() );
    $zLogAct->load();
    $zLogAct->add( $class, $user, $ip, $success_or_fail, $reason, $params );
    $zLogAct->save();
    1;
}

sub encoding {
    $_[1] and $_[0]->{__encoding} = $_[1];
    $_[0]->{__encoding} || 'UTF-8';
}

sub _html {
    my ( $zApp, $content, $iRequest ) = @_;
    $zApp->{__content_type} = 'text/html';

    require Img0ch::Template;
    require Zeromin::User;
    my ( $dir, $file );
    my $zUser = Zeromin::User->new( $zApp->{__kernel} );
    $zUser->load();
    $zApp->{__user} = $zUser;

    if ( $zUser->is_initialized() ) {
        $dir  = $iRequest->param('T_DIR')  || 'zeromin';
        $file = $iRequest->param('T_FILE') || 'index';
    }
    else {
        $dir  = 'zeromin';
        $file = 'install';
    }

    my $iTemplate = Img0ch::Template->new(
        $zApp->{__kernel},
        {   dir     => $dir,
            file    => $file,
            version => $iRequest->credit(),
        }
    );
    my $iConfig = $zApp->kernel()->get_config();
    my $self    = $iConfig->get('ZerominScript') || 'zeromin.cgi';
    $iTemplate->param({ ZerominScript => $self });
    $$content = ${ $iTemplate->to_string() };
    return;
}

sub _load {
    my ( $class, $method ) = @_;
    my $file = $class . '.pm';
    $file =~ s{::}{/}g;
    require $file;
    index($method, '_') == 0 and return 0;
    return $class->can($method);
}

sub _valid {
    my ( $zApp, $iObject, $iRequest ) = @_;

    require Zeromin::User;
    my $user  = $iRequest->param('I');
    my $pass  = $iRequest->param('P');
    my $zUser = $zApp->{__user} || Zeromin::User->new( $iObject, $user );
    $zUser->load();

    eval {
        my $is_valid = $zUser->is_valid( $pass, $user );
        if ($is_valid) {
            $zApp->{__user} = $zUser;
            return 1;
        }
        $zApp->logger(
            0,
            'FAILED TO LOGIN FROM %s WITH %s',
            [ $user, $pass ]
        );
        return 0;
    };
}

sub _request {
    my ($zApp)   = @_;
    my $iKernel  = $zApp->kernel();
    my $iRequest = $zApp->request();

    local ( $!, *FH );
    open *FH, '>./dbg.txt'    ## no critic
        or $iKernel->throw_io_exception('dbg.txt');
    require Data::Dumper;
    print {*FH} Data::Dumper::Dumper($iRequest);
    close *FH or $iKernel->throw_io_exception('dbg.txt');
    return;
}

1;
__END__
