/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package an;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import org.opengion.fukurou.business.BizLogic_TABLE;
import org.opengion.fukurou.security.HybsCryptography;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ZipFileUtil;
import org.opengion.hayabusa.common.HybsSystem;

/**
 * アンケート情報登録  (AN0001) の 業務ロジックです。
 * 
 * ここでは、アンケート登録画面で指定された項目一覧のファイル(Excel or Calc)及び
 * 画像ファイルのアーカイブ(Zip)からアンケート情報を展開します。
 * 
 * ここで登録されるデータは、FGJ='0'(仮登録)で登録されます。
 * この登録結果からアンケート回答画面のプレビューを表示し、確定する場合は、FGJを'1'にし、
 * 取り消しする場合は、物理削除します。
 * なお、確定処理は、AN0001A2で、削除処理は、AN0001Dで行います。
 *
 * @og.rev 2010/06/18 新規作成
 * 
 * @version 5.0
 * @author Hiroki Nakamura
 * @since JDK1.6,
 */
public class AN0001A extends BizLogic_TABLE {

	private int	qcnt	= 1;	// 質問のSEQ番号
	private int	acnt	= 1;	// 回答のSEQ番号(質問毎にクリア)

	private String imgOrigPath = null; 	// 画像アーカイブの展開先(仮置き)
	private String imgNewPath = null;	// 画像アーカイブの展開先(本置き)
	private Map<String,String> fileMap = new HashMap<String,String>();	// 画像ファイルの名前と絶対パスのマッピング

	/**
	 * メインカーソルの一番初めで呼ばれるチェックロジックを定義します。
	 * 
	 * ここでは、画像ファイルを仮置きの場所(filetemp/$USREID/imgtemp)に展開しています。
	 * 
	 * @rev 
	 * 
	 * @return 処理が正常終了したか
	 */
	@Override
	protected boolean fstchk() {
		String imgFileArc = var( "fileArc" );
		if( imgFileArc != null && imgFileArc.length() > 0 ) {
			// ANE0016:画像ファイルアーカイブはzip形式のファイルを指定して下さい。ファイル名={0}
			if( !imgFileArc.endsWith( ".zip" ) ) {
				error( 2, "ANE0016", imgFileArc );
			}

			imgOrigPath = HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "FILE_URL" )
							+ var( "CON.USERID" ) + File.separator + "imgtemp" + File.separator;

			String imgOrigFile = HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "FILE_URL" )
							+ var( "CON.USERID" ) + File.separator + imgFileArc;

			if( ( new File( imgOrigFile ).exists() ) ) {
				// Zipファイルを展開
				ZipFileUtil.unCompress( imgOrigPath, imgOrigFile );

				// 0.1 (2010/08/01) ディレクトリの中にある画像でも抽出できるようにする。
				getFileMap( new File( imgOrigPath ), fileMap );

				imgNewPath = HybsSystem.sys( "REAL_PATH" ) + "ankimages" + File.separator ;
			}
		}

		return true;
	}

	/**
	 * メインカーソルの各行(変更区分の各処理の後)で呼ばれるチェックロジックを定義します。
	 * 
	 * @param row 行番号
	 * @return 処理が正常終了したか
	 */
	@Override
	protected boolean allchk( final int row ) {
		if( line( "KBUNRUI" ) != null && line( "KBUNRUI" ).length() > 0
		&& !"TEXT".equals( line( "KBUNRUI" ) ) && !"XTEXT".equals( line( "KBUNRUI" ) )
		&& !"YMD".equals( line( "KBUNRUI" ) ) && !"YM".equals( line( "KBUNRUI" ) ) && !"TEXTAREA".equals( line( "KBUNRUI" ) )
		&& !"RADIO".equals( line( "KBUNRUI" ) ) && !"CHBOX".equals( line( "KBUNRUI" ) ) && !"MENU".equals( line( "KBUNRUI" ) ) && !"SCALE".equals( line( "KBUNRUI" ) ) ) {
			// ANE0014:回答方法は、次のいずれかで指定して下さい。RADIO(ラジオ) CHBOX(チェック) SCALE(スケール) TEXTAREA(複数行) YM(年月) YMD(年月日) TEXT(1行) XTEXT(1行/半角のみ) MENU(プルダウン) (値={0})
			error( 2, "ANE0014", line( "KBUNRUI" ) );
		}

		if( isLine( "QAIMAGE" ) && line( "QAIMAGE" ) != null && line( "QAIMAGE" ).length() > 0 ) {
			// ANE0017:画像が存在しません。ファイル名={0}
			String img = fileMap.get( line( "QAIMAGE" ) );
			if( img == null || img.length() == 0 || !( new File( img ) ).exists() ) {
				error( 2, "ANE0017", line( "QAIMAGE" ) );
			}

//			// ANE0025:画像以外のファイルが指定されています。ファイル名={0}
//			if( !Utils.checkImageName( line( "QAIMAGE" ) ) ) {
//				error( 2, "ANE0025", line( "QAIMAGE" ) );
//			}
		}

		return true;
	}

	/**
	 * メインカーソルの一番初めで呼ばれるロジックを定義します。
	 * 
	 * @return 処理が正常終了したか
	 */
	@Override
	protected boolean first() {
		// [暫定対応] adminユーザーで追加したアンケートは雛形として登録します。
		set( "FGTEMPLATE", "admin".equals( var( "CON.USERID" ) ) ? "1" : "0" );

		// アンケートC採番
		setCdank();

		// 暗号化キーは、アンケートコードを基にして暗号化します。
		set( "CRYPTKEY", ( new HybsCryptography() ).encrypt( var( "CDANK" ) ) );

		set( "NMANK", "-" );
		set( "FGNODPL", "0" );
		set( "FGRESDISP", "1" );
		set( "FGSHUBETSU", "1" );
		if( var( "CDORIG" ) != null ) {
			String orig = "select NMANK,SMANK,FGNODPL,KTSTAGS,IRAIADDRS,FGRESDISP,FGSHUBETSU from AN01 where CDANK='{@CDORIG}' and FGJ='1'";
			sql( orig );
		}

		String sql = "insert into AN01("
			+ "USERID,CDANK,NMANK,SMANK,DYKSTR,DYKEND,FGKOKAI,FGNODPL,KTSTAGS,IRAIADDRS,CHECKKEY,CRYPTKEY,FGTEMPLATE,FGRESDISP,FGSHUBETSU"
			+ ",BIKO,FGJ,DYSET,PGSET,PGPSET,USRSET,DYUPD,PGUPD,PGPUPD,USRUPD"
			+ ") values ("
			+ "'{@CON.USERID}','{@CDANK}','{@NMANK}','{@SMANK}','00000000','00000000','0','{@FGNODPL}','{@KTSTAGS}','{@IRAIADDRS}','{@CDANK}','{@CRYPTKEY}','{@FGTEMPLATE}','{@FGRESDISP}','{@FGSHUBETSU}'"
			+ ",'','0','{@CON.YMDH}','{@CON.PGID}','{@CON.PGPID}','{@CON.USERID}','{@CON.YMDH}','{@CON.PGID}','{@CON.PGPID}','{@CON.USERID}'"
			+ ")";
		sql( sql );

		return true;
	}

	/**
	 * アンケートCを採番します。
	 * 
	 * テンプレートデータの場合(FGTEMPLATE='1')は、初期データのシーケンス番号を被る可能性があるため、
	 * 存在チェックをかけています。
	 */
	private void setCdank() {
		String seq = StringUtil.valueOf( seq( "AN01S01" ) );
		set( "CDANK", ( "0".equals( var( "FGTEMPLATE" ) ) ? "A" : "T" ) + StringUtil.intFill( seq, 9 ) );

		if( "1".equals( var( "FGTEMPLATE" ) ) ) {
			String cnt = "select count(*) CNT from AN01 where CDANK = '{@CDANK}' and FGJ = '1'";
			sql( cnt );
			if( vari( "CNT" ) == 0 ) {
				return;
			}
			else {
				setCdank();
			}
		}
	}

	/**
	 * メインカーソルの各行(変更区分の各処理の後)で呼ばれるロジックを定義します。
	 * 
	 * @param row 行番号
	 * @return 処理が正常終了したか
	 */
	@Override
	protected boolean allrow( final int row ) {
		if( line( "QANAIYO" ) == null || line( "QANAIYO" ).length() == 0 ) {
			return true;
		}

		if( line( "KBUNRUI" ) != null && line( "KBUNRUI" ).length() > 0 ) {
			set( "KBUNRUI", line( "KBUNRUI" ) );
			set( "QAKBN", "Q" );
			updateSuoneLine();
			insertAN02( row );
		}
		else if( var( "QAKBN" ) != null ) {
			set( "QAKBN", "A" );
			insertAN03( row );
		}
		return true;
	}

	/**
	 * アンケート項目マスタに登録します。
	 * 
	 * @param row
	 * @return 処理が正常終了したか
	 */
	private boolean insertAN02( final int row ) {
		String seq = StringUtil.valueOf( seq( "AN02S01" ) );
		set( "CDSITSU", "S" + StringUtil.intFill( seq, 9 ) );
		set( "SEQNO", StringUtil.valueOf( qcnt * 10 ) );

		set( "SITSUGRP"	, isLine( "QAGROUP" )	? line( "QAGROUP" )		: null );
		set( "SITSUSGRP", isLine( "QASGROUP" )	? line( "QASGROUP" )	: null );
		set( "IMGNAI"	, null );
		set( "IMGSNAI"	, null );
		set( "FGKEY"	, isLine( "FGKEY" )		&& "1".equals( line( "FGKEY" ) ) ? "1" : "0" );
		set( "FGMUST"	, isLine( "FGMUST" )	&& "1".equals( line( "FGMUST" ) ) ? "1" : "TEXTAREA".equals( line( "KBUNRUI") ) ? "0" : "1" );
//		set( "KLENGTH"	, isLine( "KLENGTH" )	? line( "KLENGTH" )		: StringUtil.valueOf( Utils.getLength( line( "KBUNRUI" ), 200 ) ) );
		set( "KLENGTH"	, isLine( "KLENGTH" )	? line( "KLENGTH" )		: StringUtil.valueOf( Utils.getSysLength( line( "KBUNRUI" ), 100 ) ) );
		set( "KNOTE"	, isLine( "KNOTE")		? line( "KNOTE" )		: null );
		set( "SUONELINE", isLine( "SUONELINE" )	? linei( "SUONELINE" )	: 1 );
		if( vari( "SUONELINE" ) < 1 ) { set( "SUONELINE", 1 ); }
		set( "RSCODE"	, isLine( "RCODE" )		? line( "RCODE" )		: null );
		set( "INIVAL"	, isLine( "INIVAL" )	? line( "INIVAL" )		: null );
		set( "CSKEY"	, isLine( "CSKEY" )		? line( "CSKEY" )		: null ); 
		set( "CSWAY"	, isLine( "CSWAY" )		? line( "CSWAY" )		: null ); 
		set( "SUCS"		, isLine( "SUCS" )		? linei( "SUCS" )		: 0 ); 
		set( "VDCDTN"	, isLine( "VDCDTN" )	? line( "VDCDTN" )		: null ); 
		set( "SUPTRITSU", isLine( "SUPTRITSU" )	? linei( "SUPTRITSU" )	: 10 ); 
		set( "KAISETSU"	, isLine( "KAISETSU" )	? line( "KAISETSU" )	: null ); 

		// 画像ファイルのパスを求め、ファイルをリネームします。
		if( isLine( "QAIMAGE" ) && line( "QAIMAGE" ) != null && line( "QAIMAGE" ).length() > 0 ) {
			String imgOrigFile = fileMap.get( line( "QAIMAGE" ) );
			String imgNew = "1".equals( var( "FGTEMPLATE" ) ) ? "template/" : "";
			imgNew += var( "CDANK" ) + "/" + var( "CDSITSU" ) + "." + Utils.getSuffix( line( "QAIMAGE" ) );

			set( "IMGNAI", imgNew );
			set( "IMGSNAI", Utils.getSuffix( imgNew ) );

			new File( imgNewPath + imgNew ).getParentFile().mkdirs();
			FileUtil.copy( imgOrigFile, imgNewPath + imgNew );
		}

		String sql = "insert into AN02("
			+ "USERID,CDANK,CDSITSU,SEQNO,NAIYO,SITSUGRP,SITSUSGRP,IMGNAI,IMGSNAI,FGKEY,FGMUST,KBUNRUI,KLENGTH,KNOTE,SUONELINE,RSCODE,INIVAL,CSKEY,CSWAY,SUCS,VDCDTN,SUPTRITSU,KAISETSU"
			+ ",BIKO,FGJ,DYSET,PGSET,PGPSET,USRSET,DYUPD,PGUPD,PGPUPD,USRUPD"
			+ ") values ("
			+ "'{@CON.USERID}','{@CDANK}','{@CDSITSU}',{@SEQNO},[QANAIYO],'{@SITSUGRP}','{@SITSUSGRP}','{@IMGNAI}','{@IMGSNAI}','0','{@FGMUST}',[KBUNRUI],{@KLENGTH},'{@KNOTE}',{@SUONELINE},'{@RSCODE}','{@INIVAL}','{@CSKEY}','{@CSWAY}',{@SUCS},'{@VDCDTN}',{@SUPTRITSU},'{@KAISETSU}'"
			+ ",'','0','{@CON.YMDH}','{@CON.PGID}','{@CON.PGPID}','{@CON.USERID}','{@CON.YMDH}','{@CON.PGID}','{@CON.PGPID}','{@CON.USERID}'"
			+ ")";
		sql( sql );

		// カラムリソースを登録します。
		boolean rtn = call( "an.ColumnRes", "CDANK,CDSITSU,NAIYO,KBUNRUI,KLENGTH,CDKH", "{@CDANK},{@CDSITSU},[QANAIYO],[KBUNRUI],100,A" );
		if( !rtn ) { return false; }

		qcnt++;
		acnt = 1;

		return true;
	}

	/**
	 * アンケート項目詳細マスタに登録します。
	 * 
	 * @param row
	 *
	 * @return 処理が正常終了したか
	 */
	private boolean insertAN03( final int row ) {
		String seq = StringUtil.valueOf( seq( "AN03S01" ) );
		set( "KVAL", "V" + StringUtil.intFill( seq, 9 ) );
		set( "SEQNO", StringUtil.valueOf( acnt * 10 ) );

		set( "IMGKVAL", null );
		set( "IMGSKVAL", null );
		if( isLine( "FGOTHER" ) ) {
			set( "FGOTHER", "1".equals( line( "FGOTHER" ) ) ? "1" : "0" ); 
		}
		else {
			String knameTmp = line( "QANAIYO" ).trim();
			set( "FGOTHER", "|そのた|その他|OTHER|Other|other|".indexOf( "|" + knameTmp + "|" ) >= 0 ? "1" : "0" );
		}
		set( "RKCODE"	, isLine( "RCODE" )		? line( "RCODE" )		: null );
		set( "FGCORRECT", isLine( "FGCORRECT" )	? line( "FGCORRECT" )	: null );
		set( "KSCALE"	, isLine( "KSCALE" )	? linei( "KSCALE" )		: 5 );
		if( vari( "KSCALE" ) == 0 || vari( "KSCALE" ) == 1 ) { set( "KSCALE", 2 ); }
		if( vari( "KSCALE" ) == -1 ) { set( "KSCALE", -2 ); }

		// 画像ファイルのパスを求め、ファイルをリネームします。
		if( isLine( "QAIMAGE" ) && line( "QAIMAGE" ) != null && line( "QAIMAGE" ).length() > 0 ) {
			String imgOrigFile = fileMap.get( line( "QAIMAGE" ) );
			String imgNew = "1".equals( var( "FGTEMPLATE" ) ) ? "template/" : "";
			imgNew += var( "CDANK" ) + "/" + var( "CDSITSU" ) + "/" + var( "KVAL" ) + "." + Utils.getSuffix( line( "QAIMAGE" ) );

			set( "IMGKVAL", imgNew );
			set( "IMGSKVAL", Utils.getSuffix( imgNew ) );

			new File( imgNewPath + imgNew ).getParentFile().mkdirs();
			FileUtil.copy( imgOrigFile, imgNewPath + imgNew );
		}

		String sql = "insert into AN03("
			+ "USERID,CDANK,CDSITSU,KVAL,SEQNO,KNAME,IMGKVAL,IMGSKVAL,FGOTHER,RKCODE,FGCORRECT,KSCALE"
			+ ",BIKO,FGJ,DYSET,PGSET,PGPSET,USRSET,DYUPD,PGUPD,PGPUPD,USRUPD"
			+ ") values ("
			+ "'{@CON.USERID}','{@CDANK}','{@CDSITSU}','{@KVAL}',{@SEQNO},[QANAIYO],'{@IMGKVAL}','{@IMGSKVAL}','{@FGOTHER}','{@RKCODE}','{@FGCORRECT}',{@KSCALE}"
			+ ",'','0','{@CON.YMDH}','{@CON.PGID}','{@CON.PGPID}','{@CON.USERID}','{@CON.YMDH}','{@CON.PGID}','{@CON.PGPID}','{@CON.USERID}'"
			+ ")";
		sql( sql );

		if( "SCALE".equals( var( "KBUNRUI" ) ) ) {
			// カラムリソースを登録します。
//			boolean rtn = call( "an.ColumnRes", "CDANK,CDSITSU,NAIYO,KBUNRUI,KLENGTH,CDKH", "{@CDANK},{@KVAL},[QANAIYO],SCALE,20,A" );
			boolean rtn = call( "an.ColumnRes", "CDANK,CDSITSU,NAIYO,KBUNRUI,KLENGTH,CDKH", "{@CDANK},{@KVAL},[QANAIYO],SCALE,100,A" );
			if( !rtn ) { return false; }
		}

		acnt++;

		return true;
	}

	/**
	 * メインカーソルの一番最後で呼ばれるロジックを定義します。
	 * 
	 * ここでは、以下の処理を行っています。
	 * ①1時的に展開した画像ファイルの削除
	 * ②展開したアンケートデータの整合性チェック
	 * ③暗号化キーを戻り値として登録
	 * 
	 * @return 処理が正常終了したか
	 */
	@Override
	protected boolean last() {
		if( imgOrigPath != null ) {
			FileUtil.deleteFiles( new File( imgOrigPath ) );
		}

		if( "A".equals( var( "QAKBN" ) ) ) {
			updateSuoneLine();
		}

		rtn( var( "CRYPTKEY" ) );
		return call( "an.AnkCheck","CDANK","{@CDANK}" );
	}

	/**
	 * 回答選択肢の項目数が5以下の場合は、1行表示にする
	 * 
	 * @return 処理が正常終了したか
	 */
	private boolean updateSuoneLine() {
		// 回答選択肢の項目数が5以下の場合は、1行表示にする
		if( !isLine( "SUONELINE" ) && vari( "SUONELINE" ) > 0 && acnt <= 6 ) {
			String upd = "update AN02 set "
				+ "SUONELINE = 5 "
				+ "where CDANK = '{@CDANK}' "
				+ "and CDSITSU = '{@CDSITSU}' ";
			sql( upd );
		}
		return true;
	}

	/**
	 * 指定されたディレクトリを基点としたファイル名(パスを含む)の一覧を返します。
	 * 
	 * @param dir 基点となるディレクトリ
	 * @param map ファイル名一覧を格納するMap
	 */
	private static void getFileMap( final File dir, final Map<String,String> map ) {
		if( map == null ) { return; }
		if( dir.isFile() ) {
			map.put( dir.getName(), dir.getAbsolutePath() );
		}
		else if( dir.isDirectory() ) {
			File[] files = dir.listFiles();
			for( int i=0; i<files.length; i++ ) {
				getFileMap( files[i], map );
			}
		}
	}
}
