/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package an;

import java.io.File;

import org.opengion.fukurou.business.BizLogic_TABLE;
import org.opengion.fukurou.security.HybsCryptography;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ZipFileUtil;
import org.opengion.hayabusa.common.HybsSystem;

/**
 * アンケート情報登録  (AN0001) の 業務ロジックです。
 * 
 * ここでは、アンケート登録画面で指定された項目一覧のファイル(Excel or Calc)及び
 * 画像ファイルのアーカイブ(Zip)からアンケート情報を展開します。
 * 
 * ここで登録されるデータは、FGJ='0'(仮登録)で登録されます。
 * この登録結果からアンケート回答画面のプレビューを表示し、確定する場合は、FGJを'1'にし、
 * 取り消しする場合は、物理削除します。
 * なお、確定処理は、AN0001A2で、削除処理は、AN0001Dで行います。
 *
 * @og.rev 2010/06/18 新規作成
 * 
 * @version 5.0
 * @author Hiroki Nakamura
 * @since JDK1.6,
 */
public class AN0001A extends BizLogic_TABLE {

	private int	qcnt	= 1;	// 質問のSEQ番号
	private int	acnt	= 1;	// 回答のSEQ番号(質問毎にクリア)

	private String imgOrigPath = null; 	// 画像アーカイブの展開先(仮置き)
	private String imgNewPath = null;	// 画像アーカイブの展開先(本置き)

	/**
	 * メインカーソルの一番初めで呼ばれるチェックロジックを定義します。
	 * 
	 * ここでは、画像ファイルを仮置きの場所(filetemp/$USREID/imgtemp)に展開しています。
	 * 
	 * @return 処理が正常終了したか
	 */
	protected boolean fstchk() {
		String imgFileArc = var( "fileArc" );
		if( imgFileArc != null && imgFileArc.length() > 0 ) {
			// ANE0016:画像ファイルアーカイブはzip形式のファイルを指定して下さい。ファイル名={0}
			if( !imgFileArc.endsWith( ".zip" ) ) {
				error( 2, "ANE0016", imgFileArc );
			}

			imgOrigPath = HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "FILE_URL" )
							+ var( "CON.USERID" ) + File.separator + "imgtemp" + File.separator;

			String imgOrigFile = HybsSystem.sys( "REAL_PATH" ) + HybsSystem.sys( "FILE_URL" )
							+ var( "CON.USERID" ) + File.separator + imgFileArc;

			// Zipファイルを展開
			ZipFileUtil.unCompress( imgOrigPath, imgOrigFile );

			imgNewPath = HybsSystem.sys( "REAL_PATH" ) + "ankimages" + File.separator ;
		}

		return true;
	}

	/**
	 * メインカーソルの各行(変更区分の各処理の後)で呼ばれるチェックロジックを定義します。
	 * 
	 * @param row 行番号
	 * @return 処理が正常終了したか
	 */
	@Override
	protected boolean allchk( final int row ) {
//		// ANE0013:QA区分は、「Q」(質問)、「A」(回答選択肢)のいずれかを指定して下さい。値={0}
//		if( !"Q".equals( line( "QAKBN" ) ) && !"A".equals( line( "QAKBN" ) ) ) {
//			error( 2, "ANE0013", line( "QAKBN" ) );
//		}
//
//		// ERR0012:必須チェック
//		if( line( "QANAIYO" ) == null || line( "QANAIYO" ).length() == 0 ) {
//			error( 2, "ERR0012", "{#QANAIYO}" );
//		}

		// ANE0014:回答値分類の指定が不正です。値={0}
		if( "Q".equals( line( "QAKBN" ) ) 
			&& !"TEXT".equals( line( "KBUNRUI" ) ) && !"XTEXT".equals( line( "KBUNRUI" ) )
			&& !"YMD".equals( line( "KBUNRUI" ) ) && !"TEXTAREA".equals( line( "KBUNRUI" ) )
			&& !"RADIO".equals( line( "KBUNRUI" ) ) && !"CHBOX".equals( line( "KBUNRUI" ) ) && !"MENU".equals( line( "KBUNRUI" ) ) ) {
			error( 2, "ANE0014", line( "KBUNRUI" ) );
		}

		if( line( "QAIMAGE" ) != null && line( "QAIMAGE" ).length() > 0 ) {
			// ANE0017:画像が存在しません。ファイル名={0}
			if( !( new File( imgOrigPath + line( "QAIMAGE" ) ) ).exists() ) {
				error( 2, "ANE0017", line( "QAIMAGE" ) );
			}

			// ANE0025:画像以外のファイルが指定されています。ファイル名={0}
			if( !Utils.checkImageName( line( "QAIMAGE" ) ) ) {
				error( 2, "ANE0025", line( "QAIMAGE" ) );
			}
		}

		return true;
	}

	/**
	 * メインカーソルの一番初めで呼ばれるロジックを定義します。
	 * 
	 * @return 処理が正常終了したか
	 */
	@Override
	protected boolean first() {
		sql( "select nextval('AN01S01') PGSEQ" );
		set( "CDANK", "A" + StringUtil.intFill( var( "PGSEQ" ), 9 ) );

		// 暗号化キーは、アンケートコードを基にして暗号化します。
		set( "CRYPTKEY", ( new HybsCryptography() ).encrypt( var( "CDANK" ) ) );

		String sql = "insert into AN01("
			+ "USERID,CDANK,NMANK,DYKSTR,DYKEND,FGKOKAI,CHECKKEY,CRYPTKEY"
			+ ",BIKO,FGJ,DYSET,PGSET,PGPSET,USRSET,DYUPD,PGUPD,PGPUPD,USRUPD"
			+ ") values ("
			+ "'{@CON.USERID}','{@CDANK}','-','00000000','00000000','0','{@CDANK}','{@CRYPTKEY}'"
			+ ",'','0','{@CON.YMDH}','{@CON.PGID}','{@CON.PGPID}','{@CON.USERID}','{@CON.YMDH}','{@CON.PGID}','{@CON.PGPID}','{@CON.USERID}'"
			+ ")";
		sql( sql );

		return true;
	}

	/**
	 * メインカーソルの各行(変更区分の各処理の後)で呼ばれるロジックを定義します。
	 * 
	 * @param row 行番号
	 * @return 処理が正常終了したか
	 */
	@Override
	protected boolean allrow( final int row ) {
		String kbn = line( "QAKBN" );
		if( "Q".equals( kbn ) ) {
			insertAN02( row );
		}
		else {
			insertAN03( row );
		}
		return true;
	}

	/**
	 * アンケート項目マスタに登録します。
	 * 
	 * @param row
	 * @return 処理が正常終了したか
	 */
	private boolean insertAN02( final int row ) {
		sql( "select nextval('AN02S01') PGSEQ" );
		set( "CDSITSU", "S" + StringUtil.intFill( var( "PGSEQ" ), 9 ) );
		
		set( "SEQNO", StringUtil.valueOf( qcnt * 10 ) );
		set( "KLENGTH", StringUtil.valueOf( Utils.getLength( line( "KBUNRUI" ), 200 ) ) );
//		set( "FGKEY", "1".equals( line( "FGKEY" ) ) ? "1" : "0" );
		set( "FGMUST", ( "TEXTAREA".equals( line( "KBUNRUI") ) ? "0" : "1" ) );
		set( "IMGNAI", null );

		// 画像ファイルのパスを求め、ファイルをリネームします。
		if( line( "QAIMAGE" ) != null && line( "QAIMAGE" ).length() > 0 ) {
			String imgOrigFile = imgOrigPath + line( "QAIMAGE" );
			String imgNew = var( "CDANK" ) + "/" + var( "CDSITSU" ) + "." + Utils.getSuffix( line( "QAIMAGE" ) );
			set( "IMGNAI", imgNew );

			new File( imgNewPath + var( "CDANK" ) ).mkdirs();
			FileUtil.copy( imgOrigFile, imgNewPath + imgNew );
		}

		String sql = "insert into AN02("
			+ "USERID,CDANK,CDSITSU,SEQNO,NAIYO,IMGNAI,FGKEY,FGMUST,KBUNRUI,KLENGTH,KNOTE"
			+ ",BIKO,FGJ,DYSET,PGSET,PGPSET,USRSET,DYUPD,PGUPD,PGPUPD,USRUPD"
			+ ") values ("
			+ "'{@CON.USERID}','{@CDANK}','{@CDSITSU}',{@SEQNO},[QANAIYO],'{@IMGNAI}','0','{@FGMUST}',[KBUNRUI],{@KLENGTH},[KNOTE]"
			+ ",'','0','{@CON.YMDH}','{@CON.PGID}','{@CON.PGPID}','{@CON.USERID}','{@CON.YMDH}','{@CON.PGID}','{@CON.PGPID}','{@CON.USERID}'"
			+ ")";
		sql( sql );

		// カラムリソースを登録します。
		boolean rtn = call( "an.ColumnRes", "CDANK,CDSITSU,NAIYO,KBUNRUI,KLENGTH,CDKH", "{@CDANK},{@CDSITSU},[QANAIYO],[KBUNRUI],200,A" );
		if( !rtn ) { return false; }

		qcnt++;
		acnt = 1;

		return true;
	}

	/**
	 * アンケート項目詳細マスタに登録します。
	 * 
	 * @param row
	 * @return 処理が正常終了したか
	 */
	private boolean insertAN03( final int row ) {
		sql( "select nextval('AN03S01') PGSEQ" );
		set( "KVAL", "V" + StringUtil.intFill( var( "PGSEQ" ), 9 ) );

		set( "SEQNO", StringUtil.valueOf( acnt * 10 ) );
//		set( "FGOTHER", "1".equals( line( "FGOTHER" ) ) ? "1" : "0" );
		String knameTmp = line( "QANAIYO" ).trim();
		if( "そのた".equals( knameTmp ) || "その他".equals( knameTmp ) || "OTHER".equalsIgnoreCase( knameTmp ) ) {
			set( "FGOTHER", "1" );
		}
		else {
			set( "FGOTHER", "0" );
		}
		
		set( "IMGKVAL", null );

		// 画像ファイルのパスを求め、ファイルをリネームします。
		if( line( "QAIMAGE" ) != null && line( "QAIMAGE" ).length() > 0 ) {
			String imgOrigFile = imgOrigPath + line( "QAIMAGE" );
			String imgNew = var( "CDANK" ) + "/" + var( "CDSITSU" ) + "/" + var( "KVAL" ) + "." + Utils.getSuffix( line( "QAIMAGE" ) );
			set( "IMGKVAL", imgNew );

			new File( imgNewPath + var( "CDANK" ) + File.separator + var( "CDSITSU" ) ).mkdirs();
			FileUtil.copy( imgOrigFile, imgNewPath + imgNew );
		}

		String sql = "insert into AN03("
			+ "USERID,CDANK,CDSITSU,KVAL,SEQNO,KNAME,IMGKVAL,FGOTHER"
			+ ",BIKO,FGJ,DYSET,PGSET,PGPSET,USRSET,DYUPD,PGUPD,PGPUPD,USRUPD"
			+ ") values ("
			+ "'{@CON.USERID}','{@CDANK}','{@CDSITSU}','{@KVAL}',{@SEQNO},[QANAIYO],'{@IMGKVAL}','{@FGOTHER}'"
			+ ",'','0','{@CON.YMDH}','{@CON.PGID}','{@CON.PGPID}','{@CON.USERID}','{@CON.YMDH}','{@CON.PGID}','{@CON.PGPID}','{@CON.USERID}'"
			+ ")";
		sql( sql );

		acnt++;

		return true;
	}

	/**
	 * メインカーソルの一番最後で呼ばれるロジックを定義します。
	 * 
	 * ここでは、以下の処理を行っています。
	 * ①1時的に展開した画像ファイルの削除
	 * ②展開したアンケートデータの整合性チェック
	 * ③暗号化キーを戻り値として登録
	 * 
	 * @return 処理が正常終了したか
	 */
	@Override
	protected boolean last() {
		if( imgOrigPath != null ) {
			FileUtil.deleteFiles( new File( imgOrigPath ) );
		}

		rtn( var( "CRYPTKEY" ) );
		return call( "an.AnkCheck","CDANK","{@CDANK}");
	}
}
