/*

Copyright (C) NEC Corporation 2012-2013. All Rights Reserved.
Copyright (C) NEC Soft, Ltd. 2012-2013. All Rights Reserved.

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

*/

package com.necsoft.hinemos.webclient.monitor.servlet;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.lang3.StringEscapeUtils;

import com.necsoft.hinemos.webclient.accesscontrol.AccessControlConstants;
import com.necsoft.hinemos.webclient.bean.Property;
import com.necsoft.hinemos.webclient.common.Constants;
import com.necsoft.hinemos.webclient.common.DataProcess;
import com.necsoft.hinemos.webclient.exception.HinemosWebAccessException;
import com.necsoft.hinemos.webclient.exception.HinemosWebException;
import com.necsoft.hinemos.webclient.exception.HinemosWebInvalidRequestParameterException;
import com.necsoft.hinemos.webclient.exception.HinemosWebRemoteException;
import com.necsoft.hinemos.webclient.json.CommonJson;
import com.necsoft.hinemos.webclient.json.GridDataJson;
import com.necsoft.hinemos.webclient.lang.WebMessages;
import com.necsoft.hinemos.webclient.monitor.MonitorConstants;
import com.necsoft.hinemos.webclient.monitor.action.GetEventListTableDefine;
import com.necsoft.hinemos.webclient.monitor.business.ConfirmEventBusiness;
import com.necsoft.hinemos.webclient.monitor.business.EventBatchConfirmBusiness;
import com.necsoft.hinemos.webclient.monitor.business.EventUnconfirmBusiness;
import com.necsoft.hinemos.webclient.monitor.business.GetEventListBusiness;
import com.necsoft.hinemos.webclient.monitor.form.MonitorEventForm;
import com.necsoft.hinemos.webclient.monitor.util.MonitorMappingUtil;
import com.necsoft.hinemos.webclient.servlet.CommonServlet;
import com.necsoft.hinemos.webclient.util.EndpointManager;
import com.necsoft.hinemos.webclient.util.ExceptionHandlerUtils;
import com.necsoft.hinemos.webclient.util.LogUtils;
import com.necsoft.hinemos.webclient.util.LoginManager;
import com.necsoft.hinemos.webclient.util.PopulateBeanUtils;

/**
 * イベント情報データをリフレッシュするクラス
 *
 * @version 2.0.0
 * @since 1.0.0
 **/
public class GetEventListServlet extends CommonServlet {
    private static final long serialVersionUID = 996012390849870607L;

    private static LogUtils logUtils = LogUtils
            .getLogUtils(GetEventListServlet.class);

    @Override
    public CommonJson process(HttpServletRequest request,
            HttpServletResponse response) throws ServletException, IOException,
            HinemosWebAccessException {

        CommonJson result = new CommonJson();

        try {
            // Get data form
            MonitorEventForm form = getDataForm(request);

            // Action form
            String unavailableManagerIp = doAction(form, request);

            // 中間変数を宣言
            List<ArrayList<Object>> listInfoData = doLoadListInfoData(
                    request, form);

            // Create JSON
            result = doConvertToJson(listInfoData);

            // 確認／未確認に変更／一括確認の実行結果が失敗の場合
            if (unavailableManagerIp != null){
                HttpSession session = request.getSession(true);
                @SuppressWarnings("unchecked")
                LinkedHashMap<String, LoginManager> loginManagerMap = (LinkedHashMap<String, LoginManager>) session.getAttribute(AccessControlConstants.LOGIN_MANAGER);
                Set<String> managerIpSet = loginManagerMap.keySet();
                if (managerIpSet.size() > 1) {
                    result.setMessage(WebMessages.getString("exception.data.operation", request.getLocale()) + "(" + unavailableManagerIp + ")");
                } else {
                    result.setMessage(WebMessages.getString("exception.data.operation", request.getLocale()));
                }
                result.setResult(1);
            }

        } catch (HinemosWebAccessException ex) {
            logUtils.info(ex.getMessage());
            throw new HinemosWebAccessException();
        } catch (HinemosWebRemoteException ex) {
            logUtils.error(ex.getMessage(), ex.getCause());
            result = ExceptionHandlerUtils.doBuildCommonJson(
                    ExceptionHandlerUtils.LEVEL_WARNING, ex.getMessage(), request.getLocale());

        } catch (Exception ex) {
            logUtils.error(ex.getMessage(), ex.getCause());
            result = ExceptionHandlerUtils.doBuildCommonJson(
                    ExceptionHandlerUtils.LEVEL_ERROR, ex.getMessage(), request.getLocale());
        }
        return result;
    }

    /**
     * @param request
     * @return
     */
    private MonitorEventForm getDataForm(HttpServletRequest request) {
        MonitorEventForm form = new MonitorEventForm();
        PopulateBeanUtils.populateBean(form, request);
        return form;
    }

    /**
     * 確認／未確認に変更／一括確認の処理を行う
     * @param form
     * @param request
     * @throws HinemosWebException
     */
    @SuppressWarnings("unchecked")
    private String doAction(MonitorEventForm form, HttpServletRequest request
            ) throws HinemosWebException {
        String action;
        // Check action
        if (form.getAction() != null) {
            action = form.getAction();
            // get loginManager session
            HttpSession session = request.getSession(true);

            if (action.equals("confirmed") || action.equals("unconfirm")) {
                synchronized(session) {
                    // 操作権限を持っていないマネージャのIP
                    String unavailableManagerIp = null;


                    LinkedHashMap<String, LoginManager> loginManagerMap = (LinkedHashMap<String, LoginManager>) session.getAttribute(AccessControlConstants.LOGIN_MANAGER);
                    Set<String> managerIpSet = loginManagerMap.keySet();
                    int columnCount = managerIpSet.size() <= 1 ? MonitorConstants.NUMBER_HEADER_EVENT_WITHOUT_MANAGER : MonitorConstants.NUMBER_HEADER_EVENT_WITH_MANAGER;
                    ArrayList<ArrayList<Object>> list = doGetConfirmedData(request, columnCount);
                    if (list.size() > 0) {
                        if (managerIpSet.size() <= 1) {
                            for (String managerIp : managerIpSet) {
                                LoginManager loginManager = loginManagerMap.get(managerIp);
                                if (action.equals("confirmed")) {
                                    if (!doConfirmed(list, loginManager.endpointManager) && unavailableManagerIp == null) {
                                        unavailableManagerIp = managerIp;
                                    }
                                } else {
                                    if (!doUnconfirm(list, loginManager.endpointManager) && unavailableManagerIp == null) {
                                        unavailableManagerIp = managerIp;
                                    }
                                }
                            }
                        } else {
                            Map<String, ArrayList<ArrayList<Object>>> confirmMap = new HashMap<String, ArrayList<ArrayList<Object>>>();
                            for (int i = 0; i < list.size(); i++) {
                                ArrayList<Object> row = list.get(i);

                                String managerIp = (String)row.get(GetEventListTableDefine.MANAGER);
                                if (confirmMap.get(managerIp) == null) {
                                    ArrayList<ArrayList<Object>> dataList = new ArrayList<ArrayList<Object>>();
                                    dataList.add(row);
                                    confirmMap.put(managerIp, dataList);
                                } else {
                                    ArrayList<ArrayList<Object>> dataList = confirmMap.get(managerIp);
                                    dataList.add(row);
                                    confirmMap.put(managerIp, dataList);
                                }
                            }

                            Set<String> managerIpSetInconfirmMap = confirmMap.keySet();
                            for (String managerIp : managerIpSetInconfirmMap) {
                                LoginManager loginManager = loginManagerMap.get(managerIp);
                                if (action.equals("confirmed")) {
                                    if (!doConfirmed(confirmMap.get(managerIp), loginManager.endpointManager) && unavailableManagerIp == null) {
                                        unavailableManagerIp = managerIp;
                                    }
                                } else {
                                    if (!doUnconfirm(confirmMap.get(managerIp), loginManager.endpointManager) && unavailableManagerIp == null) {
                                        unavailableManagerIp = managerIp;
                                    }
                                }
                            }

                        }
                        return unavailableManagerIp;
                    }

                }
            } else if (action.equals("batchconfirm")) {

                Property condition = doLoadConditionBatchConfirm(request);
                synchronized(session) {
                    LinkedHashMap<String, LoginManager> loginManagerMap = (LinkedHashMap<String, LoginManager>) session.getAttribute(AccessControlConstants.LOGIN_MANAGER);
                    EventBatchConfirmBusiness eventBusiness = new EventBatchConfirmBusiness();

                    return eventBusiness.batchconfirm(form, condition, loginManagerMap, request.getLocale());
                }
            }
        }
        return null;
    }

    /**
     * @param list
     * @throws HinemosWebException
     */
    private boolean doConfirmed(ArrayList<ArrayList<Object>> list,
        EndpointManager endpointManager) throws HinemosWebException {
        ConfirmEventBusiness eventBusiness = new ConfirmEventBusiness();
        return eventBusiness.confirm(list, endpointManager);
    }

    /**
     *
     * @param list
     * @param endpointManager
     * @throws HinemosWebException
     */
    private boolean doUnconfirm(ArrayList<ArrayList<Object>> list,
        EndpointManager endpointManager) throws HinemosWebException {
        EventUnconfirmBusiness eventBusiness = new EventUnconfirmBusiness();
        return eventBusiness.unconfirm(list, endpointManager);
    }

    private List<ArrayList<Object>> doLoadListInfoData(
            HttpServletRequest request, MonitorEventForm form) throws HinemosWebException {
        List<ArrayList<Object>> listLogData = new ArrayList<ArrayList<Object>>();

        //get loginManager session
        HttpSession session = request.getSession(true);
        ServletContext application = request.getSession().getServletContext();
        int maxNum = (Integer)application.getAttribute(Constants.MONITOR_EVENT_MAX_ROW);
        synchronized(session) {
            @SuppressWarnings("unchecked")
            LinkedHashMap<String, LoginManager> loginManagerMap = (LinkedHashMap<String, LoginManager>) session.getAttribute(AccessControlConstants.LOGIN_MANAGER);
            form.setMaxRowNum(maxNum);
                // Check filter
                if (form.isFilter()) {
                    Property condition = doLoadCondition(request);
                    GetEventListBusiness business = new GetEventListBusiness();
                    listLogData = business.getEventList(form, condition,loginManagerMap, request.getLocale());
                } else {
                    GetEventListBusiness business = new GetEventListBusiness();
                    listLogData = business.getEventList(form, null,
                            loginManagerMap, request.getLocale());
                }
        }

        return listLogData;
    }

    /**
     * @param request
     * @return
     * @throws HinemosWebException
     */
    private Property doLoadCondition(HttpServletRequest request)
            throws HinemosWebException {
        HttpSession session = request.getSession(true);

        if (session.getAttribute(MonitorConstants.EVENT_FILTER_PROPERTIES) == null) {
            // HTTPセッションのタイムアウトが発生しました。ログアウトします。
            throw new HinemosWebException("message.sessiontimeout");
        }

        Property condition = (Property) session
                .getAttribute(MonitorConstants.EVENT_FILTER_PROPERTIES);
        convertCondition(request, condition);
        return condition;
    }

    private Property doLoadConditionBatchConfirm(HttpServletRequest request)
        throws HinemosWebException {
        HttpSession session = request.getSession(true);

        if (session.getAttribute(MonitorConstants.EVENT_BATCH_CONFIRM_PROPERTIES) == null) {
            // HTTPセッションのタイムアウトが発生しました。ログアウトします。
            throw new HinemosWebException("message.sessiontimeout");
        }

        Property condition = (Property) session
                .getAttribute(MonitorConstants.EVENT_BATCH_CONFIRM_PROPERTIES);
        convertCondition(request, condition);
        return condition;
    }

    private GridDataJson doConvertToJson(List<ArrayList<Object>> listInfoData)
            throws HinemosWebException {
        return DataProcess.doConvertTableContentToJson(listInfoData);
    }

    /**
     * @param request
     * @return
     * @throws HinemosWebException
     */
    private ArrayList<ArrayList<Object>> doGetConfirmedData(
            HttpServletRequest request, int columnCount) throws HinemosWebException {
        ArrayList<ArrayList<Object>> list = new ArrayList<ArrayList<Object>>();
        String actionData = Constants.ACTION_DATA_REQUEST;
        int numData;

        if (request.getParameter("numData") == null) {
            // イベントの確認に失敗しました
            throw new HinemosWebInvalidRequestParameterException(
                    "exception.data.confirm");
        } else {
            numData = Integer.valueOf(request.getParameter("numData"));
            if (numData <= 0) {
                // イベントの確認に失敗しました
                throw new HinemosWebInvalidRequestParameterException(
                        "exception.data.confirm");
            } else if (numData > (Integer)request.getSession().getServletContext().getAttribute(Constants.MONITOR_EVENT_MAX_ROW)) {
                numData = (Integer)request.getSession().getServletContext().getAttribute(Constants.MONITOR_EVENT_MAX_ROW);
            }
        }

        for (int count = 0; count < numData; count++) {
            ArrayList<Object> row = new ArrayList<Object>();
            String[] row_data = new String[columnCount];
            for (int cnt_col = 0; cnt_col < columnCount; cnt_col++) {
                String row_name = getNameRequestParam(actionData, count,
                        MonitorMappingUtil.getEventName(cnt_col));
                if (request.getParameter(row_name) == null) {
                    // イベントの確認に失敗しました
                    throw new HinemosWebInvalidRequestParameterException(
                            "exception.data.confirm");
                } else {
                    row_data[cnt_col] = request.getParameter(row_name);
                }
            }
            ;

            long receiveTime;
            long outputDate;
            try {
                receiveTime = Long
                        .parseLong(row_data[GetEventListTableDefine.RECEIVE_TIME]);
                outputDate = Long
                        .parseLong(row_data[GetEventListTableDefine.OUTPUT_DATE]);
            } catch (Exception e) {
                // イベントの確認に失敗しました
                throw new HinemosWebInvalidRequestParameterException(
                        "exception.data.confirm");
            }
            ;

            row.add(GetEventListTableDefine.PRIORITY,
                row_data[GetEventListTableDefine.PRIORITY]);
            row.add(GetEventListTableDefine.RECEIVE_TIME,
                new Date(receiveTime));
            row.add(GetEventListTableDefine.OUTPUT_DATE,
                new Date(outputDate));
            row.add(GetEventListTableDefine.PLUGIN_ID, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.PLUGIN_ID]));
            row.add(GetEventListTableDefine.MONITOR_ID, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.MONITOR_ID]));
            row.add(GetEventListTableDefine.MONITOR_DETAIL_ID, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.MONITOR_DETAIL_ID]));
            row.add(GetEventListTableDefine.FACILITY_ID, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.FACILITY_ID]));
            row.add(GetEventListTableDefine.SCOPE, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.SCOPE]));
            row.add(GetEventListTableDefine.APPLICATION, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.APPLICATION]));
            row.add(GetEventListTableDefine.MESSAGE_ID, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.MESSAGE_ID]));
            row.add(GetEventListTableDefine.MESSAGE, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.MESSAGE]));
            row.add(GetEventListTableDefine.CONFIRMED,
                row_data[GetEventListTableDefine.CONFIRMED]);
            row.add(GetEventListTableDefine.CONFIRM_USER,
                row_data[GetEventListTableDefine.CONFIRM_USER]);
            row.add(GetEventListTableDefine.COMMENT, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.COMMENT]));
            if (MonitorConstants.NUMBER_HEADER_EVENT_WITH_MANAGER == columnCount) {
                row.add(GetEventListTableDefine.MANAGER, StringEscapeUtils
                        .unescapeHtml4(row_data[GetEventListTableDefine.MANAGER]));
            }
            list.add(row);
        }

        return list;
    }

    /**
     * @param prefixName
     * @param count
     * @return
     */
    private String getNameRequestParam(String prefixName, int count,
            String colName) {
        StringBuffer names = new StringBuffer();
        names.append(prefixName);
        names.append("[");
        names.append(count);
        names.append("][");
        names.append(colName);
        names.append("]");

        return names.toString();
    }

}