/*
 
Copyright (C) NEC Corporation 2013. All Rights Reserved. 
Copyright (C) NEC Soft, Ltd. 2013. All Rights Reserved. 
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.necsoft.hinemos.webclient.jobmanagement.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;

import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.log4j.Logger;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.jobmanagement.bean.JobOperationConstant;
import com.clustercontrol.ws.jobmanagement.InvalidRole_Exception;
import com.necsoft.hinemos.webclient.bean.Property;
import com.necsoft.hinemos.webclient.exception.HinemosWebAccessException;
import com.necsoft.hinemos.webclient.exception.HinemosWebException;
import com.necsoft.hinemos.webclient.jobmanagement.JobManagementConstants;
import com.necsoft.hinemos.webclient.jobmanagement.bean.OperationConstant;
import com.necsoft.hinemos.webclient.jobmanagement.form.OperationForm;
import com.necsoft.hinemos.webclient.lang.WebMessages;
import com.necsoft.hinemos.webclient.servlet.CommonServlet;
import com.necsoft.hinemos.webclient.util.EndpointManager;

/**
 * ジョブオペレーション用プロパティを作成するクラス
 */
public class OperationJobProperty implements JobManagementConstants {

    private static Logger m_log = Logger.getLogger(CommonServlet.class);

    /**
     * ジョブオペレーション用プロパティを取得する
     * 
     * @return ジョブオペレーション用プロパティ
     * 
     */
    public static Property getProperty(OperationForm form, Locale locale,
            EndpointManager endpointManager) throws HinemosWebException {

        if (locale == null || (!Locale.ENGLISH.getLanguage().equals(locale.getLanguage()))) {
            locale = Locale.JAPAN;
        }

        String m_control = form.getControl();
        String m_session = StringEscapeUtils.unescapeHtml4(form.getSessionId());
        String m_jobUnit = StringEscapeUtils.unescapeHtml4(form.getJobunitId());
        String m_job = StringEscapeUtils.unescapeHtml4(form.getJobId());
        String m_facility = StringEscapeUtils.unescapeHtml4(form.getFacilityId());

        Property property;
        
        if (OperationConstant.JOB_CONTROL_START.equals(m_control)) {
            property = getStartProperty(
                    m_session, m_jobUnit, m_job, m_facility,
                    locale, endpointManager);
        } else if (OperationConstant.JOB_CONTROL_STOP.equals(m_control)) {
            property = getStopProperty(
                    m_session, m_jobUnit, m_job, m_facility,
                    locale, endpointManager);
        } else {
            throw new HinemosWebException("exception.data.param");
        }
                
        return property;
    }

    /**
     * 開始用プロパティ取得
     * 
     * @param sessionId
     *            セッションID
     * @param jobunitId
     *            所属ジョブユニットのジョブID
     * @param jobId
     *            ジョブID
     * @param facilityId
     *            ファシリティID
     * @return ジョブ開始操作用プロパティ
     * 
     */
    private static Property getStartProperty(
            String sessionId, String jobunitId, String jobId, String facilityId,
            Locale locale, EndpointManager endpointManager) throws HinemosWebException {

        if (locale == null || (!Locale.ENGLISH.getLanguage().equals(locale.getLanguage()))) {
            locale = Locale.JAPAN;
        }
        // セッションID
        Property session = new Property(JobOperationConstant.SESSION,
                WebMessages.getString("session.id", locale),
                PropertyConstant.EDITOR_TEXT);
        // ジョブユニットID
        Property jobUnit = new Property(JobOperationConstant.JOB_UNIT,
                WebMessages.getString("jobunit.id", locale),
                PropertyConstant.EDITOR_TEXT);
        // ジョブID
        Property job = new Property(JobOperationConstant.JOB,
                WebMessages.getString("job.id", locale),
                PropertyConstant.EDITOR_TEXT);
        // ファシリティID
        Property facility = new Property(JobOperationConstant.FACILITY,
                WebMessages.getString("facility.id", locale),
                PropertyConstant.EDITOR_TEXT);
        // 制御
        Property control = new Property(JobOperationConstant.CONTROL,
                WebMessages.getString("control", locale),
                PropertyConstant.EDITOR_SELECT);

        List<String> values = null;
        try {
            values = JobEndpointWrapper.getAvailableStartOperation(sessionId,
                    jobunitId, jobId, facilityId, endpointManager);
        } catch (InvalidRole_Exception e) {
            throw new HinemosWebAccessException("message.accesscontrol.16");
        } catch (Exception e) {
            m_log.warn("getStartProperty(), " + e.getMessage(), e);
            throw new HinemosWebException("message.hinemos.failure.unexpected",
                    e.getCause());
        }

        // 値を初期化
        List<String> values1 = new ArrayList<String>();
        if (values.size() > 0) {
            for (int ind = 0; ind < values.size(); ind++) {
                String value = values.get(ind);
                int type = OperationConstant.stringToType(value);
                values1.add(OperationConstant.typeToString(type, locale));
            }
        } else {
            values1.add("");
        }
        Object controlValues[][] = { values1.toArray(), values1.toArray() };

        control.setSelectValues(controlValues);
        if (values.size() >= 1) {
            control.setValue(values.get(0));
        } else {
            control.setValue("");
        }

        session.setValue(StringEscapeUtils.escapeHtml4(sessionId).replaceAll(SINGLE_QUOTE, SINGLE_QUOTE_HTML));
        jobUnit.setValue(StringEscapeUtils.escapeHtml4(jobunitId).replaceAll(SINGLE_QUOTE, SINGLE_QUOTE_HTML));
        job.setValue(StringEscapeUtils.escapeHtml4(jobId).replaceAll(SINGLE_QUOTE, SINGLE_QUOTE_HTML));
        if (facilityId != null && facilityId.length() > 0) {
            facility.setValue(StringEscapeUtils.escapeHtml4(facilityId).replaceAll(SINGLE_QUOTE, SINGLE_QUOTE_HTML));
        } else {
            facility.setValue("");
        }

        // 変更の可/不可を設定
        session.setModify(PropertyConstant.MODIFY_NG);
        jobUnit.setModify(PropertyConstant.MODIFY_NG);
        job.setModify(PropertyConstant.MODIFY_NG);
        facility.setModify(PropertyConstant.MODIFY_NG);
        control.setModify(PropertyConstant.MODIFY_OK);

        Property property = new Property(null, null, "");

        // 初期表示ツリーを構成。
        property.removeChildren();
        property.addChildren(session);
        property.addChildren(jobUnit);
        property.addChildren(job);
        if (facilityId != null && facilityId.length() > 0) {
            property.addChildren(facility);
        }
        property.addChildren(control);

        return property;
    }

    /**
     * 停止用プロパティ取得
     * 
     * @param sessionId セッションID
     * @param jobunitId 所属ジョブユニットのジョブID
     * @param jobId ジョブID
     * @param facilityId ファシリティID
     * @return ジョブ停止操作用プロパティ
     * 
     */
    private static Property getStopProperty(
            String sessionId, String jobunitId, String jobId, String facilityId,
            Locale locale, EndpointManager endpointManager) throws HinemosWebException {

        if (locale == null || (!Locale.ENGLISH.getLanguage().equals(locale.getLanguage()))) {
            locale = Locale.JAPAN;
        }

        //セッションID
        Property session =
            new Property(JobOperationConstant.SESSION, WebMessages.getString("session.id", locale), PropertyConstant.EDITOR_TEXT);
        //ジョブユニットID
        Property jobUnit =
            new Property(JobOperationConstant.JOB_UNIT, WebMessages.getString("jobunit.id", locale), PropertyConstant.EDITOR_TEXT);
        //ジョブID
        Property job =
            new Property(JobOperationConstant.JOB, WebMessages.getString("job.id", locale), PropertyConstant.EDITOR_TEXT);
        //ファシリティID
        Property facility =
            new Property(JobOperationConstant.FACILITY, WebMessages.getString("facility.id", locale), PropertyConstant.EDITOR_TEXT);
        //制御
        Property control =
            new Property(JobOperationConstant.CONTROL, WebMessages.getString("control", locale), PropertyConstant.EDITOR_SELECT);
        //終了値
        Property endValue =
            new Property(JobOperationConstant.END_VALUE, WebMessages.getString("end.value", locale), PropertyConstant.EDITOR_NUM,
                    DataRangeConstant.SMALLINT_HIGH, DataRangeConstant.SMALLINT_LOW);

        ArrayList<Property> endList = new ArrayList<Property>();
        endList.add(endValue);

        HashMap<String, Object> skipEndMap = new HashMap<String, Object>();
        String stringStopSkip = OperationConstant.typeToString(
                OperationConstant.TYPE_STOP_SKIP, locale);
        skipEndMap.put(PropertyConstant.MAP_VALUE, stringStopSkip);
        skipEndMap.put(PropertyConstant.MAP_PROPERTY, endList);

        HashMap<String, Object> mainteEndMap = new HashMap<String, Object>();
        String stringStopMaintenance = OperationConstant.typeToString(
                OperationConstant.TYPE_STOP_MAINTENANCE, locale);
        mainteEndMap.put(PropertyConstant.MAP_VALUE, stringStopMaintenance);
        mainteEndMap.put(PropertyConstant.MAP_PROPERTY, endList);

        HashMap<String, Object> forceEndMap = new HashMap<String, Object>();
        String stringStopForce = OperationConstant.typeToString(
                OperationConstant.TYPE_STOP_FORCE, locale);
        forceEndMap.put(PropertyConstant.MAP_VALUE, stringStopForce);
        forceEndMap.put(PropertyConstant.MAP_PROPERTY, endList);

        List<String> values = null;
        try {
            values = JobEndpointWrapper.getAvailableStopOperation(
                    sessionId, jobunitId, jobId, facilityId,
                    endpointManager);
        } catch (InvalidRole_Exception e) {
            throw new HinemosWebAccessException("message.accesscontrol.16");
        } catch (Exception e) {
            m_log.warn("getStartProperty(), " + e.getMessage(), e);
            throw new HinemosWebException(
                    "message.hinemos.failure.unexpected", e.getCause());
        }

        ArrayList<Object> values1 = new ArrayList<Object>();
        ArrayList<Object> values2 = new ArrayList<Object>();
        boolean hasEndValue = false;
        for (int ind = 0; ind < values.size(); ind++) {
            String value = values.get(ind);
            int type = OperationConstant.stringToType(value);
            values1.add(OperationConstant.typeToString(type, locale));
            switch (type) {
                case OperationConstant.TYPE_STOP_SKIP:
                    // Set map entry
                    values2.add(skipEndMap);
                    hasEndValue = true;
                    break;
                case OperationConstant.TYPE_STOP_MAINTENANCE:
                    // Set map entry
                    values2.add(mainteEndMap);
                    hasEndValue = true;
                    break;
                case OperationConstant.TYPE_STOP_FORCE:
                    // Set map entry
                    values2.add(forceEndMap);
                    hasEndValue = true;
                    break;
                default:
                    // Set normal entry
                    values2.add(OperationConstant.typeToString(type, locale));
                    break;
            }
        }
        if (values.size() == 0) {
            values1.add("");
            values2.add("");
        }

        //値を初期化
        Object controlValues[][] = {values1.toArray(), values2.toArray()};

        control.setSelectValues(controlValues);
        if(values.size() >= 1){
            control.setValue(values2.get(0));
        }
        else{
            control.setValue("");
        }

        endValue.setValue("");

        session.setValue(StringEscapeUtils.escapeHtml4(sessionId).replaceAll(SINGLE_QUOTE, SINGLE_QUOTE_HTML));
        jobUnit.setValue(StringEscapeUtils.escapeHtml4(jobunitId).replaceAll(SINGLE_QUOTE, SINGLE_QUOTE_HTML));
        job.setValue(StringEscapeUtils.escapeHtml4(jobId).replaceAll(SINGLE_QUOTE, SINGLE_QUOTE_HTML));
        if(facilityId != null && facilityId.length() > 0){
            facility.setValue(StringEscapeUtils.escapeHtml4(facilityId).replaceAll(SINGLE_QUOTE, SINGLE_QUOTE_HTML));
        }
        else{
            facility.setValue("");
        }

        //変更の可/不可を設定
        session.setModify(PropertyConstant.MODIFY_NG);
        jobUnit.setModify(PropertyConstant.MODIFY_NG);
        job.setModify(PropertyConstant.MODIFY_NG);
        facility.setModify(PropertyConstant.MODIFY_NG);
        control.setModify(PropertyConstant.MODIFY_OK);
        endValue.setModify(PropertyConstant.MODIFY_OK);

        Property property = new Property(null, null, "");

        // 初期表示ツリーを構成。
        property.removeChildren();
        property.addChildren(session);
        property.addChildren(jobUnit);
        property.addChildren(job);
        if(facilityId != null && facilityId.length() > 0){
            property.addChildren(facility);
        }
        property.addChildren(control);

        if(hasEndValue){
            control.addChildren(endValue);
        }

        return property;
    }
}