/*

Copyright (C) NEC Corporation 2012. All Rights Reserved.
Copyright (C) NEC Soft, Ltd. 2012. All Rights Reserved.

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

*/

package com.necsoft.hinemos.webclient.jobmanagement.servlet;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import com.necsoft.hinemos.webclient.accesscontrol.AccessControlConstants;
import com.necsoft.hinemos.webclient.common.DataProcess;
import com.necsoft.hinemos.webclient.exception.HinemosWebAccessException;
import com.necsoft.hinemos.webclient.exception.HinemosWebException;
import com.necsoft.hinemos.webclient.exception.HinemosWebInvalidRequestParameterException;
import com.necsoft.hinemos.webclient.exception.HinemosWebRemoteException;
import com.necsoft.hinemos.webclient.jobmanagement.business.GetNodeDetailBusiness;
import com.necsoft.hinemos.webclient.jobmanagement.form.NodeDetailForm;
import com.necsoft.hinemos.webclient.json.CommonJson;
import com.necsoft.hinemos.webclient.servlet.CommonServlet;
import com.necsoft.hinemos.webclient.util.ExceptionHandlerUtils;
import com.necsoft.hinemos.webclient.util.LogUtils;
import com.necsoft.hinemos.webclient.util.LoginManager;
import com.necsoft.hinemos.webclient.util.PopulateBeanUtils;

/**
 * This class used for get Node Detail info
 *
 * @version 1.0.0
 * @since 1.0.0
 **/
public class GetNodeDetailServlet extends CommonServlet {
    private static final long serialVersionUID = -7775038566425028930L;

    private static LogUtils logUtils = LogUtils
            .getLogUtils(GetNodeDetailServlet.class);

    /**
     * Get Node Detail info
     *
     * @param request
     *            : HttpServletRequest
     * @param response
     *            : HttpServletResponse
     * @throws ServletException
     * @throws IOException
     * @throws HinemosWebAccessException
     */
    @Override
    public CommonJson process(HttpServletRequest request,
            HttpServletResponse response) throws ServletException, IOException,
            HinemosWebAccessException {

        CommonJson result = new CommonJson();

        try {
            // Get table tree
            ArrayList<ArrayList<Object>> listInfoData = doLoadListInfoData(request);

            // convert to json
            result = DataProcess.doConvertTableContentToJson(listInfoData);

        } catch (HinemosWebAccessException ex) {
            logUtils.info(ex.getMessage());
            throw new HinemosWebAccessException();
        } catch (HinemosWebRemoteException ex) {
            logUtils.error(ex.getMessage(), ex.getCause());
            result = ExceptionHandlerUtils.doBuildCommonJson(
                    ExceptionHandlerUtils.LEVEL_WARNING, ex.getMessage(), request.getLocale());

        } catch (Exception ex) {
            logUtils.error(ex.getMessage(), ex.getCause());
            result = ExceptionHandlerUtils.doBuildCommonJson(
                    ExceptionHandlerUtils.LEVEL_ERROR, ex.getMessage(), request.getLocale());
        }
        return result;
    }

    /**
     * Get Node Detail info
     *
     * @param request
     *            : HttpServletRequest
     * @return ArrayList<ArrayList<Object>>
     * @throws HinemosWebException
     */
    private ArrayList<ArrayList<Object>> doLoadListInfoData(HttpServletRequest request)
            throws HinemosWebException {

        ArrayList<ArrayList<Object>> result = null;
        NodeDetailForm form = new NodeDetailForm();
        PopulateBeanUtils.populateBean(form, request);

        // Check parameters of request
        if (!isValidRequestParam(form)) {
            throw new HinemosWebInvalidRequestParameterException();
        }

        // get loginManager session
        HttpSession session = request.getSession(true);
        synchronized (session) {
            @SuppressWarnings("unchecked")
            LinkedHashMap<String, LoginManager> loginManagerMap = (LinkedHashMap<String, LoginManager>) session.getAttribute(AccessControlConstants.LOGIN_MANAGER);

            GetNodeDetailBusiness business = new GetNodeDetailBusiness();
            String managerIp = form.getManagerIpForSearch();
            // シングルログインの場合
            if (managerIp == null || managerIp.trim().isEmpty()) {
                Iterator<LoginManager> iterator = loginManagerMap.values().iterator();
                while (iterator.hasNext()) {
                    LoginManager loginManager = iterator.next();
                    result = business.getNodeDetail(form, loginManager.endpointManager);
                    break;
                }
            } else {
                result = business.getNodeDetail(form, loginManagerMap.get(managerIp).endpointManager);
            }
        }

        return result;
    }

    /**
     * Check valid parameter
     *
     * @param form
     *            : NodeDetailForm
     * @return boolean
     */
    private boolean isValidRequestParam(NodeDetailForm form) {
        boolean isValid = true;
        String sessionId = form.getSessionId();
        String jobUnitId = form.getJobunitId();
        String jobId = form.getJobId();

        // Check request sessionId (manager)
        if (sessionId == null || sessionId.equals("")) {
            isValid = false;
            logUtils.error("sessionId is invalid", sessionId);
        }

        // Check request jobUnitId (manager)
        if (jobUnitId == null || jobUnitId.equals("")) {
            isValid = false;
            logUtils.error("jobUnitId is invalid", jobUnitId);
        }

        // Check request jobId (manager)
        if (jobId == null || jobId.equals("")) {
            isValid = false;
            logUtils.error("jobId is invalid", jobId);
        }

        return isValid;
    }
}