/*

Copyright (C) NEC Corporation 2012. All Rights Reserved.
Copyright (C) NEC Soft, Ltd. 2012. All Rights Reserved.

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.necsoft.hinemos.webclient.accesscontrol.servlet;

import java.io.IOException;
import java.net.URLDecoder;
import java.rmi.AccessException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import javax.naming.CommunicationException;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import com.necsoft.hinemos.webclient.accesscontrol.AccessControlConstants;
import com.necsoft.hinemos.webclient.accesscontrol.bean.UserBean;
import com.necsoft.hinemos.webclient.accesscontrol.business.LoginAuthBusiness;
import com.necsoft.hinemos.webclient.accesscontrol.form.UserForm;
import com.necsoft.hinemos.webclient.accesscontrol.util.LoginSessionManager;
import com.necsoft.hinemos.webclient.accesscontrol.validation.LoginValidator;
import com.necsoft.hinemos.webclient.json.CommonJson;
import com.necsoft.hinemos.webclient.lang.WebMessages;
import com.necsoft.hinemos.webclient.servlet.CommonServlet;
import com.necsoft.hinemos.webclient.util.LogUtils;
import com.necsoft.hinemos.webclient.util.LoginManager;

/**
 * Hinemosマネージャに接続するクラス
 *
 * @version 1.0.0
 * @since 1.0.0
 **/
public class LoginAuthServlet extends CommonServlet {
    /**
	 *
	 */
    private static final long serialVersionUID = -8809478864238709832L;

    private static LogUtils logUtils = LogUtils
            .getLogUtils(LoginAuthServlet.class);

    /**
     * Handles the HTTP <code>GET</code> method.
     *
     * @param request
     *            servlet request
     * @param response
     *            servlet response
     */
    @Override
    public CommonJson process(HttpServletRequest request,
            HttpServletResponse response) throws ServletException, IOException {

        logout(request);
        HttpSession session = request.getSession(true);

        // ユーザID（複数、カンマ「,」で区切る）
        String userName = request.getParameter("userName");
        // パスワード（複数、カンマ「,」で区切る）
        String password = request.getParameter("password");
        // 接続先URL（複数、カンマ「,」で区切る）
        String urlServer = request.getParameter("urlServer");
        // 同じユーザＩＤ・パスワードを使用するフラグ
        String useTheSameIDAndPwd = request.getParameter("useTheSameIDAndPwd");

        String[] userNames = userName.split(AccessControlConstants.SPLIT_FOR_MULTI_LOGIN);
        String[] passwords = password.split(AccessControlConstants.SPLIT_FOR_MULTI_LOGIN);
        String[] urlServers = urlServer.split(AccessControlConstants.SPLIT_FOR_MULTI_LOGIN);

        List<UserBean> userBeanList = new ArrayList<UserBean>();
        LinkedHashMap<String, LoginManager> loginManagerMap = new LinkedHashMap<String, LoginManager>();
        List<UserForm> userFormList = new ArrayList<UserForm>();
        for (int i = 0; i < urlServers.length; i++) {
            UserForm form = new UserForm();

            // 同じユーザＩＤ・パスワードを使用するかどうかの判定。
            if (useTheSameIDAndPwd != null && Boolean.parseBoolean(useTheSameIDAndPwd)) {
                form.setUserName(URLDecoder.decode(userNames[0], request.getCharacterEncoding()));
                form.setPassword(URLDecoder.decode(passwords[0], request.getCharacterEncoding()));
            } else {
                form.setUserName(URLDecoder.decode(userNames[i], request.getCharacterEncoding()));
                form.setPassword(URLDecoder.decode(passwords[i], request.getCharacterEncoding()));
            }
            form.setUrlServer(URLDecoder.decode(urlServers[i], request.getCharacterEncoding()));
            userFormList.add(form);
        }
        try {
            session.setAttribute(AccessControlConstants.THE_SAME_ID_AND_PWD_FLG, useTheSameIDAndPwd);
            for (int i = 0; i < urlServers.length; i++) {
                UserForm form = userFormList.get(i);

                UserBean userBean = new UserBean();
                if (!isValidRequestParam(form)) {
                    userBeanList.add(userBean);
                    setUserBeanError(userFormList, userBeanList, "message.accesscontrol.6",
                            request.getLocale());
                    forwardJsp(request, response, userBeanList);
                    return null;
                }
                LoginAuthBusiness loginAuthBuss = new LoginAuthBusiness();
                LoginManager loginManager = new LoginManager();

                // Beanに設定
                setUserBean(form, userBean);
                userBeanList.add(userBean);

                loginAuthBuss.loginAuth(form, loginManager);
                loginManagerMap.put(userBean.getManagerIp(), loginManager);
            }

            // Set httpOnly to cookie
            response.setHeader("SET-COOKIE", "JSESSIONID=" + session.getId()
                    + "; Path=" + request.getContextPath() + "; HttpOnly");
            response.sendRedirect(request.getContextPath() + "/home");

            // セッション保存
            session.setAttribute(AccessControlConstants.USER_BEAN, userBeanList);
            session.setAttribute(AccessControlConstants.LOGIN_MANAGER,
                    loginManagerMap);

        } catch (CommunicationException ex) {
            // サーバへの接続例外が発生する場合
            setUserBeanError(userFormList, userBeanList, "message.accesscontrol.6",
                    request.getLocale());
            logUtils.error("message.accesscontrol.6", ex.getCause());
            // Jspにフォワードする
            forwardJsp(request, response, userBeanList);

        } catch (AccessException ex) {
            // アクセスの例外発生する場合
            setUserBeanError(userFormList, userBeanList, "message.accesscontrol.6",
                    request.getLocale());
            logUtils.error("message.accesscontrol.6", ex.getCause());
            // Jspにフォワードする
            forwardJsp(request, response, userBeanList);

        } catch (Exception ex) {
            // 予期せぬ例外発生する場合
            setUserBeanError(userFormList, userBeanList, "message.accesscontrol.6",
                    request.getLocale());
            logUtils.error("message.accesscontrol.21", ex.getCause());
            // Jspにフォワードする
            forwardJsp(request, response, userBeanList);
        } catch (Error error) {
            // 予期せぬ例外発生する場合
            setUserBeanError(userFormList, userBeanList, "message.accesscontrol.6",
                    request.getLocale());
            logUtils.error("message.accesscontrol.21", error.getCause());
            // Jspにフォワードする
            forwardJsp(request, response, userBeanList);
        }
        return null;
    }

    /**
     * This method used for logout
     *
     * @author hieunt
     * @param request
     *            : HttpServletRequest
     * @return void
     */
    private void logout(HttpServletRequest request) {
        HttpSession session = request.getSession(true);
        // セッションにuserBeanの存在チェック
        Object userBean = session
                .getAttribute(AccessControlConstants.USER_BEAN);
        if (userBean != null) {
            session.removeAttribute(AccessControlConstants.USER_BEAN);
        }

        LoginSessionManager.processLogout(session);
    }

    /**
     * This method used for check ValidRequestParam
     *
     * @author hieunt
     * @param form
     *            : UserForm
     * @return boolean
     */
    private boolean isValidRequestParam(UserForm form) {
        boolean isValid = true;
        String userName = form.getUserName();
        String password = form.getPassword();
        String urlServer = form.getUrlServer();

        // HTTP リクエストからユーザID判定する
        if (userName == null || userName.trim().equals("")
                || !LoginValidator.validUserID(userName)) {
            isValid = false;
            logUtils.error("Username is invalid ", userName);

            // Sanitize userName
            form.setUserName("");
        }

        // HTTP リクエストからパスワード判定する
        if (password == null || password.equals("")) {
            isValid = false;

            // パスワードがInvalid
            logUtils.error("Password is invalid ", password);
        }

        // Check URL connection
        // HTTP リクエストからユーザID判定する
        if (urlServer == null || urlServer.trim().equals("")
                || !LoginValidator.validURLServer(urlServer)) {
            isValid = false;
            logUtils.error("urlServer is invalid ", urlServer);

            // Sanitize urlServer
            form.setUrlServer("");
        }

        return isValid;
    }

    /**
     * This method used for forward Jsp page
     *
     * @author hieunt
     * @param request
     *            : HttpServletRequest
     * @param response
     *            : HttpServletResponse
     * @param form
     *            : UserForm
     * @throws ServletException
     * @throws IOException
     * @return boolean
     */
    private void forwardJsp(HttpServletRequest request,
            HttpServletResponse response, List<UserBean> userBeanList)
            throws ServletException, IOException {
        request.getSession(true).setAttribute(AccessControlConstants.USER_BEAN,
                userBeanList);

        // login.jspにフォワードする
        getServletContext().getRequestDispatcher(
                AccessControlConstants.VIEW_LOGIN_JSP).forward(request,
                response);
    }

    private void setUserBeanError(List<UserForm> formList, List<UserBean> userBeanList,
            String errorCode, Locale locale) {
        userBeanList.get(0).setErrMessage(WebMessages.getString(errorCode, locale));
        for (int i = 0; i < formList.size(); i++) {
            if (i > userBeanList.size() - 1) {
                UserBean userBean = new UserBean();
                setUserBean(formList.get(i), userBean);
                userBeanList.add(userBean);
            } else {
                userBeanList.get(i).setUserName(formList.get(i).getUserName());
                userBeanList.get(i).setUrlServer(formList.get(i).getUrlServer());
            }
        }
    }

    private void setUserBean(UserForm form, UserBean userBean) {
        userBean.setUserName(form.getUserName());
        userBean.setPassword(form.getPassword());
        userBean.setUrlServer(form.getUrlServer());
    }
}