/*
 
Copyright (C) NEC Corporation 2012. All Rights Reserved. 
Copyright (C) NEC Soft, Ltd. 2012. All Rights Reserved. 
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/
package com.necsoft.hinemos.webclient.accesscontrol.servlet;

import java.io.IOException;
import java.rmi.AccessException;
import java.util.Locale;

import javax.naming.CommunicationException;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import com.necsoft.hinemos.webclient.accesscontrol.AccessControlConstants;
import com.necsoft.hinemos.webclient.accesscontrol.bean.UserBean;
import com.necsoft.hinemos.webclient.accesscontrol.business.LoginAuthBusiness;
import com.necsoft.hinemos.webclient.accesscontrol.form.UserForm;
import com.necsoft.hinemos.webclient.accesscontrol.util.LoginSessionManager;
import com.necsoft.hinemos.webclient.accesscontrol.validation.LoginValidator;
import com.necsoft.hinemos.webclient.json.CommonJson;
import com.necsoft.hinemos.webclient.lang.WebMessages;
import com.necsoft.hinemos.webclient.servlet.CommonServlet;
import com.necsoft.hinemos.webclient.util.LogUtils;
import com.necsoft.hinemos.webclient.util.LoginManager;
import com.necsoft.hinemos.webclient.util.PopulateBeanUtils;

/**
 * Hinemosマネージャに接続するクラス
 * 
 * @version 1.0.0
 * @since 1.0.0
 **/
public class LoginAuthServlet extends CommonServlet {
    /**
	 * 
	 */
    private static final long serialVersionUID = -8809478864238709832L;

    private static LogUtils logUtils = LogUtils
            .getLogUtils(LoginAuthServlet.class);

    /**
     * Handles the HTTP <code>GET</code> method.
     * 
     * @param request
     *            servlet request
     * @param response
     *            servlet response
     */
    public CommonJson process(HttpServletRequest request,
            HttpServletResponse response) throws ServletException, IOException {

        // 中間変数を宣言
        UserForm form = new UserForm();
        LoginManager loginManager = new LoginManager();
        UserBean userBean = new UserBean();
        LoginAuthBusiness loginAuthBuss = new LoginAuthBusiness();
        HttpSession session;

        logout(request);
        session = request.getSession(true);
        PopulateBeanUtils.populateBean(form, request);
        if (!isValidRequestParam(form)) {
            setUserBeanError(form, userBean, "message.accesscontrol.6",
                    request.getLocale());
            forwardJsp(request, response, userBean);
            return null;
        }

        try {
            loginAuthBuss.loginAuth(form, loginManager);

            // Beanに設定
            setUserBean(form, userBean);

            // セッション保存
            session.setAttribute(AccessControlConstants.USER_BEAN, userBean);
            session.setAttribute("loginManager", loginManager);

            // Set httpOnly to cookie
            response.setHeader("SET-COOKIE", "JSESSIONID=" + session.getId()
                    + "; Path=" + request.getContextPath() + "; HttpOnly");
            response.sendRedirect(request.getContextPath() + "/home");
            return null;

        } catch (CommunicationException ex) {
            // サーバへの接続例外が発生する場合
            setUserBeanError(form, userBean, "message.accesscontrol.6",
                    request.getLocale());
            logUtils.error("message.accesscontrol.6", ex.getCause());

        } catch (AccessException ex) {
            // アクセスの例外発生する場合
            setUserBeanError(form, userBean, "message.accesscontrol.6",
                    request.getLocale());
            logUtils.error("message.accesscontrol.6", ex.getCause());

        } catch (Exception ex) {
            // 予期せぬ例外発生する場合
            setUserBeanError(form, userBean, "message.accesscontrol.21",
                    request.getLocale());
            logUtils.error("message.accesscontrol.21", ex.getCause());
        }

        // Jspにフォワードする
        forwardJsp(request, response, userBean);
        return null;
    }

    /**
     * This method used for logout
     * 
     * @param request
     *            : HttpServletRequest
     * @return void
     */
    private void logout(HttpServletRequest request) {
        // セッションにuserBeanの存在チェック
        HttpSession session = request.getSession(true);
        UserBean userBean = (UserBean) session
                .getAttribute(AccessControlConstants.USER_BEAN);

        if (userBean != null) {
            session.removeAttribute(AccessControlConstants.USER_BEAN);
        }

        LoginSessionManager.processLogout(session);
    }

    /**
     * This method used for check ValidRequestParam
     * 
     * @param form
     *            : UserForm
     * @return boolean
     */
    private boolean isValidRequestParam(UserForm form) {
        boolean isValid = true;
        String userName = form.getUserName();
        String password = form.getPassword();
        String urlServer = form.getUrlServer();

        // HTTP リクエストからユーザID判定する
        if (userName == null || userName.trim().equals("")
                || !LoginValidator.validUserID(userName)) {
            isValid = false;
            logUtils.error("Username is invalid ", userName);

            // Sanitize userName
            form.setUserName("");
        }

        // HTTP リクエストからパスワード判定する
        if (password == null || password.equals("")) {
            isValid = false;

            // パスワードがInvalid
            logUtils.error("Password is invalid ", password);
        }

        // Check URL connection
        // HTTP リクエストからユーザID判定する
        if (urlServer == null || urlServer.trim().equals("")
                || !LoginValidator.validURLServer(urlServer)) {
            isValid = false;
            logUtils.error("urlServer is invalid ", urlServer);

            // Sanitize urlServer
            form.setUrlServer("");
        }

        return isValid;
    }

    /**
     * This method used for forward Jsp page
     * 
     * @param request
     *            : HttpServletRequest
     * @param response
     *            : HttpServletResponse
     * @param form
     *            : UserForm
     * @throws ServletException
     * @throws IOException
     * @return boolean
     */
    private void forwardJsp(HttpServletRequest request,
            HttpServletResponse response, UserBean userBean)
            throws ServletException, IOException {
        request.getSession(true).setAttribute(AccessControlConstants.USER_BEAN,
                userBean);

        // login.jspにフォワードする
        getServletContext().getRequestDispatcher(
                AccessControlConstants.VIEW_LOGIN_JSP).forward(request,
                response);
    }

    private void setUserBeanError(UserForm form, UserBean userBean,
            String errorCode, Locale locale) {
        userBean.setUserName(form.getUserName());
        userBean.setUrlServer(form.getUrlServer());
        userBean.setErrMessage(WebMessages.getString(errorCode, locale));
    }

    private void setUserBean(UserForm form, UserBean userBean) {
        userBean.setUserName(form.getUserName());
        userBean.setPassword(form.getPassword());
        userBean.setUrlServer(form.getUrlServer());
    }
}