/*
 
Copyright (C) NEC Corporation 2012. All Rights Reserved. 
Copyright (C) NEC Soft, Ltd. 2012. All Rights Reserved. 
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.necsoft.hinemos.webclient.monitor.servlet;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.commons.lang3.StringEscapeUtils;
import com.necsoft.hinemos.webclient.accesscontrol.AccessControlConstants;
import com.necsoft.hinemos.webclient.bean.Property;
import com.necsoft.hinemos.webclient.common.Constants;
import com.necsoft.hinemos.webclient.common.DataProcess;
import com.necsoft.hinemos.webclient.exception.HinemosWebAccessException;
import com.necsoft.hinemos.webclient.exception.HinemosWebException;
import com.necsoft.hinemos.webclient.exception.HinemosWebInvalidRequestParameterException;
import com.necsoft.hinemos.webclient.exception.HinemosWebRemoteException;
import com.necsoft.hinemos.webclient.json.CommonJson;
import com.necsoft.hinemos.webclient.json.GridDataJson;
import com.necsoft.hinemos.webclient.monitor.MonitorConstants;
import com.necsoft.hinemos.webclient.monitor.action.GetEventListTableDefine;
import com.necsoft.hinemos.webclient.monitor.business.ConfirmEventBusiness;
import com.necsoft.hinemos.webclient.monitor.business.GetEventListBusiness;
import com.necsoft.hinemos.webclient.monitor.form.MonitorEventForm;
import com.necsoft.hinemos.webclient.monitor.util.MonitorMappingUtil;
import com.necsoft.hinemos.webclient.servlet.CommonServlet;
import com.necsoft.hinemos.webclient.util.EndpointManager;
import com.necsoft.hinemos.webclient.util.ExceptionHandlerUtils;
import com.necsoft.hinemos.webclient.util.LogUtils;
import com.necsoft.hinemos.webclient.util.LoginManager;
import com.necsoft.hinemos.webclient.util.PopulateBeanUtils;

/**
 * イベント情報データをリフレッシュするクラス
 * 
 * @version 1.0.0
 * @since 1.0.0
 **/
public class GetEventListServlet extends CommonServlet {
    private static final long serialVersionUID = 996012390849870607L;

    private static LogUtils logUtils = LogUtils
            .getLogUtils(GetEventListServlet.class);

    public CommonJson process(HttpServletRequest request,
            HttpServletResponse response) throws ServletException, IOException,
            HinemosWebAccessException {

        CommonJson result = new CommonJson();

        try {
            // Get data form
            MonitorEventForm form = getDataForm(request);

            // Action form
            doAction(form, request);

            // 中間変数を宣言
            ArrayList<ArrayList<Object>> listInfoData = doLoadListInfoData(
                    request, form);

            // Create JSON
            result = doConvertToJson(listInfoData);

        } catch (HinemosWebAccessException ex) {
            logUtils.info(ex.getMessage());
            throw new HinemosWebAccessException();
        } catch (HinemosWebRemoteException ex) {
            logUtils.error(ex.getMessage(), ex.getCause());
            result = ExceptionHandlerUtils.doBuildCommonJson(
                    ExceptionHandlerUtils.LEVEL_WARNING, ex.getMessage(), request.getLocale());

        } catch (Exception ex) {
            logUtils.error(ex.getMessage(), ex.getCause());
            result = ExceptionHandlerUtils.doBuildCommonJson(
                    ExceptionHandlerUtils.LEVEL_ERROR, ex.getMessage(), request.getLocale());
        }
        return result;
    }

    /**
     * @param request
     * @return
     */
    private MonitorEventForm getDataForm(HttpServletRequest request) {
        MonitorEventForm form = new MonitorEventForm();
        PopulateBeanUtils.populateBean(form, request);
        return form;
    }

    /**
     * @param form
     * @param request
     * @param namingContext
     * @throws HinemosWebException
     */
    private void doAction(MonitorEventForm form, HttpServletRequest request
            ) throws HinemosWebException {
        String action;
        // Check action
        if (form.getAction() != null) {
            action = form.getAction();
            if (action.equals("confirmed")) {
                ArrayList<ArrayList<Object>> list = doGetConfirmedData(request);
                if (list.size() > 0) {
                    //get loginManager session
                    HttpSession session = request.getSession(true);
                    LoginManager loginManager = null;
                    synchronized(session) {
                        loginManager = (LoginManager) session.getAttribute(AccessControlConstants.LOGIN_MANAGER);

                        doConfirmed(list, loginManager.endpointManager);
                    }
                }
            }
        }

    }

    /**
     * @param list
     * @throws HinemosWebException
     */
    private void doConfirmed(ArrayList<ArrayList<Object>> list,
        EndpointManager endpointManager) throws HinemosWebException {
        ConfirmEventBusiness eventBusiness = new ConfirmEventBusiness();
        eventBusiness.confirm(list, endpointManager);
    }

    private ArrayList<ArrayList<Object>> doLoadListInfoData(
            HttpServletRequest request, MonitorEventForm form) throws HinemosWebException {
        ArrayList<ArrayList<Object>> listLogData;

        //get loginManager session
        HttpSession session = request.getSession(true);
        LoginManager loginManager = null;
        synchronized(session) {
            loginManager = (LoginManager) session.getAttribute(AccessControlConstants.LOGIN_MANAGER);
            // Check filter
            if (form.isFilter()) {
                Property condition = doLoadCondition(request);
                GetEventListBusiness business = new GetEventListBusiness();
                listLogData = business.getEventList(form, condition, loginManager.endpointManager, request.getLocale());
            } else {
                GetEventListBusiness business = new GetEventListBusiness();
                listLogData = business.getEventList(form, null, loginManager.endpointManager, request.getLocale());
            }
        }

        return listLogData;
    }

    /**
     * @param request
     * @return
     * @throws HinemosWebException
     */
    private Property doLoadCondition(HttpServletRequest request)
            throws HinemosWebException {
        HttpSession session = request.getSession(true);
        
        if (session.getAttribute(MonitorConstants.EVENT_FILTER_PROPERTIES) == null) {
            // HTTPセッションのタイムアウトが発生しました。ログアウトします。
            throw new HinemosWebException("message.sessiontimeout");
        }

        Property condition = (Property) session
                .getAttribute(MonitorConstants.EVENT_FILTER_PROPERTIES);
        convertCondition(request, condition);
        return condition;
    }

    private GridDataJson doConvertToJson(ArrayList<ArrayList<Object>> listInfoData)
            throws HinemosWebException {
        return DataProcess.doConvertTableContentToJson(listInfoData);
    }

    /**
     * @param request
     * @return
     * @throws HinemosWebException
     */
    private ArrayList<ArrayList<Object>> doGetConfirmedData(
            HttpServletRequest request) throws HinemosWebException {
        ArrayList<ArrayList<Object>> list = new ArrayList<ArrayList<Object>>();
        String actionData = Constants.ACTION_DATA_REQUEST;
        int numData;

        if (request.getParameter("numData") == null) {
            // イベントの確認に失敗しました
            throw new HinemosWebInvalidRequestParameterException(
                    "exception.data.confirm");
        } else {
            numData = Integer.valueOf(request.getParameter("numData"));
            if (numData <= 0) {
                // イベントの確認に失敗しました
                throw new HinemosWebInvalidRequestParameterException(
                        "exception.data.confirm");
            } else if (numData > Constants.EVENT_ROW_NUM) {
                numData = Constants.EVENT_ROW_NUM;
            }
        }

        for (int count = 0; count < numData; count++) {
            ArrayList<Object> row = new ArrayList<Object>();
            String[] row_data = new String[MonitorConstants.NUMBER_HEADER_EVENT];
            for (int cnt_col = 0; cnt_col < MonitorConstants.NUMBER_HEADER_EVENT; cnt_col++) {
                String row_name = getNameRequestParam(actionData, count,
                        MonitorMappingUtil.getEventName(cnt_col));
                if (request.getParameter(row_name) == null) {
                    // イベントの確認に失敗しました
                    throw new HinemosWebInvalidRequestParameterException(
                            "exception.data.confirm");
                } else {
                    row_data[cnt_col] = request.getParameter(row_name);
                }
            }
            ;

            long receiveTime;
            long outputDate;
            try {
                receiveTime = Long
                        .parseLong(row_data[GetEventListTableDefine.RECEIVE_TIME]);
                outputDate = Long
                        .parseLong(row_data[GetEventListTableDefine.OUTPUT_DATE]);
            } catch (Exception e) {
                // イベントの確認に失敗しました
                throw new HinemosWebInvalidRequestParameterException(
                        "exception.data.confirm");
            }
            ;

            row.add(GetEventListTableDefine.PRIORITY,
                row_data[GetEventListTableDefine.PRIORITY]);
            row.add(GetEventListTableDefine.RECEIVE_TIME, 
                new Date(receiveTime));
            row.add(GetEventListTableDefine.OUTPUT_DATE, 
                new Date(outputDate));
            row.add(GetEventListTableDefine.PLUGIN_ID, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.PLUGIN_ID]));
            row.add(GetEventListTableDefine.MONITOR_ID, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.MONITOR_ID]));
            row.add(GetEventListTableDefine.MONITOR_DETAIL_ID, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.MONITOR_DETAIL_ID]));
            row.add(GetEventListTableDefine.FACILITY_ID, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.FACILITY_ID]));
            row.add(GetEventListTableDefine.SCOPE, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.SCOPE]));
            row.add(GetEventListTableDefine.APPLICATION, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.APPLICATION]));
            row.add(GetEventListTableDefine.MESSAGE_ID, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.MESSAGE_ID]));
            row.add(GetEventListTableDefine.MESSAGE, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.MESSAGE]));
            row.add(GetEventListTableDefine.CONFIRMED,
                row_data[GetEventListTableDefine.CONFIRMED]);
            row.add(GetEventListTableDefine.CONFIRM_USER,
                row_data[GetEventListTableDefine.CONFIRM_USER]);
            row.add(GetEventListTableDefine.COMMENT, StringEscapeUtils
                .unescapeHtml4(row_data[GetEventListTableDefine.COMMENT]));
            list.add(row);
        }

        return list;
    }

    /**
     * @param prefixName
     * @param count
     * @return
     */
    private String getNameRequestParam(String prefixName, int count,
            String colName) {
        StringBuffer names = new StringBuffer();
        names.append(prefixName);
        names.append("[");
        names.append(count);
        names.append("][");
        names.append(colName);
        names.append("]");

        return names.toString();
    }

}