/*
 
Copyright (C) NEC Corporation 2012. All Rights Reserved. 
Copyright (C) NEC Soft, Ltd. 2012. All Rights Reserved. 
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/



package com.necsoft.hinemos.webclient.validator;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

@SuppressWarnings({ "rawtypes", "unchecked" })
public class ActionErrors implements Serializable {
    private static final long serialVersionUID = 1L;

    /**
     * <p>
     * Compares ActionErrorItem objects.
     * </p>
     */
    private static final Comparator ACTION_ITEM_COMPARATOR = new Comparator() {
        public int compare(Object o1, Object o2) {
            return ((ActionErrorItem) o1).getOrder()
                    - ((ActionErrorItem) o2).getOrder();
        }
    };

    /**
     * <p>
     * The accumulated set of <code>ActionError</code> objects (represented as
     * an ArrayList) for each property, keyed by property name.
     * </p>
     */
    protected HashMap errors = new HashMap();

    /**
     * <p>
     * The current number of the property/key being added. This is used to
     * maintain the order errors are added.
     * </p>
     */
    protected int iCount = 0;

    // --------------------------------------------------------- Public Methods

    /**
     * <p>
     * Create an empty <code>ActionErrors</code> object.
     * </p>
     */
    public ActionErrors() {
        super();
    }

    /**
     * <p>
     * Create an <code>ActionErrors</code> object initialized with the given
     * errors.
     * </p>
     * 
     * @param errors
     *            The errors to be initially added to this object. This
     *            parameter can be <code>null</code>.
     */
    public ActionErrors(ActionErrors errors) {
        super();
        this.add(errors);
    }

    /**
     * <p>
     * Add a error to the set of errors for the specified property. An order of
     * the property/key is maintained based on the initial addition of the
     * property/key.
     * </p>
     * 
     * @param property
     *            Property name
     * @param error
     *            The error to be added
     */
    public void add(String property, ActionError error) {
        ActionErrorItem item = (ActionErrorItem) errors.get(property);
        List list;

        if (item == null) {
            list = new ArrayList();
            item = new ActionErrorItem(list, iCount++, property);

            errors.put(property, item);
        } else {
            list = item.getList();
        }

        list.add(error);
    }

    /**
     * <p>
     * Adds the errors from the given <code>ActionErrors</code> object to this
     * set of errors. The errors are added in the order they are returned from
     * the <code>properties</code> method. If a error's property is already in
     * the current <code>ActionErrors</code> object, it is added to the end of
     * the list for that property. If a error's property is not in the current
     * list it is added to the end of the properties.
     * </p>
     * 
     * @param actionErrors
     *            The <code>ActionErrors</code> object to be added. This
     *            parameter can be <code>null</code>.
     * @since Struts 1.1
     */
    public void add(ActionErrors actionErrors) {
        if (actionErrors == null) {
            return;
        }

        // loop over properties
        Iterator props = actionErrors.properties();

        while (props.hasNext()) {
            String property = (String) props.next();

            // loop over errors for each property
            Iterator msgs = actionErrors.get(property);

            while (msgs.hasNext()) {
                ActionError msg = (ActionError) msgs.next();

                this.add(property, msg);
            }
        }
    }

    /**
     * <p>
     * Clear all errors recorded by this object.
     * </p>
     */
    public void clear() {
        errors.clear();
    }

    /**
     * <p>
     * Return <code>true</code> if there are no errors recorded in this
     * collection, or <code>false</code> otherwise.
     * </p>
     * 
     * @return <code>true</code> if there are no errors recorded in this
     *         collection; <code>false</code> otherwise.
     * @since Struts 1.1
     */
    public boolean isEmpty() {
        return (errors.isEmpty());
    }

    /**
     * <p>
     * Return the set of all recorded errors, without distinction by which
     * property the errors are associated with. If there are no errors recorded,
     * an empty enumeration is returned.
     * </p>
     * 
     * @return An iterator over the errors for all properties.
     */
    public Iterator get() {
        if (errors.isEmpty()) {
            return Collections.EMPTY_LIST.iterator();
        }

        ArrayList results = new ArrayList();
        ArrayList actionItems = new ArrayList();

        for (Iterator i = errors.values().iterator(); i.hasNext();) {
            actionItems.add(i.next());
        }

        // Sort ActionErrorItems based on the initial order the
        // property/key was added to ActionErrors.
        Collections.sort(actionItems, ACTION_ITEM_COMPARATOR);

        for (Iterator i = actionItems.iterator(); i.hasNext();) {
            ActionErrorItem ami = (ActionErrorItem) i.next();

            for (Iterator msgsIter = ami.getList().iterator(); msgsIter
                    .hasNext();) {
                results.add(msgsIter.next());
            }
        }

        return results.iterator();
    }

    /**
     * <p>
     * Return the set of errors related to a specific property. If there are no
     * such errors, an empty enumeration is returned.
     * </p>
     * 
     * @param property
     *            Property name
     * @return An iterator over the errors for the specified property.
     */
    public Iterator get(String property) {
        ActionErrorItem item = (ActionErrorItem) errors.get(property);

        if (item == null) {
            return (Collections.EMPTY_LIST.iterator());
        } else {
            return (item.getList().iterator());
        }
    }

    /**
     * <p>
     * Return the set of property names for which at least one error has been
     * recorded. If there are no errors, an empty <code>Iterator</code> is
     * returned.
     * 
     * @return An iterator over the property names for which errors exist.
     */
    public Iterator properties() {
        if (errors.isEmpty()) {
            return Collections.EMPTY_LIST.iterator();
        }

        ArrayList results = new ArrayList();
        ArrayList actionItems = new ArrayList();

        for (Iterator i = errors.values().iterator(); i.hasNext();) {
            actionItems.add(i.next());
        }

        // Sort ActionErrorItems based on the initial order the
        // property/key was added to ActionErrors.
        Collections.sort(actionItems, ACTION_ITEM_COMPARATOR);

        for (Iterator i = actionItems.iterator(); i.hasNext();) {
            ActionErrorItem ami = (ActionErrorItem) i.next();

            results.add(ami.getProperty());
        }

        return results.iterator();
    }

    /**
     * <p>
     * Return the number of errors recorded for all properties (including global
     * errors). <strong>NOTE</strong> - it is more efficient to call
     * <code>isEmpty</code> if all you care about is whether or not there are
     * any errors at all.
     * </p>
     * 
     * @return The number of errors associated with all properties.
     */
    public int size() {
        int total = 0;

        for (Iterator i = errors.values().iterator(); i.hasNext();) {
            ActionErrorItem ami = (ActionErrorItem) i.next();

            total += ami.getList().size();
        }

        return (total);
    }

    /**
     * <p>
     * Return the number of errors associated with the specified property.
     * </p>
     * 
     * @param property
     *            Property name
     * @return The number of errors associated with the property.
     */
    public int size(String property) {
        ActionErrorItem item = (ActionErrorItem) errors.get(property);

        return (item == null) ? 0 : item.getList().size();
    }

    /**
     * <p>
     * Returns a String representation of this ActionErrors' property name=error
     * list mapping.
     * </p>
     * 
     * @return String representation of the errors
     * @see Object#toString()
     */
    public String toString() {
        return this.errors.toString();
    }

    /**
     * <p>
     * This class is used to store a set of errors associated with a
     * property/key and the position it was initially added to list.
     * </p>
     */
    protected class ActionErrorItem implements Serializable {
        /**
		 * 
		 */
        private static final long serialVersionUID = 4483045484934834232L;

        /**
         * <p>
         * The list of <code>ActionError</code>s.
         * </p>
         */
        protected List list = null;

        /**
         * <p>
         * The position in the list of errors.
         * </p>
         */
        protected int iOrder = 0;

        /**
         * <p>
         * The property associated with <code>ActionError</code>.
         * </p>
         */
        protected String property = null;

        /**
         * <p>
         * Construct an instance of this class.
         * </p>
         * 
         * @param list
         *            The list of ActionErrors.
         * @param iOrder
         *            The position in the list of errors.
         * @param property
         *            The property associated with ActionError.
         */
        public ActionErrorItem(List list, int iOrder, String property) {
            this.list = list;
            this.iOrder = iOrder;
            this.property = property;
        }

        /**
         * <p>
         * Retrieve the list of errors associated with this item.
         * </p>
         * 
         * @return The list of errors associated with this item.
         */
        public List getList() {
            return list;
        }

        /**
         * <p>
         * Set the list of errors associated with this item.
         * </p>
         * 
         * @param list
         *            The list of errors associated with this item.
         */
        public void setList(List list) {
            this.list = list;
        }

        /**
         * <p>
         * Retrieve the position in the error list.
         * </p>
         * 
         * @return The position in the error list.
         */
        public int getOrder() {
            return iOrder;
        }

        /**
         * <p>
         * Set the position in the error list.
         * </p>
         * 
         * @param iOrder
         *            The position in the error list.
         */
        public void setOrder(int iOrder) {
            this.iOrder = iOrder;
        }

        /**
         * <p>
         * Retrieve the property associated with this item.
         * </p>
         * 
         * @return The property associated with this item.
         */
        public String getProperty() {
            return property;
        }

        /**
         * <p>
         * Set the property associated with this item.
         * </p>
         * 
         * @param property
         *            The property associated with this item.
         */
        public void setProperty(String property) {
            this.property = property;
        }

        /**
         * <p>
         * Construct a string representation of this object.
         * </p>
         * 
         * @return A string representation of this object.
         */
        public String toString() {
            return this.list.toString();
        }
    }
}