#!/bin/bash

# Copyright (C) 2012 NTT DATA Corporation
# 
# This program is free software; you can redistribute it and/or
# Modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation, version 2.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details

# bash configuration
SCRIPT_DIR=$(cd $(dirname $0);pwd)
. ${SCRIPT_DIR}/../hinemos.cfg
. ${SCRIPT_DIR}/../sbin/hinemos_utility.sh

export PROG=`basename $0`
DIR=`dirname $0`
USER=`/usr/bin/whoami`
HOST=`hostname`

cd ${HINEMOS_HOME}/var/log

########################################
#  Local Variable
########################################

# thread dump count
THREAD_DUMP_CNT=3

# process check interval [sec]
#   this variable is valid with -W option
SHUTDOWN_CHECK_INTERVAL=1

# process check timeout [sec]
#   this variable is valid with -W option
SHUTDOWN_CHECK_TIMEOUT=300

########################################
#  Local Message
########################################

# INFO
MSG_I001="waiting for Java Virtual Machine shutdown..."
MSG_I002="Java Virtual Machine stopped"

# WARN

# ERROR
MSG_E001="shutdown timeout. please check status of Java Virtual Machine"

########################################
# Function
########################################

function usage {
	echo "usage : ${PROG} [-F|-W]"
	echo "options:"
	echo "  -F   force shutdown without process check (dependency, current status)"
	echo "  -W   do not wait until operation completes"
}

function shutdown {
	Logging "${MSG_I001}"
	
	# store thread dump
	if [ -e ${JVM_PID_FILE} ]
	then
		for CNT in `seq 1 ${THREAD_DUMP_CNT}`
		do
			kill -3 `cat ${JVM_PID_FILE}`
			echo Thread Dump ${CNT}
			sleep 1
		done
		cp ${HINEMOS_LOG_DIR}/jvm_stdout.log ${HINEMOS_LOG_DIR}/jvm_stdout.log.`date +%Y-%m-%d_%H%M`
	fi
	
	if [ -f ${JVM_PID_FILE} ]
	then
		read PID < ${JVM_PID_FILE}
		if [ `ps --no-headers --pid ${PID} e | grep "${JAVA_HOME}/bin/java.*com.clustercontrol.HinemosManagerMain" | wc -l` -gt 0 ]
		then
			if [ "x${FORCE}" == "xtrue" ]
			then
				# shutdown (send KILL signal)
				kill -9 ${PID}
			else
				# shutdown (send TERM signal)
				kill ${PID}
			fi
		fi
	fi
	
	TIME_COUNT=0
	while [ 1 ]
	do
		if [ "x"${SHUTDOWN_WAIT} = "xfalse" ]
		then
			Logging "${MSG_I002} (with -W option)"
			break
		fi
		
		GetJVMProcessState
		RET=$?
		if [ ${RET} -eq 1 ] || [ ${RET} -eq 2 ]
		then
			echo "done"
			Logging "${MSG_I002}"
			break
		fi
	
		if [ "${TIME_COUNT}" -ge ${SHUTDOWN_CHECK_TIMEOUT} ]
		then
			Logging "${MSG_E001}"
			exit -4
		fi
	
		echo -n "."
		/bin/sleep ${SHUTDOWN_CHECK_INTERVAL}
		TIME_COUNT=$((${TIME_COUNT} + ${SHUTDOWN_CHECK_INTERVAL}))
	done
}

########################################
# SHELL
########################################

# check argument
for OPT in $@
do
	case ${OPT} in
		--help)
			usage
			exit 0
			;;
	esac
done

# option check
FORCE="false"
SHUTDOWN_WAIT="true"
while getopts FW OPT
do
	case ${OPT} in
		F)
			FORCE="true"
			;;
		W)
			SHUTDOWN_WAIT="false"
			;;
		*)
			ExitIllegalOptionErrorWithoutLogger
			;;
	esac
done

shift $(( $OPTIND - 1 ))

if [ ! $# = 0 ]
then
	ExitIllegalArgumentError
fi

# check user executed
CheckUser ${JVM_USER}

# check process
if [ "x${FORCE}" != "xtrue" ]
then
	CheckJVMProcessRunning
fi

# shutdown JVM
shutdown

# remove pid file
rm -f ${JVM_PID_FILE}
rm -f ${HINEMOS_LOCK_FILE}

exit 0
