using System;
using System.Collections.Generic;
using System.Drawing;
using GustFront;
using RVII;
using App = RVII.RVIIManager;

public class Window : IHasCommand
{
    private abstract class WindowObject : Action, IHasMember
    {
        private ScriptEngine myEngine;
        public ScriptEngine Engine { get { return myEngine; } }

        public WindowObject Parent { get; set; }
        public virtual bool Visible { get; set; }
        public virtual Rectangle Bounds { get; set; }
        public virtual Rectangle ClientBounds { get; set; }
        public virtual Color Color { get; set; }
        public virtual FontInfo Font { get; set; }
        public virtual string Text { get; set; }

        protected WindowObject(ScriptEngine engine, UnregisterActionCallback unreg)
            : base(unreg)
        {
            this.myEngine = engine;

            this.Visible = true;
            this.Bounds = Rectangle.Empty;
            this.ClientBounds = Rectangle.Empty;
            this.Color = Color.White;
            this.Font = FontInfo.DefaultFont;
            this.Text = String.Empty;
        }

        private List<WindowObject> myChildren = new List<WindowObject>();

        public void Add(WindowObject wo)
        {
            wo.Parent = this;
            myChildren.Add(wo);
            RegisterAction(wo);
        }

        public void Remove(WindowObject wo)
        {
            UnregisterAction(wo);
            myChildren.Remove(wo);
            wo.Parent = null;
        }

        public IList<WindowObject> GetChildren()
        {
            return myChildren.AsReadOnly();
        }

        public override void EndOfAction()
        {
            if (Parent != null) Parent.Remove(this);
            base.EndOfAction();
        }

        protected abstract void OnRender(IGraphicsDevice r);
        public override void Render(IGraphicsDevice r)
        {
            if (Visible) {
                OnRender(r);
                base.Render(r);
            }
        }

        public virtual bool GetMember(string name, ScriptValue[] @params, out ScriptValue result)
        {
            switch (name) {
                case "Add":
                    Add((WindowObject)@params[0].Instance);
                    result = null;
                    return true;
                case "Remove":
                    Remove((WindowObject)@params[0].Instance);
                    result = null;
                    return true;
                case "Parent":
                    result = new ScriptValue(Parent);
                    return true;
                case "Children":
                    result = new ScriptValue(CollectionUtil.CreateStdList(myChildren));
                    return true;
                case "Dispose":
                    EndOfAction();
                    result = null;
                    return true;
                case "Visible":
                    result = Visible;
                    return true;
                case "X":
                    result = Bounds.X;
                    return true;
                case "Y":
                    result = Bounds.Y;
                    return true;
                case "Width":
                    result = Bounds.Width;
                    return true;
                case "Height":
                    result = Bounds.Height;
                    return true;
                case "ClientWidth":
                    result = ClientBounds.Width;
                    return true;
                case "ClientHeight":
                    result = ClientBounds.Height;
                    return true;
                case "SetClientBounds":
                    ClientBounds = new Rectangle(
                        @params[0].AsInt32(), @params[1].AsInt32(),
                        @params[2].AsInt32(), @params[3].AsInt32());
                    result = null;
                    return true;
                case "Color":
                    result = Color.ToArgb();
                    return true;
                case "Font":
                    result = new ScriptValue(Font.Source);
                    return true;
                case "Text":
                    result = Text;
                    return true;
            }
            result = null;
            return false;
        }

        public virtual bool SetMember(string name, ScriptValue[] @params)
        {
            switch (name) {
                case "Visible":
                    Visible = @params[0].AsBoolean();
                    return true;
                case "X":
                    Bounds = new Rectangle(@params[0].AsInt32(), Bounds.Y, Bounds.Width, Bounds.Height);
                    return true;
                case "Y":
                    Bounds = new Rectangle(Bounds.X, @params[0].AsInt32(), Bounds.Width, Bounds.Height);
                    return true;
                case "Width":
                    Bounds = new Rectangle(Bounds.X, Bounds.Y, @params[0].AsInt32(), Bounds.Height);
                    return true;
                case "Height":
                    Bounds = new Rectangle(Bounds.X, Bounds.Y, Bounds.Width, @params[0].AsInt32());
                    return true;
                case "Color":
                    Color = Color.FromArgb(@params[0].AsInt32());
                    return true;
                case "Font":
                    Font = (FontInfo)((UserElement)@params[0].Instance).Data;
                    return true;
                case "Text":
                    Text = @params[0].AsString();
                    return true;
            }
            return false;
        }
    }

    private class WOPanel : WindowObject
    {
        private struct FillingParam
        {
            public Rectangle Bounds;
            public Color Color;
        }

        bool closing;
        List<FillingParam> myBackColor;
        List<FillingParam> myForeColor;

        public WOPanel(ScriptEngine engine, UnregisterActionCallback unreg)
            : base(engine, unreg)
        {
            myBackColor = new List<FillingParam>(2);
            myForeColor = new List<FillingParam>(16);

            engine.RunProcessAction(this);
        }

        public override void EndOfAction()
        {
            foreach (Action obj in GetRegisteredActions()) {
                obj.EndOfAction();
            }
            base.EndOfAction();

            closing = true;
        }

        public override void Tick(TimeSpan delta)
        {
            if (!closing) {
                base.Tick(delta);
            } else {
                Engine.UnregisterThreadAction(this);
            }
        }

        protected override void OnRender(IGraphicsDevice r)
        {
        }

        public override void Render(IGraphicsDevice r)
        {
            if (Visible) {
                GraphicsObjects go = (GraphicsObjects)r;
                Rectangle b;
                foreach (FillingParam fp in myBackColor) {
                    b = fp.Bounds; b.Offset(Bounds.Location);
                    Visual.Paint(go, b, Color.FromArgb((int)fp.Color.A * Color.A / 255, fp.Color));
                }
                b = ClientBounds; b.Offset(Bounds.Location);
                Visual.BeginWindowMode(go);
                try {
                    base.Render(r);
                } finally {
                    Visual.EndWindowMode(go, b, Color);
                }
                foreach (FillingParam fp in myForeColor) {
                    b = fp.Bounds; b.Offset(Bounds.Location);
                    Visual.Paint(go, b, Color.FromArgb((int)fp.Color.A * Color.A / 255, fp.Color));
                }
            }
        }

        public override bool GetMember(string name, ScriptValue[] @params, out ScriptValue result)
        {
            FillingParam obj;

            switch (name) {
                case "AddBackColor":
                    obj.Bounds = new Rectangle(
                        @params[0].AsInt32(), @params[1].AsInt32(),
                        @params[2].AsInt32(), @params[3].AsInt32());
                    obj.Color = Color.FromArgb(@params[4].AsInt32());
                    myBackColor.Add(obj);
                    result = obj.GetHashCode();
                    return true;
                case "AddForeColor":
                    obj.Bounds = new Rectangle(
                        @params[0].AsInt32(), @params[1].AsInt32(),
                        @params[2].AsInt32(), @params[3].AsInt32());
                    obj.Color = Color.FromArgb(@params[4].AsInt32());
                    myForeColor.Add(obj);
                    result = obj.GetHashCode();
                    return true;
                default:
                    return base.GetMember(name, @params, out result);
            }
        }
    }

    private class WOFill : WindowObject
    {
        public WOFill(ScriptEngine engine, UnregisterActionCallback unreg) : base(engine, unreg) { }

        protected override void OnRender(IGraphicsDevice r)
        {
            Visual.Paint((GraphicsObjects)r, Bounds, Color);
        }
    }

    private class WOLabel : WindowObject
    {
        public RVII.StringAlignment LineAlignment { get; set; }
        public RVII.StringAlignment Alignment { get; set; }

        public WOLabel(ScriptEngine engine, UnregisterActionCallback unreg)
            : base(engine, unreg)
        {
            LineAlignment = RVII.StringAlignment.Near;
            Alignment = RVII.StringAlignment.Near;
        }

        protected override void OnRender(IGraphicsDevice r)
        {
            if (Bounds.Width == 0 && Bounds.Height == 0) {
                Visual.DrawString((GraphicsObjects)r, Font, Text, Bounds.Location, Color);
            } else {
                Visual.DrawString((GraphicsObjects)r, Font, Text, Bounds, Color, LineAlignment, Alignment);
            }
        }

        public override bool GetMember(string name, ScriptValue[] @params, out ScriptValue result)
        {
            switch (name) {
                case "LineAlignment":
                    result = (int)LineAlignment;
                    return true;
                case "Alignment":
                    result = (int)Alignment;
                    return true;
                default:
                    return base.GetMember(name, @params, out result);
            }
        }

        public override bool SetMember(string name, ScriptValue[] @params)
        {
            switch (name) {
                case "LineAlignment":
                    LineAlignment = (RVII.StringAlignment)@params[0].AsInt32();
                    return true;
                case "Alignment":
                    Alignment = (RVII.StringAlignment)@params[0].AsInt32();
                    return true;
                default:
                    return base.SetMember(name, @params);
            }
        }
    }

    private class WOImage : WindowObject
    {
        private TextureInfo myTexture;
        private Rectangle mySrcBounds;

        public TextureInfo Texture { get { return myTexture; } }
        public Rectangle SrcBounds { get { return mySrcBounds; } set { mySrcBounds = value; } }

        public WOImage(ScriptEngine engine, UnregisterActionCallback unreg, TextureInfo tex)
            : base(engine, unreg)
        {
            myTexture = tex;
            Bounds = new Rectangle(0, 0, tex.Instance.GetLevelDescription(0).Width, tex.Instance.GetLevelDescription(0).Height);
            mySrcBounds = Bounds;
        }

        protected override void OnRender(IGraphicsDevice r)
        {
            GraphicsObjects go = (GraphicsObjects)r;

            go.SpriteBatch.Transform = SharpDX.Matrix.Scaling((float)Bounds.Width / mySrcBounds.Width, (float)Bounds.Height / mySrcBounds.Height, 1.0f) * SharpDX.Matrix.Translation(GSConv.Vector3(Bounds.Location));
            go.SpriteBatch.Draw(myTexture.Instance, GSConv.Color(Color), GSConv.Rectangle(mySrcBounds), null, null);
            go.SpriteBatch.Transform = SharpDX.Matrix.Identity;
        }

        public override bool GetMember(string name, ScriptValue[] @params, out ScriptValue result)
        {
            switch (name) {
                case "SrcX":
                    result = mySrcBounds.X;
                    return true;
                case "SrcY":
                    result = mySrcBounds.Y;
                    return true;
                case "SrcWidth":
                    result = mySrcBounds.Width;
                    return true;
                case "SrcHeight":
                    result = mySrcBounds.Height;
                    return true;
                default:
                    return base.GetMember(name, @params, out result);
            }
        }

        public override bool SetMember(string name, ScriptValue[] @params)
        {
            switch (name) {
                case "SrcX":
                    mySrcBounds.X = @params[0].AsInt32();
                    return true;
                case "SrcY":
                    mySrcBounds.Y = @params[0].AsInt32();
                    return true;
                case "SrcWidth":
                    mySrcBounds.Width = @params[0].AsInt32();
                    return true;
                case "SrcHeight":
                    mySrcBounds.Height = @params[0].AsInt32();
                    return true;
                default:
                    return base.SetMember(name, @params);
            }
        }
    }

    private ScriptEngine myTarget = null;

    public void SetTarget(ScriptEngine obj)
    {
        myTarget = obj;
    }

    private ScriptValue cmdCreatePanel(ScriptValue[] @params)
    {
        WOPanel obj = new WOPanel(myTarget, null);

        if (@params.Length == 4) {
            obj.Bounds = new Rectangle(
                @params[0].AsInt32(), @params[1].AsInt32(),
                @params[2].AsInt32(), @params[3].AsInt32());
            obj.ClientBounds = new Rectangle(0, 0, obj.Bounds.Width, obj.Bounds.Height);
        } else if (@params.Length == 1) {
            WindowObject wo = (WindowObject)@params[0].Instance;
            obj.Bounds = wo.Bounds;
            obj.ClientBounds = new Rectangle(0, 0, wo.Bounds.Width, wo.Bounds.Height);
            obj.Add(wo);
        } else {
            obj.Bounds = App.Global.GetViewBounds();
            obj.ClientBounds = new Rectangle(0, 0, obj.Bounds.Width, obj.Bounds.Height);
        }

        return new ScriptValue(obj);
    }

    private ScriptValue cmdCreateFill(ScriptValue[] @params)
    {
        WOFill obj = new WOFill(myTarget, null);
        obj.Bounds = new Rectangle(
            @params[0].AsInt32(), @params[1].AsInt32(),
            @params[2].AsInt32(), @params[3].AsInt32());
        obj.Color = Color.FromArgb(@params[4].AsInt32());
        return new ScriptValue(obj);
    }

    private ScriptValue cmdCreateLabel(ScriptValue[] @params)
    {
        WOLabel obj = new WOLabel(myTarget, null);
        if (@params.Length == 5) {
            obj.Bounds = new Rectangle(@params[0].AsInt32(), @params[1].AsInt32(), 0, 0);
            obj.Font = (FontInfo)((UserElement)@params[2].Instance).Data;
            obj.Color = Color.FromArgb(@params[3].AsInt32());
            obj.Text = @params[4].AsString();
        } else {
            obj.Bounds = new Rectangle(
                @params[0].AsInt32(), @params[1].AsInt32(),
                @params[2].AsInt32(), @params[3].AsInt32());
            obj.LineAlignment = (RVII.StringAlignment)@params[4].AsInt32();
            obj.Alignment = (RVII.StringAlignment)@params[5].AsInt32();
            obj.Font = (FontInfo)((UserElement)@params[6].Instance).Data;
            obj.Color = Color.FromArgb(@params[7].AsInt32());
            obj.Text = @params[8].AsString();
        }
        return new ScriptValue(obj);
    }

    private ScriptValue cmdCreateImage(ScriptValue[] @params)
    {
        WOImage obj = new WOImage(myTarget, null, (TextureInfo)@params[0].Instance);
        if (@params.Length == 3) {
            obj.Bounds = new Rectangle(
                @params[1].AsInt32(), @params[2].AsInt32(),
                obj.SrcBounds.Width, obj.SrcBounds.Height);
        } else if (@params.Length == 5) {
            obj.Bounds = new Rectangle(
                @params[1].AsInt32(), @params[2].AsInt32(),
                @params[3].AsInt32(), @params[4].AsInt32());
        } else if (@params.Length == 7) {
            obj.SrcBounds = new Rectangle(
                @params[3].AsInt32(), @params[4].AsInt32(),
                @params[5].AsInt32(), @params[6].AsInt32());
            obj.Bounds = new Rectangle(
                @params[1].AsInt32(), @params[2].AsInt32(),
                obj.SrcBounds.Width, obj.SrcBounds.Height);
        } else if (@params.Length == 9) {
            obj.Bounds = new Rectangle(
                @params[1].AsInt32(), @params[2].AsInt32(),
                @params[3].AsInt32(), @params[4].AsInt32());
            obj.SrcBounds = new Rectangle(
                @params[5].AsInt32(), @params[6].AsInt32(),
                @params[7].AsInt32(), @params[8].AsInt32());
        }
        return new ScriptValue(obj);
    }

    public IDictionary<string, CommandImpl> GetCommandMap()
    {
        Dictionary<string, CommandImpl> dic = new Dictionary<string, CommandImpl>();

        dic.Add("CreatePanel", cmdCreatePanel);
        dic.Add("CreateFill", cmdCreateFill);
        dic.Add("CreateLabel", cmdCreateLabel);
        dic.Add("CreateImage", cmdCreateImage);

        return dic;
    }
}
