
/*	
	GameboyAdvanceVM 
		- Nintendo GameboyAdvance Emulator
	Copyright 2002 Y_N y_n@users.sourceforge.jp
	Homepage https://sourceforge.jp/projects/gbaemu/
*/


#define	FOREG		(u8)((opcode>>16)&0xF)	/*1st operand register*/
#define	SOREG		(u8)((opcode)&0xF)		/*2nd operand register*/
#define	DEREG		(u8)((opcode>>12)&0xF)	/*Destination register*/

#define	REG_0_4		(u8)((opcode)&0xF)
#define	REG_12_4	(u8)((opcode>>12)&0xF)
#define	REG_16_4	(u8)((opcode>>16)&0xF)

#define	GETCOND		(u8)((opcode>>28)&0x0F)	/*RfBVR[h*/

#define	IMM_OPERAND	(opcode & BIT_25_)		/*ltO*/

_inline void SET_SUB_FLAGS(u32 a, u32 b, u32 c)
{	//sub,rsb,rsc,cmp
	if(!c)SF(Z_); else RF(Z_);
	if(c & BIT_31_)SF(N_); else RF(N_);
	if(((a & ~b) | (a & ~c) | (~b & ~c))>>31)SF(C_); else RF(C_);
	if(((a & ~(b | c)) | ((b & c) & ~a))>>31)SF(V_); else RF(V_);
}

_inline void SET_ADD_FLAGS(u32 a, u32 b, u32 c)
{	//add,adc,cmn
	if(!c)SF(Z_); else RF(Z_);
	if(c & BIT_31_)SF(N_); else RF(N_);
	if(((a & b) | (a & ~c) | (b & ~c)) & BIT_31_)SF(C_); else RF(C_);
	if(((a & b & ~c) | (~a & ~b & c)) & BIT_31_)SF(V_); else RF(V_);
}

u32 imm_shift(u32 opcode)
{	/*Vtgl*/
	u32	bit, temp32;
	u8	shift, shift_type, rm;

	rm	= SOREG;
	shift_type = (u8)((opcode>>5)&3);

	if(opcode & 0x08){	/*Rs(8-11)*/
		shift = (opcode>>8) & 0xF;
		switch(shift_type){
		case 0:	/*00 = logical left@_IVtg O*/
			return ((arm.reg[rm])<<shift);
		case 1:	/*01 = logical right _IEVtg O*/
			return ((arm.reg[rm])>>shift);
		case 2:	/*10 = arithmetic right ZpIEVtg 31rbg̒l*/
			temp32 = ((arm.reg[rm])>>shift);
			if(arm.reg[rm] & BIT_31_){
				for(bit=0x80000000; shift; bit>>=1){
					temp32 |= bit;
					shift--;
				}
			}
			return temp32;
		case 3:	/*11 = rotate right lւ*/
			return ((arm.reg[rm])>>shift)|((arm.reg[rm])<<(32-shift));
		}
	}else{	/*Rs(7-11)*/
		shift = (opcode>>7) & 0x1F;
		switch(shift_type){
		case 0:	/*logical left*/
			return ((arm.reg[rm])<<shift);
		case 1:	/*logical right*/
			return ((arm.reg[rm])>>shift);
			break;
		case 2:	/*10 = arithmetic right ZpIEVtg 31rbg̒l*/
			temp32 = ((arm.reg[rm])>>shift);
			if(arm.reg[rm] & BIT_31_){
				for(bit=0x80000000; shift; bit>>=1){
					temp32 |= bit;
					shift--;
				}
			}
			return temp32;
		case 3:	/*11 = rotate right lւ*/
			return ((arm.reg[rm])>>shift)|((arm.reg[rm])<<(32-shift));
		}
	}

	return 0;
}

u32 imm_rotate(u32 opcode)
{	/*E[e[gl*/
	u32	shift, imm_val;

	shift	= ((opcode>>8)&0xF)<<1;
	imm_val	= opcode&0xFF;

	return (imm_val>>shift)|((imm_val)<<(32-shift));
}

_inline void arm_adc()
{	/*Rd := Rn + Op2 + Carry <Data Processing> (L[tZ)*/
	u32	opcode, operand, temp;
	u8	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	temp = LSB_FC;
	if((arm.reg[rn] + operand + temp)>0xFFFFFFFF)SF(C_); else RF(C_);

	arm.reg[rd] = arm.reg[rn] + operand + temp;

	if(!arm.reg[rd])SF(Z_); else RF(Z_);
	if(arm.reg[rd] & BIT_31_)SF(N_); else RF(N_);
/*
      z_flag = (A + data + (c_flag ? 1 : 0)) & 0xFF;
      n_flag = temp;
      v_flag = ((~(A ^ data)) & (A ^ temp) & 0x80);
      if(temp > 0x9F)temp += 0x60;
      c_flag = (temp > 0xFF);
*/
	arm.cycle = 1;
}

_inline void arm_add()
{	/*Rd := Rn + Op2 <Data Processing> (Z)*/
	u32	opcode, operand;
	u8	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if((arm.reg[rn] + operand)>0xFFFFFFFF)SF(C_); else RF(C_);

	arm.reg[rd] = arm.reg[rn] + operand;

/*	u32 op1 = BASE_REG;	// x[XWX^
	u32 op2 = DP_REG_OPERAND(IMM_SHIFT);//ItZbg
	DEST_REG = op1 + op2;	//*/

	SET_ADD_FLAGS(arm.reg[rn], operand, arm.reg[rd]);

	if(!arm.reg[rd])SF(Z_); else RF(Z_);
	if(arm.reg[rd] & BIT_31_)SF(N_); else RF(N_);

	arm.cycle = 1;
}

_inline void arm_and()
{	/*Rd := Rn AND Op2 <Data Processing> (_)*/
	u32	opcode, operand;
	u8	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;
	
	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	arm.reg[rd] = arm.reg[rn] & operand;

	if(!arm.reg[rd])SF(Z_); else RF(Z_);
	if(arm.reg[rd] & BIT_31_)SF(N_); else RF(N_);
	
	arm.cycle = 1;
}

_inline void arm_b()
{	/*R15 := address ()*/
	u32	opcode,	offset, address;

	opcode	= OPCODE;
	/*l͂S{*/
	offset	= ((opcode&0x00FFFFFF)<<2);

	address = PC + offset + 4;

	/*ItZbg͈̔͂MSB^Ȃ}CiX Zl{̒l*/
	if(offset & BIT_25_)address -= BIT_25_<<1;	/*tItZbg*/

	PC = address;

	arm.cycle = 3;
}

_inline void arm_bic()
{	/*Rd := Rn AND NOT Op2 <Data Processing> (rbgNA)*/
	u32	opcode, operand;
	u8	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	arm.reg[rd] = arm.reg[rn] & ~operand;

	if(!arm.reg[rd])SF(Z_); else RF(Z_);
	if(arm.reg[rd] & BIT_31_)SF(N_); else RF(N_);
	
	arm.cycle = 1;
}

_inline void arm_bl()
{	/*R14 := R15, R15 := address (ƃN)*/
	u32	opcode,	offset, address;

	opcode	= OPCODE;
	/*l͂S{*/
	offset	= ((opcode&0x00FFFFFF)<<2);

	R14 = PC + 4;
	address = PC + offset + 4;

	/*ItZbg͈̔͂MSB^Ȃ}CiX Zl{̒l*/
	if(offset & BIT_25_)address -= BIT_25_<<1;	/*tItZbg*/

	PC = address;

	arm.cycle = 3;
}

_inline void arm_bx()
{	/*R15 := Rn (Ɩ߃Xe[g̕ύX)*/
	u32	opcode;
	u8 rn;
	
	opcode	= OPCODE;
	rn		= REG_0_4;
	R15		= arm.reg[rn];

	if(R15 & BIT_1_){
		CPSR |= T_;		/*ŉʃrbgPȂTHUMBXe[gɕύX*/
	}else{
		CPSR &= ~T_;	/*ŉʃrbgOȂARMXe[gɕύX*/
	}

	arm.cycle = 3;
}

_inline void arm_cdp()
{	/*(Coprocessor-specific)*/

	arm.cycle = 2;
}

_inline void arm_cmn()
{	/*CPSR flags := Rn + Op2 <Data Processing> (RvZbTf[^)*/
	u32	opcode, operand;
	u32	temp;
	u8	rn;

	opcode	= OPCODE;
	rn		= FOREG;

	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	temp = arm.reg[rn] + operand;

/*	if(!temp)SF(Z_); else RF(Z_);
	if((arm.reg[rn] - operand)>0xFFFFFFFF)SF(C_); else RF(C_);
	if(temp & BIT_31_)SF(N_); else RF(N_);
*/	SET_ADD_FLAGS(arm.reg[rn], operand, temp);

	arm.cycle = 1;
}

_inline void arm_cmp()
{	/*CPSR flags := Rn - Op2 <Data Processing> (r)*/
	u32	opcode, operand;
	u32	temp;
	u8	rn;

	opcode	= OPCODE;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	temp = arm.reg[rn] - operand;

/*	if((arm.reg[rn] - operand)>0)SF(C_); else RF(C_);
	if(!temp)SF(Z_); else RF(Z_);
	if(temp & BIT_31_)SF(N_); else RF(N_);
*/
	SET_SUB_FLAGS(arm.reg[rn], operand, temp);

	arm.cycle = 1;
}

_inline void arm_eor()
{	/*Rd := (Rn AND NOT Op2) OR (op2 AND NOT Rn) <Data Processing> (rI_a)*/
	u32	opcode, operand;
	u8	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;
	
	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	} 

	arm.reg[rd] = arm.reg[rn] ^ operand;

	if(!arm.reg[rd])SF(Z_); else RF(Z_);
	if(arm.reg[rd] & BIT_31_)SF(N_); else RF(N_);
	
	arm.cycle = 1;
}

_inline void arm_ldc()
{	/*Coprocessor load (RvZbTւ̓])*/
	u32	opcode;
	u8	rd, rn, cp;
	u8	offset;

	opcode	= OPCODE;
	rd		= (u8)((opcode>>12)&0x0F);
	rn		= (u8)((opcode>>16)&0x0F);
	cp		= (u8)((opcode)&0x0F);
	offset	= (u8)(opcode);

	arm.cycle = 3;
}

_inline void arm_ldm()
{	/*Stack manipulation (̃WX^̃[h,POP)*/
	u32	opcode,	address, bit;
	u8	rn, lst;

	opcode	= OPCODE;
	rn		= (u8)((opcode>>16)&0x0F);	/*16-19 4bit*/

	address = arm.reg[rn];

	for(bit=BIT_15_,lst=15; bit; bit>>=1,lst--){
		if(opcode & bit){//TRACE("%2d,%08X\n",lst, bit);
			if(opcode & BIT_24_){	/*Pre: ]OɃItZbgǉ*/
				if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
					address += 4;
				}else{					/*Down: ItZbgx[X猸Z*/
					address -= 4;
				}
			}
			arm.reg[lst] = RMem32(address);
			if(!(opcode & BIT_24_)){/*Post: ]ɃItZbgǉ*/
				if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
					address += 4;
				}else{					/*Down: ItZbgx[X猸Z*/
					address -= 4;
				}
			}
		}
	}

	if(opcode & BIT_21_){	/*Write-back SP(rn)̓eXV*/
		arm.reg[rn] = address;
	}
	
	arm.cycle = 3;
}

_inline void arm_ldr()
{	/*Rd := (address) (烌WX^ւ̓ǂݍ)*/
	u32	opcode;
	u32	offset, address;
	u32	rd, rn;

	opcode = OPCODE;
	rd     = DEREG;
	rn     = (u8)((opcode>>16)&0xF);	/*AhX̃x[XWX^*/
	
	address = arm.reg[rn];

	if(rn==15)address += 8;	/*ΏۂPC̏ꍇ*/

//	if(opcode & BIT_24_){	/*1 = Pre: ItZbgǉOɓ]*/
//		arm.reg[rd] = (opcode & BIT_22_)?RMem8(address):RMem32(address);
//	}

	if(opcode & BIT_25_){	/*1 = ItZbg̓WX^[*/
		offset = imm_shift(opcode);
	}else{					/*0 = ItZbg𑦒l*/
		offset = opcode & 0xFFF;
	}

	if(opcode & BIT_23_){	/*1 = Up: ItZbgx[XɉZ*/
		address += offset;
	}else{					/*0 = Down: ItZbgx[X猸Z*/
		address -= offset;
	}

//	if(!(opcode & BIT_24_)){	/*0 = Post: ItZbgǉ]*/
		arm.reg[rd] = (opcode & BIT_22_)?RMem8(address):RMem32(address);
//	}

	arm.cycle = 3;
}

_inline void disasm_arm_ldrs()
{	/*Rd := (address) (WX^ւ̃[h) Register offset*/
	u32	opcode, address;
	u8	cond, rd, rn, rm;

	opcode	= OPCODE;
	cond	= GETCOND;
	rd		= DEREG;					/*ΏۃWX^*/
	rn		= (u8)((opcode>>16)&0xF);	/*AhX̃x[XWX^*/
	rm		= (u8)(opcode&0xF);

	address = arm.reg[rn];

	if(opcode & BIT_24_){	/*1 = Pre: ]OɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += arm.reg[rm];
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= arm.reg[rm];
		}
	}

	if(opcode & BIT_5_){	/*H*/
		if(opcode & BIT_6_){	/*S*/
			/*Signed Halfwords*/
			WMem16(address, arm.reg[rd]);
		}else{
			/*Signed byte*/
			WMem8(address, arm.reg[rd]);
		}
	}else{
		if(opcode & BIT_6_){	/*S*/
			/*Unsigned Halfwords*/
			WMem16(address, arm.reg[rd]);
		}else{
			/*SWP instruction*/
		}
	}

	if(!(opcode & BIT_24_)){	/*0 = Post: ]ɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += arm.reg[rm];
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= arm.reg[rm];
		}
	}

	if(opcode & BIT_21_){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rd] = address;
	}

	arm.cycle = 3;
}

_inline void disasm_arm_ldrs_imm()
{	/*Rd := (address) (WX^ւ̃[h) Immidiate offset*/
	u32	opcode,	offset, address;
	u8	cond, rd, rn;

	opcode	= OPCODE;
	cond	= GETCOND;
	rd		= DEREG;					/*ΏۃWX^*/
	rn		= (u8)((opcode>>16)&0xF);	/*AhX̃x[XWX^*/
	offset	= (opcode&0xF)|((opcode>>4)&0xF0);

	address = arm.reg[rn];

	if(opcode & BIT_24_){	/*1 = Pre: ]OɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if(opcode & BIT_5_){	/*H*/
		if(opcode & BIT_6_){	/*S*/
			/*Signed Halfwords*/
			WMem16(address, arm.reg[rd]);
		}else{
			/*Signed byte*/
			WMem8(address, arm.reg[rd]);
		}
	}else{
		if(opcode & BIT_6_){	/*S*/
			/*Unsigned Halfwords*/
			WMem16(address, arm.reg[rd]);
		}else{
			/*SWP instruction*/
		}
	}

	if(!(opcode & BIT_24_)){	/*0 = Post: ]ɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if(opcode & BIT_21_){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rd] = address;
	}

	arm.cycle = 3;
}

_inline void arm_mcr()
{	/*cRn := rRn {<op>cRm} (CPUWX^RvZbTWX^ւ̓])*/
	u32	opcode;
	u8	rn, rd, rm;

	opcode	= OPCODE;
	rn		= (u8)((opcode>>16)&0x0F);
	rd		= (u8)((opcode>>12)&0x0F);
	rm		= (u8)((opcode)&0x0F);
	
	arm.cycle = 3;

}

_inline void arm_mla()
{	/*Rd := (Rm * Rs) + Rn (ϘaZ)*/
	u32	opcode;
	u8	rd, rm, rs, rn;

	opcode	= OPCODE;
	
	rd		= (u8)((opcode>>16)&0x0F);
	rs		= (u8)((opcode>>8)&0x0F);
	rm		= (u8)((opcode)&0x0F);
	rn		= (u8)((opcode>>12)&0x0F);

	if(((arm.reg[rm] * arm.reg[rs]) + arm.reg[rn])>0xFFFFFFFF)SF(C_); else RF(C_);

	arm.reg[rd] = (arm.reg[rm] * arm.reg[rs]) + arm.reg[rn];

	if(!arm.reg[rd])SF(Z_); else RF(Z_);
	if(arm.reg[rd] & BIT_31_)SF(N_); else RF(N_);

	arm.cycle = 2;
}

_inline void arm_mov()
{	/*Rd : = Op2 <Data Processing> (WX^͒萔̑)*/
	u32	opcode, operand;
	u8	rd;

	opcode	= OPCODE;
	rd		= DEREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	arm.reg[rd] = operand;

	if(!arm.reg[rd])SF(Z_); else RF(Z_);
	if(arm.reg[rd] & BIT_31_)SF(N_); else RF(N_);

	arm.cycle = 1;
}

_inline void arm_mrc()
{	/*Rn := cRn {<op>cRm} (RvZbTWX^CPUWX^ւ̓])*/
	u32	opcode;
	u8	rn, rd, rm;

	opcode	= OPCODE;
	rn		= (u8)((opcode>>16)&0x0F);
	rd		= (u8)((opcode>>12)&0x0F);
	rm		= (u8)((opcode)&0x0F);

	arm.cycle = 3;
}

_inline void arm_mrs()
{	/*Rn := PSR (Xe[^XWX^烌WX^ւ̓])*/
	u32	opcode,	spsr;
	u8	rd;

	opcode	= OPCODE;
	rd		= (u8)((opcode>>12)&0x0F);
	spsr	= ((opcode>>22)&0x01);	/*src PSRtO*/

	arm.reg[rd] = spsr?arm.reg[SR]:arm.reg[CPSR];
	
	arm.cycle = 3;
}

_inline void arm_msr()
{	/*PSR := Rm (WX^Xe[^XWX^ւ̓])*/
	u32	opcode, dpsr;
	u8	rm;

	opcode	= OPCODE;
	rm		= (u8)((opcode)&0x0F);
	dpsr	= (opcode & BIT_22_);	/*dest PSRtO*/

	arm.reg[dpsr?SR:CPSR] = arm.reg[rm];
	
	arm.cycle = 3;
}

_inline void arm_mul()
{	/*Rd := Rm * Rs (ώZ)*/
	u32	opcode;
	u8	rd, rm, rs;

	opcode	= OPCODE;
	rd		= (u8)((opcode>>16)&0x0F);
	rs		= (u8)((opcode>>8)&0x0F);
	rm		= (u8)((opcode)&0x0F);

	if((arm.reg[rm] * arm.reg[rs])>0xFFFFFFFF)SF(C_); else RF(C_);

	arm.reg[rd] = arm.reg[rm] * arm.reg[rs];

	if(!arm.reg[rd])SF(Z_); else RF(Z_);
	if(arm.reg[rd] & BIT_31_)SF(N_); else RF(N_);

	arm.cycle = 3;
}

_inline void arm_mvn()
{	/*Rd := 0xFFFFFFFF EOR Op2 <Data Processing> (1̕␔)*/
	u32	opcode, operand;
	u8	rd;

	opcode	= OPCODE;
	rd		= DEREG;
	
	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	arm.reg[rd] = 0xFFFFFFFF ^ operand;

	if(!arm.reg[rd])SF(Z_); else RF(Z_);
	if(arm.reg[rd] & BIT_31_)SF(N_); else RF(N_);
	
	arm.cycle = 1;
}

_inline void arm_orr()
{	/*Rd := Rn OR Op2 <Data Processing> (_a)*/
	u32	opcode, operand;
	u8	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){	/*Immediate Operand*/
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	arm.reg[rd] = arm.reg[rn] | operand;

	if(!arm.reg[rd])SF(Z_); else RF(Z_);
	if(arm.reg[rd] & BIT_31_)SF(N_); else RF(N_);

	arm.cycle = 1;
}

_inline void arm_rsb()
{	/*Rd := Op2 - Rn <Data Processing> (tZ)*/
	u32	opcode, operand;
	u8	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}


	if((operand - arm.reg[rn])>0)SF(C_); else RF(C_);

	arm.reg[rd] = operand - arm.reg[rn];

	if(!arm.reg[rd])SF(Z_); else RF(Z_);
	if(arm.reg[rd] & BIT_31_)SF(N_); else RF(N_);

	arm.cycle = 1;
}

_inline void arm_rsc()
{	/*Rd := Op2 - Rn - 1 + Carry <Data Processing> (L[ttZ)*/
	u32	opcode, operand, temp;
	u8	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;
	
	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	temp = LSB_FC;
	if((operand - arm.reg[rn] - 1 + LSB_FC)>0)SF(C_); else RF(C_);

	arm.reg[rd] = operand - arm.reg[rn] - 1 + LSB_FC;
	
	if(!arm.reg[rd])SF(Z_); else RF(Z_);
	//if(arm.reg[rd] & BIT_31_)SF(N_); else RF(N_);

	arm.cycle = 1;
}

_inline void arm_sbc()
{	/*Rd := Rn - Op2 - 1 + Carry <Data Processing> (L[tZ)*/
	u32	opcode, operand, temp;
	u8	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	temp = LSB_FC;
	if((arm.reg[rn] - operand - 1 + temp)>0)SF(C_); else RF(C_);

	arm.reg[rd] = arm.reg[rn] - operand - 1 + temp;

	if(!arm.reg[rd])SF(Z_); else RF(Z_);
	if(arm.reg[rd] & BIT_31_)SF(N_); else RF(N_);

	arm.cycle = 1;
}

_inline void arm_stc()
{	/*address := CRn (RvZbTWX^̓e֊i[)*/
	u32	opcode;
	u8	rd, rn, cp;
	u8	offset;

	opcode	= OPCODE;
	rd		= (u8)((opcode>>12)&0x0F);
	rn		= (u8)((opcode>>16)&0x0F);
	cp		= (u8)((opcode)&0x0F);
	offset	= (u8)(opcode);

	arm.cycle = 3;
}

_inline void arm_stm()
{	/*Stack manipulation (̃WX^֊i[,Push)*/
	u32	opcode,	address, bit;
	u8	rn;		/*base register*/
	u8	lst;	/*register list*/

	opcode	= OPCODE;
	rn		= (u8)((opcode>>16)&0x0F);	/*16-19 4bit*/

	/*base address*/
	address = arm.reg[rn];

	for(bit=1,lst=0; bit!=BIT_16_; bit<<=1,lst++){
		if(opcode & bit){
			if(opcode & BIT_24_){	/*Pre: ItZbg̒ǉOɓ]*/
				if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
					address += 4;
				}else{					/*Down: ItZbgx[X猸Z*/
					address -= 4;
				}
			}
			WMem32(address, arm.reg[lst]);
			if(!(opcode & BIT_24_)){/*Post: ItZbgǉɓ]*/
				if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
					address += 4;
				}else{					/*Down: ItZbgx[X猸Z*/
					address -= 4;
				}
			}
		}
	}

	if(opcode & BIT_21_){	/*Write-back sp(rn)̓eXV*/
		arm.reg[rn] = address;
	}
	
	arm.cycle = 3;
}

_inline void arm_str()
{	/*<address> := Rd (WX^֊i[)*/
	u32	opcode,offset, address;
	u8	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= (u8)((opcode>>16)&0xF);	/*AhX̃x[XWX^*/

	address = arm.reg[rn];

	if(rn==15)address += 8;	/*ΏۂPC̏ꍇ*/

	if(opcode & BIT_24_){	/*1 = Pre: ItZbg̒ǉOɓ]*/
		if(opcode & BIT_22_){	/*Byte ]*/
			WMem8(address, arm.reg[rd]);
		}else{					/*Double word ]*/
			WMem32(address, arm.reg[rd]);
		}
	}

	if(opcode & BIT_25_){	/*1 = ItZbg̓WX^*/
		offset = imm_shift(opcode);
	}else{					/*0 = ItZbg͑l*/
		offset = opcode & 0xFFF;
	}

	if(opcode & BIT_23_){	/*1 = Up: ItZbgx[XɉZ*/
		address += offset;
	}else{					/*0 = Down: ItZbgx[X猸Z*/
		address -= offset;
	}

	if(!(opcode & BIT_24_)){	/*0 = Post: ItZbg̒ǉɓ]*/
		if(opcode & BIT_22_){	/*Byte ]*/
			WMem8(address, arm.reg[rd]);
		}else{					/*Double word ]*/
			WMem32(address, arm.reg[rd]);
		}
	}
	
	arm.cycle = 2;
}

_inline disasm_arm_strs()
{	/*<address> := Rd (WX^֊i[) Register offset*/
	u32	opcode, address;
	u8	cond, rd, rn, rm;

	opcode	= OPCODE;
	cond	= GETCOND;
	rd		= DEREG;					/*ΏۃWX^*/
	rn		= (u8)((opcode>>16)&0xF);	/*AhX̃x[XWX^*/
	rm		= (u8)(opcode&0xF);

	address = arm.reg[rn];

	if(opcode & BIT_24_){	/*1 = Pre: ]OɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += arm.reg[rm];
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= arm.reg[rm];
		}
	}

	if(opcode & BIT_5_){	/*H*/
		if(opcode & BIT_6_){	/*S*/
			/*Signed Halfwords*/
			arm.reg[rd] = RMem16(address);
		}else{
			/*Signed byte*/
			arm.reg[rd] = RMem8(address);
		}
	}else{
		if(opcode & BIT_6_){	/*S*/
			/*Unsigned Halfwords*/
			arm.reg[rd] = RMem16(address);
		}else{
			/*SWP instruction*/
		}
	}

	if(!(opcode & BIT_24_)){	/*0 = Post: ]ɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += arm.reg[rm];
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= arm.reg[rm];
		}
	}

	if(opcode & BIT_21_){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rd] = address;
	}

	arm.cycle = 3;
}

_inline disasm_arm_strs_imm()
{	/*<address> := Rd (WX^֊i[) Immidiate offset*/
	u32	opcode,	offset, address;
	u8	cond, rd, rn;

	opcode	= OPCODE;
	cond	= GETCOND;
	rd		= DEREG;					/*ΏۃWX^*/
	rn		= (u8)((opcode>>16)&0xF);	/*AhX̃x[XWX^*/
	offset	= (opcode&0xF)|((opcode>>4)&0xF0);

	address = arm.reg[rn];

	if(opcode & BIT_24_){	/*1 = Pre: ]OɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if(opcode & BIT_5_){	/*H*/
		if(opcode & BIT_6_){	/*S*/
			/*Signed Halfwords*/
			arm.reg[rd] = RMem16(address);
		}else{
			/*Signed byte*/
			arm.reg[rd] = RMem8(address);
		}
	}else{
		if(opcode & BIT_6_){	/*S*/
			/*Unsigned Halfwords*/
			arm.reg[rd] = RMem16(address);
		}else{
			/*SWP instruction*/
		}
	}

	if(!(opcode & BIT_24_)){	/*0 = Post: ]ɃItZbgǉ*/
		if(opcode & BIT_23_){	/*Up: ItZbgx[XɉZ*/
			address += offset;
		}else{					/*Down: ItZbgx[X猸Z*/
			address -= offset;
		}
	}

	if(opcode & BIT_21_){	/*Write-back: ݐ̃AhXcĂ*/
		arm.reg[rd] = address;
	}

	arm.cycle = 3;
}

_inline void arm_sub()
{	/*Rd := Rn - Op2 <Data Processing> (Z)*/
	u32	opcode, operand;
	u8	rd, rn;

	opcode	= OPCODE;
	rd		= DEREG;
	rn		= FOREG;

	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	if((arm.reg[rn] - operand)>0)SF(C_); else RF(C_);

	arm.reg[rd] = arm.reg[rn] - operand;

	if(!arm.reg[rd])SF(Z_); else RF(Z_);
	if(arm.reg[rd] & BIT_31_)SF(N_); else RF(N_);

	arm.cycle = 1;
}

_inline void arm_swi()
{	/*OS call (\tgEFA荞)*/
	u32	opcode, comment;

	opcode	= OPCODE;
	comment	= opcode & 0x00FFFFFF;

	arm.cycle = 3;
}

_inline void arm_swp()
{	/*Rd := [Rn], [Rn] := Rm (Pf[^̌)*/
	u32	opcode;
	u8	rd, rn, rm;

	opcode	= OPCODE;
	rd		= (u8)((opcode>>12)&0x0F);
	rn		= (u8)((opcode>>16)&0x0F);
	rm		= (u8)((opcode)&0x0F);

	arm.reg[rd] = arm.reg[rn];
	arm.reg[rn] = arm.reg[rm];

	arm.cycle = 4;
}

_inline void arm_teq()
{	/*CPSR flags := Rn EOR Op2 <Data Processing> (rbgp^[̔r)*/
	u32	opcode,	operand, temp;
	u8	rn;

	opcode	= OPCODE;
	rn		= FOREG;

	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	/*CPSR = arm.reg[rn] ^ operand;*/
	temp = arm.reg[rn] ^ operand;

	if(!temp)SF(Z_); else RF(Z_);
	if(temp & BIT_31_)SF(N_); else RF(N_);

	arm.cycle = 1;
}

_inline void arm_tst()
{	/*CPSR flags := Rn AND Op2 <Data Processing> (w肵rbg̃eXg)*/
	u32	opcode,	operand, temp;
	u8	rn;

	opcode	= OPCODE;
	rn		= FOREG;

	if(IMM_OPERAND){
		operand = imm_rotate(opcode);
	}else{
		operand = imm_shift(opcode);
	}

	/*CPSR = arm.reg[rn] & operand;*/
	temp = arm.reg[rn] & operand;

	if(!temp)SF(Z_); else RF(Z_);
	if(temp & BIT_31_)SF(N_); else RF(N_);

	arm.cycle = 1;
}

int CMainFrame::exec_arm_state()
{
	u32	opcode;
	BOOL pass, cond;
	u8	opcode_4_4, opcode_21_4, opcode_22_6, opcode_25_3;
	u8	opcode_26_2, opcode_28_4;
	char message[50];

	opcode = OPCODE;
	cond = TRUE;

	/*IyR[hw肳͈͂̃rbgo*/
	opcode_4_4 = (u8)((opcode>>4)&0x0F);	/*4-7 4bit*/
	opcode_21_4 = (u8)((opcode>>21)&0x0F);	/*21-24 4bit*/
	opcode_22_6 = (u8)((opcode>>22)&0x3F);	/*22-27 6bit*/
	opcode_25_3 = (u8)((opcode>>25)&0x07);	/*25-27 3bit*/
	opcode_26_2 = (u8)((opcode>>26)&0x03);	/*26-27 2bit*/
	opcode_28_4 = (u8)((opcode>>28)&0x0F);	/*28-31 4bit*/

	switch(opcode_28_4){	/*s - Condition Field*/
	case 0x0:
		if(FZ)cond = FALSE;
		break;		/*if Z set		- equal*/
	case 0x1:
		if(!FZ)cond = FALSE;
		break;		/*if Z clear	- not equal*/
	case 0x2:
		if(FC)cond = FALSE;
		break;		/*if C set		- unsigned higher or same*/
	case 0x3:
		if(!FC)cond = FALSE;
		break;		/*if C clear	- unsigne lower*/
	case 0x4:
		if(FN)cond = FALSE;
		break;		/*if N set		- negative*/
	case 0x5:
		if(!FN)cond = FALSE;
		break;		/*if N clear	- positive or zero*/
	case 0x6:
		if(FV)cond = FALSE;
		break;		/*if V set		- overflow*/
	case 0x7:
		if(!FV)cond = FALSE;
		break;		/*if V clear	- no overflow*/
	case 0x8:
		if(FC && !FZ)cond = FALSE;
		break;		/*if C set and Z clear	- unsigned higher*/
	case 0x9:
		if(!FC || FZ)cond = FALSE;
		break;		/*if C clear and Z set	- unsigned lower or same*/
	case 0xA:
		if((FN && FV) || (!FN && !FV))cond = FALSE;
		break;		/*if ((N set and V set) or (N clear and V clear))	- greater or equal*/
	case 0xB:
		if((!FN && FV) || (FN && !FV))cond = FALSE;
		break;		/*if ((N set and V clear) or (N clear and V set))	- less than*/
	case 0xC:
		if((!FZ && (FN || FV)) || (!FN || !FV))cond = FALSE;
		break;		/*if (Z clear and(N or V set) or (N or V clear))- greater than*/
	case 0xD:
		if((FZ || (FN && !FV)) || (!FN && FV))cond = FALSE;
		break;		/*if (Z set or(N set and V clear) or (N clear and V set))- less than or equal*/
	default:
		cond = FALSE;
		break;		/*(ignored)	- always */
	}

	if(!cond){
		switch(opcode_25_3){
		case 0x5:	/*101*/
			if(opcode&BIT_24_){
				arm_bl();		/*BL - Branch with Link*/
			}else{
				arm_b();		/*B - Branch*/
			}
			pass = FALSE;
			break;
		case 0x6:	/*110*/
			if(opcode&BIT_20_){
				arm_ldc();		/*LDC - Load from Coprocessor*/	
			}else{
				arm_stc();		/*STC - Store to Coprocessor*/
			}
			pass = FALSE;
			break;
		case 0x7:	/*111*/
			if(!(opcode&BIT_24_)){	/*24bit==0*/
				if(opcode&BIT_4_){
					if(opcode&BIT_20_){
						arm_mrc();		/*MRC - move from coprocessor to register*/
					}else{
						arm_mcr();		/*MCR - move from register to coprocessor*/
					}
				}else{
					arm_cdp();			/*CDP - Coprocessor Data Operations*/
				}
			}else{
				arm_swi();		/*SWI - Software interruput*/			
			}
			pass = FALSE;
			break;
		default:
			pass = TRUE;
			break;
		}

		if(pass){
			if((opcode&0x0FFFFFF0)==0x012FFF10)arm_bx();
			else if(opcode_26_2==0x1){
				if(opcode & BIT_20_){
					arm_ldr();				/*LDR - Rd := (address)*/
				}else{
					arm_str();				/*STR - <address> := Rd*/
				}
			}else if(!opcode_26_2){	/*00 = 26-27bit*/
				switch(opcode_21_4){	/*21-24bit*/
				case 0x0: arm_and();break;	/*0000 = AND - Rd:= Op1 AND Op2*/
				case 0x1: arm_eor();break;	/*0001 = EOR - Rd:= Op1 EOR Op2*/
				case 0x2: arm_sub();break;	/*0010 = SUB - Rd:= Op1 - Op2*/
				case 0x3: arm_rsb();break;	/*0011 = RSB - Rd:= Op2 - Op1*/
				case 0x4: arm_add();break;	/*0100 = ADD - Rd:= Op1 + Op2*/
				case 0x5: arm_adc();break;	/*0101 = ADC - Rd:= Op1 + Op2 + C*/
				case 0x6: arm_sbc();break;	/*0110 = SBC - Rd:= Op1 - Op2 + C*/
				case 0x7: arm_rsc();break;	/*0111 = RSC - Rd:= Op2 - Op1 + C*/
				case 0x8: arm_tst();break;	/*1000 = TST - set condition codes on Op1 AND Op2*/
				case 0x9: arm_teq();break;	/*1001 = TEQ - set condition codes on Op1 EOR Op2*/
				case 0xA: arm_cmp();break;	/*1010 = CMP - set condition codes on Op1 - Op2*/
				case 0xB: arm_cmn();break;	/*1011 = CMN - set condition codes on Op1 + Op2*/
				case 0xC: arm_orr();break;	/*1100 = ORR - Rd:= Op1 OR Op2*/
				case 0xD: arm_mov();break;	/*1101 = MOV - Rd:= Op2*/
				case 0xE: arm_bic();break;	/*1110 = BIC - Rd:= Op1 AND NOT Op2*/
				case 0xF: arm_mvn();break;	/*1111 = MVN - Rd:= NOT Op2*/
				}
			}else if(!(opcode_22_6) && (opcode_4_4==0x09)){
				if(opcode&BIT_21_){
					arm_mla();/*MLA - Rd := (Rm * Rs) + Rn*/
				}else{
					arm_mul();				/*MUL - Rd := Rm * Rs*/
				}
			}else{
				pass = TRUE;
				if(opcode_25_3==0x04){
					if(opcode & BIT_20_){	/*Load from memory*/
						arm_ldm();
					}else{					/*Store to memory*/
						arm_stm();
					}
					pass = FALSE;
				}
				if(((opcode>>23)&0x1F)==0x02 && pass){	/*23_5_00010*/
					if(((opcode>>16)&0x3F)==0x0F){	/*16_6_001111*/
						if(!(opcode&0xFFF)){	/*0_12_000000000000*/
							arm_mrs();
							pass = FALSE;	/*MRS - transfer PSR contents to a register*/
						}
					}else if(((opcode>>4)&0x3FFFF)==0x29F00){	/*12_10_101001111100000000*/
						arm_msr();
						pass = FALSE;	/*MSR - transfer register contents to PSR*/
					}
				}
				if(pass){
					opcode_25_3 = (u8)((opcode>>25)&0x07);
					if(opcode_25_3==0x3 && (opcode&0x10)){	/*Undefine opcode*/
						sprintf(message, "Undefine ARM opcode \"0x%08X\" at 0x%08X", opcode, PC);
					}else{	/*Unknow opcode*/
						sprintf(message, "Unknow ARM opcode \"0x%08X\" at 0x%08X", opcode, PC);
					}
					MessageBox(message, NULL, MB_ICONSTOP);
					CPUIsRunning = FALSE;
				}
			}
		}
	}

	PC += 4;

	return 0;
}

void InitializeRegister()
{
	R0	=	0x00000000;
	R1	=	0x00000000;
	R2	=	0x00000000;
	R3	=	0x00000000;
	R4	=	0x00000000;
	R5	=	0x00000000;
	R6	=	0x00000000;
	R7	=	0x00000000;
	R8	=	0x00000000;
	R9	=	0x00000000;
	R10	=	0x00000000;
	R11	=	0x00000000;
	R12	=	0x00000000;
	SP	=	0x03007F00;	/*R13==SR X^bN|C^*/
	R14	=	0x00000000;	/*R14==LR NWX^*/
	PC	=	0x08000000;	/*R15==PC vO|C^*/
	CPSR=	0x0000001F;	/*R16==CPSR Xe[^Xz*/
}

