/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.content.board;

import java.text.SimpleDateFormat;
import java.util.*;
import java.io.*;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;

import com.aibonware.viewnaver.*;
import com.aibonware.viewnaver.model.*;
import com.aibonware.viewnaver.task.*;
import com.aibonware.viewnaver.window.*;
import com.aibonware.viewnaver.command.*;
import com.aibonware.viewnaver.content.*;
import com.aibonware.viewnaver.browser.*;
import com.aibonware.viewnaver.format.*;
import com.aibonware.viewnaver.config.*;

@SuppressWarnings("serial")
public abstract class BaseBoardView<C extends Board> extends TabContentView {
	public C board;
	private Browser browser;
	protected JTextField pageText;
	protected JButton closeButton;
	protected JToolBar toolbar;
	BaseBoardContent<C> content;
	private NThread selectedThread;
	private DefaultPosterSelection posterSelection = new DefaultPosterSelection();

	JButton nextButton = new JButton("");
	JButton prevButton = new JButton("");

	public BaseBoardView(TabContentCategory category) {
		super(category);

		browser = ViewNaver.instance.getBrowserFactory().createBrowser();

		final JPopupMenu popup = new JPopupMenu();

		ViewNaver.instance.posterMenuItemCreator.addPosterMenuItems(this, posterSelection, popup);
		popup.addSeparator();

		popup.add(new JMenuItem(new AbstractAction("Cancel") {
			public void actionPerformed(ActionEvent e) {
				popup.setVisible(false);
			}
		}));

		browser.addBrowserListener(new BrowserListener() {
			public boolean beforeNavigate(final String url) {
				if(url.startsWith("thread:")) {
					SwingUtilities.invokeLater(new Runnable() {
						public void run() {
							int nid = Integer.parseInt(url.substring("thread:".length()));
							final NThread threadTitle = BaseBoardView.this.board.getTitleByNid(nid);
							
							TaskFlow taskFlow = ViewNaver.instance.taskFlowList.getThreadFlow(BaseBoardView.this.board.boardName, nid);
							taskFlow.execute(new ShowThreadCommand(threadTitle, BaseBoardView.this.board.boardName));
						}
					});
				} else if(url.startsWith("poster:")) {
					SwingUtilities.invokeLater(new Runnable() {	
						public void run() {
							int nid = Integer.parseInt(url.substring("poster:".length()));
							selectedThread = BaseBoardView.this.board.getTitleByNid(nid);
							Poster poster = new Poster(selectedThread.poster, selectedThread.country);

							posterSelection.setCurrentPoster(poster);

							Point mousePoint = MouseInfo.getPointerInfo().getLocation();
							Point compPoint = browser.getComponent().getLocationOnScreen();
		
							popup.show(browser.getComponent(), mousePoint.x - compPoint.x, mousePoint.y - compPoint.y);
						}
					});
					
				} else {
					SwingUtilities.invokeLater(new Runnable() {	
						public void run() {
							new OpenURLCommand(BaseBoardView.this, url, false).run();
						}
					});
				}

				return false;
			}
		});

		browser.beginDocWriting();
		browser.writeDocHtml(
				"<html><head>"
				+ "<meta http-equiv=\"Content-Type\" content=\"text/html\">"
				+ "<style> "
				+ "TABLE  { border-width: 0px 1px 1px 0px ;  border-style: solid; border-color: black;  } "
				+ "TD { font-family: 'MS UI Gothic'; font-size: 10pt; border-width: 1px 0px 0px 1x; border-style: solid;  border-color: black; }"
				+ "A:link, A:visited, A:active { color: black; border-width: 1px; text-decoration: none; }"
				+ "A:hover { color: gray; border-width: 1px; text-decoration: none; }"
				+ "</style>"
				+ "</head><body><font color=\"black\"></font></body></html>");
		browser.endDocWriting();

		closeButton = new JButton("");
		pageText = new JTextField("");

		pageText.setMaximumSize(new Dimension(30, 30));

		prevButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if(board.pageNo < 1) {
					ViewNaver.println("I[łB");
					return;
				}

				setSpecifiedPageNo(board.pageNo-1);
				loadNewPage(board, getSpecifiedPageNo(), false);
			}
		});

		nextButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setSpecifiedPageNo(board.pageNo+1);
				loadNewPage(board, getSpecifiedPageNo(), false);
			}
		});

		closeButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				ViewNaver.instance.mainWindow.container.removeContent(content);
			}
		});

		toolbar = new JToolBar();

		toolbar.add(closeButton);
		toolbar.addSeparator();
		toolbar.add(prevButton);
		toolbar.add(pageText);
		toolbar.add(nextButton);

		toolbar.setFloatable(false);

		add(toolbar, BorderLayout.NORTH);
		add(browser.getComponent(), BorderLayout.CENTER);
	}

	protected void setSpecifiedPageNo(int newPageNo) {
		pageText.setText(String.valueOf(newPageNo));
	}
	
	protected int getSpecifiedPageNo() {
		int newPageNo = -1;
		
		try {
			newPageNo = Integer.parseInt(pageText.getText());
		} catch(NumberFormatException e) {
		}
		
		return newPageNo;
	}
	
	protected abstract void loadNewPage(C oldContent, int pageNo, boolean forceUpdate);

	private boolean hasImageThread(C c) {
		for(int i=0; i<c.getThreadCount(); i++) {
			if(c.getTitle(i).smallImageUrl != null) return true;
		}
		
		return false;
	}
	
	protected abstract ThreadListConfig getConfig();
	
	private static SimpleDateFormat df = new SimpleDateFormat("yy/MM/dd");
	
	public void setClientContent(TabContent c) {
		content = (BaseBoardContent<C>)c;

		board = content.target;

		if(board != null) {
			pageText.setText(String.valueOf(board.pageNo));
		} else {
			pageText.setText("");
		}

//		browser.setBodyHtml("<" + SPAN + " id=\"thread:-1\"></" + SPAN + ">");
		
		StringWriter writer = new StringWriter();
		PrintWriter out = new PrintWriter(writer);


		boolean hasImageThread = hasImageThread(board);
		ThreadListConfig config = getConfig();

		if(config.showTotalPages) out.println("y[W: " + board.totalPages + " y[W<br>");
		
//		out.println("<table border=1 bordercolor=white bgcolor=black cellpadding=0 cellspacing=1><tbody>");
		out.println("<table border=1 cellspacing=0><tbody>");
		
		
		Vector<String> v = new Vector<String>();
		
		if(config.showNid) v.addElement("nid");
		if(config.showDispId) v.addElement("id");
		if(config.showImage && hasImageThread) v.addElement("image");
		if(config.showCountry) v.addElement("cn");
		if(config.showPoster) v.addElement("poster");
		if(config.showArticleNum) v.addElement("res");
		v.addElement("new");
		if(config.showViewNum) v.addElement("view");
		if(config.showTitle) v.addElement("title");
		if(config.showDate) v.addElement("date");

		String[] headerNames = new String[v.size()];
		v.toArray(headerNames);
/*		
		if(hasImageThread) {
			headerNames = new String[] {"id", "image", "cn",  "poster", "res", "new", "view", "title"};
		} else {
			headerNames = new String[] {"id", "cn",  "poster", "res", "new", "view", "title"};
		}
*/
		DefaultFormatter formatter = new DefaultFormatter();
		
		out.println("<tr>");
		for(int i=0; i<headerNames.length; i++) out.println("<td>" + headerNames[i] + "</td>");
		out.println("</tr");
		
		for(int i=0; i<board.getThreadCount(); i++) {
			NThread thread = board.getTitle(i);

			FilterType filterType = ViewNaver.instance.config.filterConfig.filters.get(new Poster(thread.poster, thread.country));
			if(filterType == null) filterType = FilterType.NOAction;

			if(filterType.equals(FilterType.Hidden)) {
				board.removeTitle(i);
				i--;
				continue;
			}

			String baseTitle = thread.title;

			if(baseTitle.trim().length() == 0) {
				baseTitle = "<i>(󗓂̃^Cg)</i>";
			}
			
			String title = "";
			for(int j=0; j<thread.depth; j++) title = title + "@";
			if(thread.depth > 0) title = title + "";
			title = title + baseTitle;

			
			if(filterType.equals(FilterType.NOAction)) {
				String newstr;
				if(thread.oldArticleNum == Integer.MIN_VALUE) {
					newstr = "n";
				} else if(thread.oldArticleNum == thread.articleNum) {
					newstr = "&nbsp;";
				} else { 
					newstr = String.valueOf(thread.articleNum - thread.oldArticleNum);
				}
				
				String dispIdText = thread.dispId;
				if(dispIdText.equals("")) dispIdText = "-";

				boolean isFavorite = ViewNaver.instance.config.favoritePersonConfig.persons.contains(new Poster(thread.poster, thread.country));
				
				out.print("<tr>");
				if(config.showNid)  out.print("<td>" + thread.nid + "</td>");
				if(config.showDispId) out.print("<td>" + dispIdText + "</td>");
				if(config.showImage) out.print(hasImageThread ? "<td><img src=\"" + thread.smallImageUrl + "\"></td>" : "");
				if(config.showCountry) out.print("<td>" + thread.country.toHtml() + "</td>");
				if(config.showPoster) out.print("<td><a href=\"poster:" + thread.nid + "\">" + formatter.formatPerson(thread.poster, isFavorite)  + "</td>");
				if(config.showArticleNum) out.print("<td>" + thread.articleNum + "</td>");
				out.print("<td>" + newstr + "</td>");
				if(config.showViewNum) out.print("<td>" + thread.viewNum + "</td>");
				if(config.showTitle) out.print("<td><a href=\"thread:" + thread.nid + "\">" +  title + "</a></td>");
				if(config.showDate) out.print("<td>" + df.format(thread.createDate) + "</td>");
				out.println("</tr>");

//				tableModel.addRow(new Object[] {
//						thread.dispId, thread.country, thread.poster, thread.articleNum, newstr, thread.viewNum, ViewNaver.htmlToStr(title)});
			} else {
				out.println("<tr><td colspan=" + v.size() + ">(̨ς)</td></tr>");
	

//				tableModel.addRow(new Object[] {"", "", "(̨ς)", "", "", ""});
			}
		}
		
		out.println("</tbody></table>");
		
		out.close();
		browser.setBodyHtml(writer.toString());
	}

	@Override public void refreshContent() {
		setClientContent(content);
	}
}
