/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.inaver.servlet;

import java.io.*;
import java.util.*;

import javax.servlet.*;
import javax.servlet.http.*;
import org.w3c.dom.*;
import org.apache.xerces.dom.*;
import org.apache.xerces.parsers.*;
import org.apache.xml.serialize.*;
import org.xml.sax.*;

import com.aibonware.inaver.*;
import com.aibonware.inaver.store.*;

@SuppressWarnings("serial")
public class NdesServlet extends HttpServlet {
//    private static SimpleDateFormat tf = new SimpleDateFormat("MM/dd HH:mm");
//    private static final SimpleDateFormat df = new SimpleDateFormat("yyyy.MM.dd");

    public NdesServlet() {}

    private static class NDESException extends Exception {
        public NDESException(String s) {
            super(s);
        }
        
        public NDESException(Exception e) {
            super(e);
        }
    }
    
    private abstract static class RequestPacket {
        public abstract Vector<Element> exec(Document doc, BackupStore store)  throws StoreException;
    }

    private static int getTagInt(Element tag, String tagName) {
        NodeList children = tag.getElementsByTagName(tagName);
        if(children.getLength() == 0) return -1;
        return Integer.parseInt(children.item(0).getTextContent());
    }

    private static class GetThreadListRequest extends RequestPacket {
        private Vector<String> requiredAttributes = new Vector<String>();
        private int BoardID;
        private int StartNID;
        private int EndNID;
        private int Number;
              
        public GetThreadListRequest(Element tag) {
            BoardID = getTagInt(tag, "BoardID");
            StartNID = getTagInt(tag, "StartNID");
            EndNID = getTagInt(tag, "EndNID");
            Number = getTagInt(tag, "Number");
            
            NodeList children = tag.getElementsByTagName("RequiredAttributes");
            if(children.getLength() == 0) return;
            Element requiredAttributesTag = (Element)children.item(0);
            
            for(int i=0; i<requiredAttributesTag.getChildNodes().getLength(); i++) {
                Element requiredAttributeTag = (Element)requiredAttributesTag.getChildNodes().item(i);
                requiredAttributes.addElement(requiredAttributeTag.getTextContent());
            }
        }
        
        public Vector<Element> exec(Document doc, BackupStore store) throws StoreException {
            return store.queryNDESThreadList(
                doc,
                requiredAttributes,
                BoardID,
                StartNID,
                EndNID,
                Number);
        }
   }

    private static class GetCommentListRequest extends RequestPacket {
        private Vector<String> requiredAttributes = new Vector<String>();
        private int BoardID;
        private int NID;
        private int StartCommentID;
        private int EndCommentID;
        private int Number;
        
        public GetCommentListRequest(Element tag) {
            BoardID = getTagInt(tag, "BoardID");
            NID = getTagInt(tag, "NID");
            StartCommentID = getTagInt(tag, "StartCommentID");
            EndCommentID = getTagInt(tag, "EndCommentID");
            Number = getTagInt(tag, "Number");
            
            NodeList children = tag.getElementsByTagName("RequiredAttributes");
            if(children.getLength() == 0) return;
            Element requiredAttributesTag = (Element)children.item(0);
            
            for(int i=0; i<requiredAttributesTag.getChildNodes().getLength(); i++) {
                Element requiredAttributeTag = (Element)requiredAttributesTag.getChildNodes().item(i);
                requiredAttributes.addElement(requiredAttributeTag.getTextContent());
            }
        }
        
        public Vector<Element> exec(Document doc, BackupStore store) throws StoreException {
            return store.queryNDESCommentList(
                    doc,
                    requiredAttributes,
                    BoardID,
                    NID,
                    StartCommentID,
                    EndCommentID,
                    Number);
            }
       }

    private static Vector<RequestPacket> parseRequestPacket(String text) throws SAXException, NDESException, IOException {
        Vector<RequestPacket> requests = new Vector<RequestPacket>();
        
        DOMParser parser = new DOMParser();
        parser.parse(new InputSource(new StringReader(text)));
        Document doc = parser.getDocument();
        
        Element packetTag = (Element)doc.getDocumentElement().getChildNodes().item(0);

        for(int i=0; i<packetTag.getChildNodes().getLength(); i++) {
            Element requestTag = (Element)packetTag.getChildNodes().item(i);
            if(!requestTag.getTagName().equals("Request")) continue;
            
            String requestType = requestTag.getAttribute("Type");
            
            if(requestType.equals("GetThreadList")) {
                RequestPacket request = new GetThreadListRequest(requestTag);
                requests.addElement(request);
            } 
            else if(requestType.equals("GetCommentList")) {
                RequestPacket request = new GetCommentListRequest(requestTag);
                requests.addElement(request);
            } 
            else {
                throw new NDESException("invalid request type: "+ requestType);
            }
            
            
        }
        
        return requests;
    }
    
    private static Element createResponsePacket(Document doc) {
        Element ndes = doc.createElement("ndes");

        Element packet = doc.createElement("Packet");
        packet.setAttribute("Type", "Response");
        packet.setAttribute("Country", "JP");
        
        ndes.appendChild(packet);
        doc.appendChild(ndes);
        
        return packet;
    }
    
    public static void main(PrintWriter out, String requestText) throws Exception {
        SQLStoreProvider storeProvider = INaver.getInstance().getStoreProvider();
        BackupStore store = storeProvider.openBackupStore();
        
        Vector<RequestPacket> requestPackets = parseRequestPacket(requestText);

        Document doc = new DocumentImpl();
        Element responsePacket = createResponsePacket(doc);

        for(int i=0; i<requestPackets.size(); i++) {
            Vector<Element> results = requestPackets.elementAt(i).exec(doc, store);
            
            for(int j=0; j<results.size(); j++) {
                responsePacket.appendChild(results.elementAt(j));
            }
        }

        Element status = doc.createElement("Status");
        status.setAttribute("Code", "0");
        responsePacket.appendChild(status);
        
        OutputFormat format = new OutputFormat(doc, "Shift_JIS", true);

        XMLSerializer serializer = new XMLSerializer(out, format);
        serializer.asDOMSerializer(); 
        serializer.serialize(doc.getDocumentElement());
        
        store.close();
    }
    
    public void doPost(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
        try {
           
            request.setCharacterEncoding("Shift_JIS");
            response.setContentType("text/xml; charset=Shift_JIS");

            PrintWriter out = response.getWriter();

            main(out, request.getParameter("request"));
            
            out.close();
        } 
        catch(Exception e) {
            Log.err(e);

            try {
                PrintWriter out = response.getWriter();
                out.println(e.getMessage());
                
                out.close();
            } catch(Exception e2) {
            }
        }
    }
}
