/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.inaver;

import java.util.*;
import java.text.*;
import com.aibonware.inaver.conv.*;
import com.aibonware.inaver.store.*;
import com.aibonware.inaver.task.*;
import com.aibonware.inaver.task.stat.*;
import com.aibonware.nvrlib.web.*;
import com.aibonware.nvrlib.parser.*;
import com.aibonware.nvrlib.*;
import com.aibonware.inaver.model.*;

/**
 * iNaverVXe{
 */
public class INaver implements AppTools {
	public static final String VERSION = "2.30";

	private static INaver instance = null; // ̃NX̗B̃CX^X

	private final SQLStoreProvider storeProvider;
	public final ParserProvider parserProvider;
	private Scheduler masterScheduler;
	private Scheduler backupScheduler;
	private Scheduler crawlerResumer;
	private WebSession currentWebSession; // }`ZbVɍ
	private NaverLib naverLib;

	public String serviceBaseUrl;
	public String naverBaseUrl;
	private Random rand;

    private ContentsConverter defaultConverter;
    private ContentsConverter nullConverter;
    private ContentsConverter nekoConverter;

	/**
	 * J̃RXgN^B
	 * etB[h̐ContentsConverteȑsB
	 */
	private INaver() {
		Env env = Env.getEnv();
		
		this.rand = new Random();
		this.serviceBaseUrl = env.serviceBaseUrl;
		this.naverBaseUrl = env.naverBaseUrl;

        defaultConverter = new StupidConverter();
        nullConverter = new NullConverter();
        nekoConverter = new NekoConverter();

		storeProvider = new SQLStoreProvider();
		parserProvider = new ParserProvider(this);
		crawlerResumer = new Scheduler(false);
		masterScheduler = createMasterScheduler();
		backupScheduler = createBackupScheduler();
		naverLib = new NaverLib(this);

        defaultConverter.init();
        nullConverter.init();
        nekoConverter.init();

		SimpleDateFormat df = new SimpleDateFormat("yyyy.MM.dd");

		if(!INaver.VERSION.equals(env.currentVersion)) {
			env.currentVersion = INaver.VERSION;
			env.currentVersionDate = df.format(new Date());

			try {
				env.flush();
			} catch(Exception e) {
				// NOP
			}
		}
	}

	/**
	 * iNaverT[rXJnB
	 * @param dbHost f[^x[XzXgB
	 * @param dbPass f[^x[XڑppX[hB
	 * @throws Exception 
	 */
	public static boolean start(String dbPass) {
		if(instance != null) return false;
		
		Env env = Env.getEnv();
		instance = new INaver();

		// f[^x[X̏
		try {
			instance.storeProvider.init(env.dbHost, dbPass);
		} catch(StoreException e) {
			Log.err(e);
		}
		
		return true;
	}

	public void startMasterScheduler() {
		crawlerResumer.start();
		masterScheduler.start();

		masterScheduler.addTask(new BoardCrawler(Env.getEnv().boardCrawlCount));
	}

	public void startBackupScheduler() {
		backupScheduler.start();
	}
	
	public void stopMasterScheduler() {
		crawlerResumer.exitScheduler();
		masterScheduler.exitScheduler();

		crawlerResumer = new Scheduler(false);
		masterScheduler = createMasterScheduler();
	}

	public void stopBackupScheduler() {
		backupScheduler.exitScheduler();
		backupScheduler = createBackupScheduler();
	}
	
	private Scheduler createMasterScheduler() {
		Scheduler newScheduler = new Scheduler(false);
		return newScheduler;
	}

	private Scheduler createBackupScheduler() {
		Scheduler newScheduler = new Scheduler(true);
		
//		StatWorkComposite dailyWorks = new StatWorkComposite(StatWorkComposite.PERIOD_DAY);
//		dailyWorks.addWork(new StatWorkArticlesPerThread(StatWork.PERIOD_DAY, 10));
//		dailyWorks.addWork(new StatWorkThreadsPerPoster(StatWork.PERIOD_DAY, 10));
//		dailyWorks.addWork(new StatWorkArticlesPerPoster(StatWork.PERIOD_DAY, 10));

		StatWorkComposite weeklyWorks = new StatWorkComposite(StatWorkComposite.PERIOD_WEEK);
//		weeklyWorks.addWork(new StatWorkArticlesPerThread(StatWork.PERIOD_WEEK, 50));
		weeklyWorks.addWork(new StatWorkPopularPoster(StatWork.PERIOD_WEEK, 50));
		weeklyWorks.addWork(new StatWorkThreadsPerPoster(StatWork.PERIOD_WEEK, 20));
		weeklyWorks.addWork(new StatWorkArticlesPerPoster(StatWork.PERIOD_WEEK, 150));

//		StatWorkComposite monthlyWorks = new StatWorkComposite(StatWorkComposite.PERIOD_MONTH);
//		monthlyWorks.addWork(new StatWorkArticlesPerThread(StatWork.PERIOD_MONTH, 100));
//		monthlyWorks.addWork(new StatWorkPopularPoster(StatWork.PERIOD_MONTH, 100));
//		monthlyWorks.addWork(new StatWorkThreadsPerPoster(StatWork.PERIOD_MONTH, 100));
//		monthlyWorks.addWork(new StatWorkArticlesPerPoster(StatWork.PERIOD_MONTH, 150));

//		newScheduler.addDailyWorks(dailyWorks);
//		newScheduler.addWeeklyWorks(new RoutineWorkIDList());
		newScheduler.addWeeklyWorks(weeklyWorks);
//		newScheduler.addMonthlyWorks(monthlyWorks);

		return newScheduler;
	}

	/**
	 * ̃NX̗B̃CX^X擾
	 * @return
	 */
	public static INaver getInstance() {
		return instance;
	}

	/**
	 * ContentsConverterCX^X擾
	 * @return
	 */
	public ContentsConverter getContentsConverter(int converterType) {
        switch(converterType) {
        case ContentsConverter.CONVERTER_TYPE_NEKO: return nekoConverter;
        case ContentsConverter.CONVERTER_TYPE_NULL: return nullConverter;
        default: return defaultConverter;
        }
    }

	/**
	 * StoreProviderCX^X擾B
	 * @return
	 */
	public SQLStoreProvider getStoreProvider() {
		return storeProvider;
	}

	/**
	 * WebProviderCX^X擾B
	 * @return
	 */
	public WebStore getWebStore() {
		return naverLib.getWebStore();
	}

	public ParserProvider getParserProvider() {
		return parserProvider;
	}

	public Scheduler getMasterScheduler() {
		return masterScheduler;
	}
	
	public Scheduler getBackupScheduler() {
		return backupScheduler;
	}
	
	public Scheduler getCrawlerResumer() {
		return crawlerResumer;
	}

	public static String getCountryHtml(int code) {
		if(code == 1) {
			return "<font color=\"red\"></font>";
		} else if(code == 2) { 
			return "<font color=\"blue\"></font>";
		} else return "";
	}

	public static String getShortCountryHtml(int code) {
		if(code == 1) {
			return "J";
		} else if(code == 2) { 
			return "K";
		} else return "";
	}

	public int getRand() {
		return rand.nextInt();
	}

	public volatile int loginUserCount = 0;
	
	public void setCurrentWebSession(WebSession session) {
		currentWebSession = session;
	}
	
	public WebSession getCurrentWebSession() {
		return currentWebSession;
	}
		
	public String getProxyHost() {
		return null;
	}
	
	public int getProxyPort() {
		return 0;
	}
	
	public String getNaverBaseUrl() {
		return naverBaseUrl;
	}
	
	private int taskCount = 0;
	
	private Vector<UserTask> userTasks = new Vector<UserTask>();
	
	public synchronized void addUserTask(UserTask task) {
		taskCount++;
		userTasks.addElement(task);
		task.taskId = taskCount;
	}

	public Vector<UserTask> getUserTask(UserConfig user) {
		Vector<UserTask> result = new Vector<UserTask>();
		for(int i=0; i<userTasks.size(); i++) {
			UserTask task = userTasks.elementAt(i);
			if(task.userId == user.userId) result.addElement(task);
		}
		
		return result;
	}

	public UserTask getUserTask(UserConfig user, int taskId) {
		for(int i=0; i<userTasks.size(); i++) {
			UserTask task = userTasks.elementAt(i);
			if(task.userId == user.userId && task.taskId == taskId) return task;
		}

		return null;
	}
	
	public synchronized void removeUserTask(UserTask task) {
		userTasks.removeElement(task);
	}

	public int getTaskCount() {
		return userTasks.size();
	}
}
