/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.inaver.task;

import java.text.*;
import java.util.*;
import java.util.prefs.*;
import com.aibonware.inaver.*;
import com.aibonware.inaver.store.BackupStore;

import EDU.oswego.cs.dl.util.concurrent.*;

/**
 * ̐ꂽRg̑}ev[gύX邽
 * EBhE > ݒ > Java > R[h > R[hƃRg
 */
public class Scheduler extends Thread {
	private static final SimpleDateFormat df = new SimpleDateFormat("yyyy.MM.dd");

	private TaskList taskList = new TaskList();
	private Task curTask = null;
	private Mutex curTaskLock = new Mutex();
	private Vector<RoutineWork> dailyWorks = new Vector<RoutineWork>();
	private Vector<RoutineWork> weeklyWorks = new Vector<RoutineWork>();
	private Vector<RoutineWork> monthlyWorks = new Vector<RoutineWork>();
	private int routineWorkCount = 0;
	private boolean hasRoutineWork = false;
	public volatile boolean nowStarted = false;
	volatile private long taskStartedTime = 0;
	volatile private boolean poemdb = false;
	
	public void addDailyWorks(RoutineWork work) {
		dailyWorks.addElement(work);
		hasRoutineWork = true;
	}

	public void addWeeklyWorks(RoutineWork work) {
		weeklyWorks.addElement(work);
		hasRoutineWork = true;
	}

	public void addMonthlyWorks(RoutineWork work) {
		monthlyWorks.addElement(work);
		hasRoutineWork = true;
	}

	public Scheduler(boolean poemdb) {
		this.poemdb = poemdb;
	}

	public void addTask(Task task) {
		try {
			if(task.stage != null) task.stage.addTask(task);
			taskList.put(task);
		} catch(InterruptedException e) {
		}
	}

	public void addUrgentTask(Task task) {
		try {
			curTaskLock.acquire();

			if(curTask != null && curTask instanceof ResumeableTask) {
				((ResumeableTask)curTask).suspend();

				taskList.putFirst(new Task[] {task, curTask});

				curTask.done.release();
				curTask = null;
			} else {
				taskList.putFirst(task);
			}

			curTaskLock.release();
		} 
		catch(InterruptedException e) {
		}
	}

	public void run() {
		try {
			nowStarted = true;
			
			for(;;) {
				Task task = (Task)taskList.take();

				if(task.stage != null && !task.stage.available) {
					task.done.release();
					task.stage.removeTask(task);
					if(task.stage.isEmpty()) task.stage.afterStage();
					continue;
				}

				curTaskLock.acquire();
				curTask = task;
				curTaskLock.release();

				if(!(task instanceof WaitTask)) taskStartedTime = System.currentTimeMillis();

				try {
					task.execute();
				} catch(RuntimeException e) {
					Log.err(e);
				}

				taskStartedTime = 0;
				
				task.done.release();

				if(task.stage != null) {
					task.stage.removeTask(task);
					if(task.stage.isEmpty()) task.stage.afterStage();
				} 

				curTaskLock.acquire();
				curTask = null;
				curTaskLock.release();

				if(task instanceof ExitTask) break;
				
				if(hasRoutineWork) {
					if((routineWorkCount % 10) == 0) {
						taskStartedTime = System.currentTimeMillis();
						processRoutineWork();
						taskStartedTime = 0;
					}
					routineWorkCount++;
				}
			}

		} catch(InterruptedException e) {
			Log.err("scheduler interrupted");
		}

		taskStartedTime = 0;
		nowStarted = false;
	}

	private void processRoutineWork() {
		// |GDB
		if(poemdb) {
			try {
				Preferences poemPref = Preferences.userRoot().node("/com/eromedayo/poem");
				boolean crawlPoem = poemPref.getBoolean("CrawlPoem", false);
	
				if(crawlPoem) {
					Log.info("start poemdb");
					String poemKeyword = poemPref.get("PoemKeyword", "|G");			

					BackupStore backup = INaver.getInstance().getStoreProvider().openBackupStore();
					backup.updatePoemDB(poemKeyword);
					backup.close();

					poemPref.putBoolean("CrawlPoem", false);
					poemPref.flush();
					
					Log.info("end poemdb");
				}
				
			} catch(Exception e) {
				Log.err(e);
			}
		}
		
		
		GregorianCalendar tmpCalendar = new GregorianCalendar();

		if(tmpCalendar.get(Calendar.HOUR_OF_DAY) < 4) return;

		GregorianCalendar currentCalendar = new GregorianCalendar();
		currentCalendar.clear();
		
		currentCalendar.set(
			tmpCalendar.get(Calendar.YEAR),
			tmpCalendar.get(Calendar.MONTH),
			tmpCalendar.get(Calendar. DAY_OF_MONTH));
			
		currentCalendar.setFirstDayOfWeek(Calendar.MONDAY);

		Date lastDate = Env.getEnv().getLastDailyWork();
		GregorianCalendar lastCalendarPlus6 = new GregorianCalendar();
		lastCalendarPlus6.setTime(lastDate);
		lastCalendarPlus6.setFirstDayOfWeek(Calendar.MONDAY);
		lastCalendarPlus6.add(Calendar.HOUR_OF_DAY, 5);

		currentCalendar.add(Calendar.DATE, -1);
		if(!currentCalendar.after(lastCalendarPlus6)) return;
		
		Date newDailyWorkDate = currentCalendar.getTime();

		try {
			boolean isNewDay = true;
			boolean isNewWeek = true;
			boolean isNewMonth = true;
			
			while(currentCalendar.after(lastCalendarPlus6)) {
				Date curDate = currentCalendar.getTime();
				Log.info("routine work " + df.format(curDate));
						
				for(int i=0; i<dailyWorks.size(); i++) {
					RoutineWork work = (RoutineWork)dailyWorks.elementAt(i);
					work.execute(curDate, isNewDay);
				}
				
				if(currentCalendar.get(Calendar.DAY_OF_WEEK) == currentCalendar.getActualMaximum(Calendar.DAY_OF_WEEK)) {
					for(int i=0; i<weeklyWorks.size(); i++) {
						RoutineWork work = (RoutineWork)weeklyWorks.elementAt(i);
						work.execute(curDate, isNewWeek);
					}
				}

				if(currentCalendar.get(Calendar.DAY_OF_MONTH) == currentCalendar.getActualMaximum(Calendar.DAY_OF_MONTH)) {
					for(int i=0; i<monthlyWorks.size(); i++) {
						RoutineWork work = (RoutineWork)monthlyWorks.elementAt(i);
						work.execute(curDate, isNewMonth);
					}
				}
				
				currentCalendar.add(Calendar.DATE, -1);

				isNewDay = false;
				isNewWeek = false;
				isNewMonth = false;
			}
		} 
		catch(RoutineWorkException e) {
			Log.err(e);
		}
			
		Env.getEnv().setLastDailyWork(newDailyWorkDate);
	}

	public void exitScheduler() {
		nowStarted = false;
		ExitTask exitTask = new ExitTask();
		addUrgentTask(exitTask);
		exitTask.waitDone();
	}
	
	public long getBusyTime() {
		if(taskStartedTime == 0) return 0;
		return System.currentTimeMillis() - taskStartedTime;
	}
}
