/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.inaver.conv;

import java.io.*;
import java.util.*;
import java.util.regex.*;
import org.w3c.dom.*;
import org.w3c.dom.html.*;
import org.xml.sax.*;
import org.cyberneko.html.parsers.*;
import com.aibonware.inaver.*;

/**
 * 
 */
public class NekoConverter implements ContentsConverter {
//	private static final int shortTextSize = 300;
	
	private static class Replacer {
		public Pattern pattern;
		public String replace;
		
		public Replacer(String pattern, String replace) {
			this.pattern =  Pattern.compile(pattern, Pattern.CASE_INSENSITIVE);
			this.replace = replace;
		}

		public String exec(String s) {
			Matcher matcher = pattern.matcher(s);
			return matcher.replaceAll(replace);
		}
	}
	
	private Vector<Replacer> replacers;
	private Replacer preTagConverter = new Replacer("\\n", "<br>\\n");
	
	private String escape(String s) {
		StringBuffer buf = new StringBuffer(s.length()*2);
		int len = s.length();

		for(int i=0; i<len; i++) {
			char c = s.charAt(i);
			
			switch(c) {
				case '"': buf.append("&quot;"); break;
				case '&': buf.append("&amp;"); break;
				case '<': buf.append("&lt;"); break;
				case '>': buf.append("&gt;"); break;
//				case ' ': buf.append("&nbsp;"); break;
				default: buf.append(c); 
			}
		}

		return buf.toString();
	}


	public void init() {
		replacers = new Vector<Replacer>();

		// sE^u͋󔒂ɒu
		replacers.addElement(new Replacer("[\\r\\n\\t" + ((char)0xa0) + "]", " "));
		
		// s^Osɒu
		replacers.addElement(new Replacer("<br\\s*/?>", "\n"));
		
		// A󔒂1ɒu
		replacers.addElement(new Replacer(" ++", " "));
		
		// As1ɒu
		replacers.addElement(new Replacer("\\n[ \\n]*", "\n"));
		
		// ss^Oɖ߂
		replacers.addElement(new Replacer("\\n", "<br>"));
	}
/*	
	private void dispNode(int indent, Node node) {
		for(int i=0; i<indent; i++) System.out.print("__");
		String fullName = node.getClass().getName();
		System.out.print(fullName.substring(fullName.lastIndexOf('.')+1));
		if(node instanceof Text) System.out.print(": " + node.getNodeValue());
		System.out.println();

		NodeList nodes = node.getChildNodes();

		for(int i=0; i<nodes.getLength(); i++) {
			dispNode(indent+1, nodes.item(i));
		}
	}
*/
 	private static class NekoContext {
//		public int curTextSize = 0;
		public Vector<String> texts = new Vector<String>();
//		private Vector<String> shortTexts = new Vector<String>();
//		public boolean shortFinished = false;
		
		NekoContext() {}

		public void add(String text) {
			texts.addElement(text);
//			if(!shortFinished) shortTexts.addElement(text);
		}
 	}
         
	public String convert(String imageUrl, String htmlContents) throws ConverterException {
		try {
			NekoContext ct = new NekoContext();
			
			DOMParser parser = new DOMParser();
			parser.setProperty("http://cyberneko.org/html/properties/default-encoding", "Shift_JIS");
			parser.parse(new InputSource(new StringReader(htmlContents)));

//			dispNode(0, parser.getDocument());

			convert(ct, parser.getDocument());
            String text = "";
            if(!imageUrl.equals("null")) {
                text = "<a href=\"" + imageUrl + "\">#img</a><br>\n";
            }
            
			text = text + merge(ct.texts);
			
			for(int i=0; i<replacers.size(); i++) {
				Replacer replacer = (Replacer)replacers.elementAt(i);
				text = replacer.exec(text);
			}
		
			return text;

		} catch(IOException e) {
			throw new ConverterException(e);
		} catch(SAXException e) {
			throw new ConverterException(e);
		}
	}
	
	private void processChildren(NekoContext ct, Node node) {
		Node child = node.getFirstChild();
		
		while(child != null) {
			convert(ct, child);
			child = child.getNextSibling();
		}
	}

	private void convert(NekoContext ct, Node node) {
		String nodeName = node.getNodeName();
		
		if(false) ;
		else if(node instanceof HTMLDocument) processChildren(ct, node);
		else if(node instanceof HTMLHtmlElement) processChildren(ct, node);
		else if(node instanceof HTMLBodyElement) processChildren(ct, node);
		else if(node instanceof HTMLBRElement) {
			ct.add("<br>");
		}
		else if(node instanceof HTMLTableElement) processChildren(ct, node);
		else if(node instanceof HTMLTableRowElement) {
			processChildren(ct, node);
			ct.add("<br>");
		}
		else if(node instanceof HTMLTableCellElement) {
			processChildren(ct, node);
			ct.add(" ");
		}
		else if(node instanceof HTMLHeadElement) ;
		else if(node instanceof HTMLDivElement) {
			processChildren(ct, node);
			ct.add("<br>");
		}
		else if(node instanceof HTMLParagraphElement) {
			processChildren(ct, node);
			ct.add("<br>");
		}
		else if(node instanceof HTMLFontElement) {
            String color = ((HTMLFontElement)node).getColor();
            if(color.equalsIgnoreCase("#ffffff") || color.equalsIgnoreCase("white")) {
                color = "gray";
            }
            
			ct.add("<font color=\"" + ((HTMLFontElement)node).getColor() + "\">");
			processChildren(ct, node);
            ct.add("</font>");
		}
		else if(node instanceof HTMLStyleElement) ;
		else if(node instanceof HTMLScriptElement) ;
		else if(nodeName.equals("CENTER")) {
			processChildren(ct, node);
		}

		else if(node instanceof HTMLAnchorElement) {
			HTMLAnchorElement a = (HTMLAnchorElement)node;
			ct.add("<a href=\"" + a.getHref() + "\">");
			processChildren(ct, node);
			ct.add("</a>");
		}
		else if(node instanceof HTMLImageElement) {
			HTMLImageElement img = (HTMLImageElement)node;
			String src = img.getSrc();
			
			// dynsrc̑Ή?

			ct.texts.addElement("<a href=\"" + src + "\">#img</a> ");
		}
		else if(node instanceof HTMLAppletElement) ;
		else if(node instanceof HTMLAreaElement) ;
		else if(node instanceof HTMLMapElement) ;
		else if(node instanceof HTMLBaseElement) ;
		else if(node instanceof HTMLBaseFontElement) ;
		else if(node instanceof HTMLButtonElement) {
			ct.texts.addElement("#button(");
			processChildren(ct, node);
			ct.texts.addElement(")");
		}
		else if(node instanceof HTMLDirectoryElement) {
			processChildren(ct, node);
		}
		else if(node instanceof HTMLLIElement) {
			processChildren(ct, node);
			ct.texts.addElement("<br>");
		}
		else if(node instanceof HTMLDListElement) {
			processChildren(ct, node);
		}
		else if(nodeName.equals("DT")) {
			processChildren(ct, node);
			ct.texts.addElement(" ");
		} else if(nodeName.equals("DD")) {
			processChildren(ct, node);
			ct.texts.addElement("<br>");
		}
		else if(node instanceof HTMLFieldSetElement) {
			HTMLFieldSetElement fs = (HTMLFieldSetElement)node;
			if(fs.getTitle() != null) ct.texts.addElement(fs.getTitle());
			ct.texts.addElement("<br>");
			processChildren(ct, node);
		}
		else if(node instanceof HTMLFormElement) {
			processChildren(ct, node);
		}
		else if(node instanceof HTMLFrameElement) {
			HTMLFrameElement e = (HTMLFrameElement)node;
			ct.texts.addElement("<a href=\"" + e.getSrc() + "\">#frame</a><br>");
		}
		else if(node instanceof HTMLFrameSetElement) {
			processChildren(ct, node);
		}
		else if(node instanceof HTMLHeadingElement) {
			processChildren(ct, node);
			ct.texts.addElement("<br>");
		}
		else if(node instanceof HTMLHRElement) {
			ct.texts.addElement("<hr>");
		}
		else if(node instanceof HTMLIFrameElement) {
			HTMLIFrameElement e = (HTMLIFrameElement)node;
			ct.texts.addElement("<a href=\"" + e.getSrc() + "\">#iframe</a><br>");
		}
		else if(node instanceof HTMLInputElement) {
			ct.texts.addElement("#input");
		}
		else if(node instanceof HTMLIsIndexElement) {
			ct.texts.addElement("#isindex");
		}
		else if(node instanceof HTMLTableColElement) ;
		else if(node instanceof HTMLLabelElement) {
			processChildren(ct, node);			
		}
		else if(node instanceof HTMLLegendElement) {
			processChildren(ct, node);
			ct.texts.addElement("<br>");
		}
		else if(node instanceof HTMLLinkElement) ;
		else if(node instanceof HTMLMenuElement) {
			processChildren(ct, node);
			ct.texts.addElement("<br>");
		}
		else if(node instanceof HTMLMetaElement) ;
		else if(node instanceof HTMLModElement) {
			ct.texts.addElement("#" + nodeName + "(");
			processChildren(ct, node);
			ct.texts.addElement(")");
		}
		else if(node instanceof HTMLObjectElement) {
			HTMLObjectElement e = (HTMLObjectElement)node;
			if(e.getData() != null && !e.getData().equals("")) {
				ct.texts.addElement("<a href=\"" + e.getData() + "\">#object</a><br>");
			} else {
				ct.texts.addElement("#object");
			}
		}
		else if(node instanceof HTMLOListElement) {
			processChildren(ct, node);
			ct.texts.addElement("<br>");
		}
		else if(node instanceof HTMLOptGroupElement) {
			processChildren(ct, node);
		}
		else if(node instanceof HTMLOptionElement) {
			processChildren(ct, node);
			ct.texts.addElement("<br>");
		}
		else if(node instanceof HTMLParamElement) ;
		else if(node instanceof HTMLPreElement) {
			NekoContext preContext = new NekoContext();
			processChildren(preContext, node);
			String preTexts = merge(preContext.texts);
			preTexts = preTagConverter.exec(preTexts);
			ct.texts.addElement(escape(preTexts));
		}
		else if(nodeName.equals("BLOCKQUOTE")) {
			processChildren(ct, node);
			ct.texts.addElement("<br>");
		}
		else if(node instanceof HTMLQuoteElement) {
			processChildren(ct, node);
		}
		else if(node instanceof HTMLSelectElement) {
			processChildren(ct, node);
		}
		else if(node instanceof HTMLTableCaptionElement) {
			processChildren(ct, node);
			ct.texts.addElement("<br>");
		}
		else if(node instanceof HTMLTableSectionElement) {
			processChildren(ct, node);
		}
		else if(node instanceof HTMLTextAreaElement) {
			processChildren(ct, node);
			ct.texts.addElement("<br>");
		}
		else if(node instanceof HTMLTitleElement) ;
		else if(node instanceof HTMLUListElement ) {
			processChildren(ct, node);
		}
		else if(node instanceof Comment) ;
		else if(node instanceof Text) {
			String text = escape(node.getNodeValue());
/*			int newCurTextSize = ct.curTextSize + text.length();

			if(ct.curTextSize < shortTextSize && newCurTextSize >= shortTextSize) {
				int divPos = shortTextSize - ct.curTextSize; 
				text = text.substring(0,  divPos) + ((char)0xffff) + text.substring(divPos);
			}
			
			ct.curTextSize += newCurTextSize;
*/
            ct.texts.addElement(text);
		} else if(nodeName.equals("B")) {
			ct.texts.addElement("<b>");
			processChildren(ct, node);
			ct.texts.addElement("</b>");
		} else if(nodeName.equals("SPAN")) {
			processChildren(ct, node);
		} else if(nodeName.equals("TT")) {
			processChildren(ct, node);
		} else if(nodeName.equals("I")) {
			processChildren(ct, node);
		} else if(nodeName.equals("U")) {
			processChildren(ct, node);
		} else if(nodeName.equals("SMALL")) {
			processChildren(ct, node);
		} else if(nodeName.equals("NOBR")) {
			processChildren(ct, node);
		} else if(nodeName.equals("NOSCRIPT")) {
			processChildren(ct, node);
		} else if(nodeName.equals("STRONG")) {
			processChildren(ct, node);
		} else if(nodeName.equals("NOFRAMES")) {
			processChildren(ct, node);
        } else if(nodeName.equals("BIG")) {
            processChildren(ct, node);
        } else if(nodeName.equals("MARQUEE")) {
            processChildren(ct, node);
        } else if(nodeName.equals("EMBED")) {
            ct.texts.addElement("#embed<br>");
		}else {
			ct.texts.addElement("#unknown_tag(" + nodeName + ")");
			Log.err("unknown tag: " + nodeName);
		} 
	}

	private String merge(Vector<String> texts) {
		int len = texts.size();
		StringBuffer buf = new StringBuffer();
		
		for(int i=0; i<len; i++) {
			String s = (String)texts.elementAt(i);
			buf.append(s);			
		}
		
		return buf.toString();
	}

	public String getShortContents(
			int desiredLineNum,
			int desiredCharNum,
			String contents,
			String link) {
        return contents;
	}
}
