/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.inaver;

import java.text.*;
import java.util.*;
import java.util.prefs.*;

/**
 * 
 */
public class Env {
	public String rootDir;
	public String serviceBaseUrl;
	public String naverBaseUrl;
	public String dbHost;
	public String currentVersion;
	public String currentVersionDate;
	public String announcement;
    public String enableUserConfig;
    public int trafficLimit;
    public String buildNo;
    public String information;
    public int boardCrawlCount;
    public int searchLimitTime;
    public int newArtNum;
    public int uniqueLimit;
    public String rlistDir;
    public String rankDir;
   
    private String mobileUserAgentsText;
    private String[] mobileUserAgentsArray;

    private static final SimpleDateFormat df = new SimpleDateFormat("yyyy.MM.dd");
	private static Env instance = null;

	private Preferences pref;
	
	private Env() {
		pref = Preferences.userRoot().node("/com/aibonware/inaver");
		rootDir = pref.get("RootDir", "C:\\work\\source\\RootDir\\");
		serviceBaseUrl = pref.get("ServiceBaseUrl", "http://localhost/indev/");
		naverBaseUrl = pref.get("NaverBaseUrl", "http://bbs.enjoykorea.naver.co.jp/");
		dbHost = pref.get("DbHost", "localhost");
		currentVersion = pref.get("CurrentVersion", INaver.VERSION);
		currentVersionDate = pref.get("CurrentVersionDate", df.format(new Date()));
		announcement = pref.get("Announcement", "");
        enableUserConfig = pref.get("EnableUserConfig", "True");
        buildNo = pref.get("BuildNo", "");
        information = pref.get("Information", "");
        String sTrafficLimit = pref.get("TrafficLimit", "0");
        boardCrawlCount = pref.getInt("BoardCrawlCount", 0);
        searchLimitTime = pref.getInt("SearchLimitTime", 180);
        newArtNum = pref.getInt("NewArtNum", 200);
        uniqueLimit = pref.getInt("UniqueLimit", 25);
        rlistDir = pref.get("ArticleListDir", "C:\\work\\source\\RootDir\\rlist\\");
        rankDir = pref.get("RankingDir", "C:\\work\\source\\RootDir\\ranking\\");

        mobileUserAgentsText = pref.get("MobileUserAgents", "docomo");
        convertMobileAgentsTextToArray();
	    
        try {
            trafficLimit = Integer.parseInt(sTrafficLimit);
        } catch(NumberFormatException e) {
            trafficLimit = -1;
        }
	}

	private void convertMobileAgentsTextToArray() {
	       Vector<String> vAgents = new Vector<String>();
	        StringTokenizer tokens = new StringTokenizer(mobileUserAgentsText, " ");
	        
	        while(tokens.hasMoreTokens()) vAgents.addElement(tokens.nextToken());

	        mobileUserAgentsArray = new String[vAgents.size()];
	        vAgents.toArray(mobileUserAgentsArray);
	}
	
	public void setMobileUserAgentsText(String text) {
		this.mobileUserAgentsText = text;
		convertMobileAgentsTextToArray();
	}
	
	public String getMobileUserAgentsText() {
		return mobileUserAgentsText;
	}

	public String[] getMobileUserAgentsArray() {
		return mobileUserAgentsArray;
	}
	
	public void flush() throws BackingStoreException {
		pref.put("RootDir", rootDir);
		pref.put("ServiceBaseUrl", serviceBaseUrl);
		pref.put("NaverBaseUrl", naverBaseUrl);
		pref.put("DbHost", dbHost);
		pref.put("CurrentVersion", currentVersion);
		pref.put("CurrentVersionDate", currentVersionDate);
		pref.put("Announcement", announcement);
        pref.put("EnableUserConfig", enableUserConfig);
        pref.put("TrafficLimit", String.valueOf(trafficLimit));
        pref.put("BuildNo", buildNo);
        pref.put("Information", information);
        pref.putInt("BoardCrawlCount", boardCrawlCount);
        pref.putInt("SearchLimitTime", searchLimitTime);
        pref.putInt("NewArtNum", newArtNum);
        pref.putInt("UniqueLimit", uniqueLimit);
        pref.put("ArticleListDir", rlistDir);
        pref.put("RankingDir", rankDir);
        pref.put("MobileUserAgents", mobileUserAgentsText);
        
		pref.flush();
	}
	
	public static synchronized Env getEnv() {
		if(instance == null) instance = new Env();
		return instance;
	}
	
	public Date getLastDailyWork() {
		try {
			String sCurDate = df.format(new Date());
			
			return df.parse(pref.get("LastDailyWork", sCurDate));
		} catch(ParseException e) {
			Log.err(e);
			return new Date();
		}
	}

	public void setLastDailyWork(Date newDate) {
		try {
			String sCurDate = df.format(newDate);

			pref.put("LastDailyWork", sCurDate);
			pref.flush();

		} catch(BackingStoreException e) {
			Log.err(e);
		}
	}
}
