/*
 * Copyright (c) 2007, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.enjoybase.conv;

import java.text.*;
import java.sql.*;
import java.io.*;
import com.enjoybase.*;
import com.enjoybase.model.*;

public class ManifestConverter {
	public String html = "";
	public String text = "";
	private final Storage store;
	private final String editor;

	public ManifestConverter(Storage store, String text, String editor) {
		this.text = text.trim();
		this.store = store;
		this.editor = editor;
	}

	public void convert() throws ConverterException {
		BufferedReader reader = new BufferedReader(new StringReader(text));
		StringWriter textStringWriter = new StringWriter();
		StringWriter htmlStringWriter = new StringWriter();

		PrintWriter textWriter = new PrintWriter(textStringWriter);
		PrintWriter htmlWriter = new PrintWriter(htmlStringWriter);

		String line;
		int lineno = 1;

		try {
			while((line = reader.readLine()) != null) {
				if(line.length() != 0) {
					parseLine(lineno, line, textWriter, htmlWriter);
				} else {
					textWriter.println();
					htmlWriter.println("<br>");
				}

				lineno++;
			}
		} catch(IOException e) {
			throw new ConverterException(e);
		}

		textWriter.close();
		htmlWriter.close();

		text = textStringWriter.toString();
		html = htmlStringWriter.toString();
	}

	private void parseLine(int lineno, String line, PrintWriter textWriter, PrintWriter htmlWriter) throws ConverterException {
		int indent = 0;
		
		// Cfg̐[߂
		while(indent < line.length() && line.charAt(indent) == '\t') {
			indent++;
		}
		
		line = line.trim();
		if(line.length() == 0) return;

		// CfgHTMLo
		for(int i=0; i<indent; i++) {
			textWriter.print("\t");
			htmlWriter.print("&nbsp;&nbsp;&nbsp;&nbsp;");
		}

		if(line.length() >= 2 && line.substring(0, 2).equalsIgnoreCase("N:")) {
			try {
				// VKgsbN̍쐬
				Topic topic = store.createNewTopic(editor);

				String title = line.substring(2);
				textWriter.print(topic.id + ":" + title);
				htmlWriter.print(createTopicLink(title, topic.id));
			} catch(SQLException e) {
				throw new ConverterException(e);
			} catch(ParseException e) {
				throw new ConverterException(e);
			}
		} else {
			int colon = line.indexOf(':');

			if(colon == -1) {
				// JeS
				textWriter.print(line);
				htmlWriter.print("<b>" + Enjoybase.toHtmlText(line) + "</b>");
			} else {
				// gsbN
				try {
					int id = Integer.parseInt(line.substring(0, colon));

					String title = line.substring(colon+1);
					textWriter.print(id + ":" + title);
					htmlWriter.print(createTopicLink(title, id));
				} catch(NumberFormatException e) {
					throw new ConverterException(lineno + "sځF'" + line.substring(0, colon) + "'͕sȃgsbNIDł");
				}
			}
		}
		
		textWriter.println();
		htmlWriter.println("<br>");
	}
	
	private String createTopicLink(String title, int id) {
		return "<a href=\"./topic?id=" + id + "\">" + Enjoybase.toHtmlText(title) + "</a>";
	}
	
	public static void main(String args[]) throws Exception {
		String text = 
			"test\n" +
			"\taaa\n" + 
			"\n" + 
			"\tbbb\n";
		
		ManifestConverter conv = new ManifestConverter(null, text, "test");
		conv.convert();
		
		System.out.println(conv.html);
	}
}
