/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.idlnvr.servlet;

import java.io.*;
import java.util.*;
import javax.servlet.http.*;

import com.aibonware.idlnvr.*;

@SuppressWarnings("serial")
public class MainServlet extends BaseServlet {
	protected int getRequiredUserLevel() {
		return User.USER_LEVEL_ANONYMOUS;
	}

	public static class AllPersonDrawer implements PersonWalker {
		private final User user;
		private final PrintWriter out;
		private final Database db;
	
		public AllPersonDrawer(PrintWriter out, User user, Database db) {
			this.out = out;
			this.user = user;
			this.db = db;
		}
	
		public void init() {}

		public void walk(Person person) throws Exception {
			if(!person.visible) return;

			out.println("<a name=\"" +  person.personId + "\"></a>");
			out.print(IdlNvr.getCountryString(person.country) + " ");
			out.println("<b>" + person.name + "</b>");

			if(!user.isGuest) {
				out.println(" <a href=\"./detail?id=" + person.name + "&c=" + person.country + "\">ڍ</a>");
			}

			if(!user.isGuest && user.allowEdit && (user.name.equals(person.editorName) || person.freeEdit)) {
				out.println(" <a href=\"./edit?id=" + person.name + "&c=" + person.country + "\">ҏW</a>");
			}

			if(person.profileLevel == User.PROFILE_LEVEL_ALL
			 || (person.profileLevel == User.PROFILE_LEVEL_LOGIN_USER && !user.isGuest)
			 || (person.profileLevel == User.PROFILE_LEVEL_EDITOR && user.allowEdit)) {
				out.println("<a href=\"./profile?id=" + person.editorId + "\"></a>");
			}

			out.print("&nbsp;&nbsp;");
			Vector<Board> boards = db.getPersonBoard(person.personId);
			String delim = "";

			for(int j=0; j<boards.size(); j++) {
				out.print(delim + boards.elementAt(j).name);
				delim = ", ";
			}
			out.println("<br>");

			if(person.showSubId) {
				String aliasText = "";
				String aliasTextDelim = "";

				if(person.aliasNames.size() > 0) {
					for(int i=0; i<person.aliasNames.size(); i++) {
						aliasText = aliasText + aliasTextDelim + person.aliasNames.elementAt(i);
						aliasTextDelim = " ";
					}

					out.println("TuIDXg: <b>" + aliasText + "</b><br>");
				}
			}

			out.println(person.html);
			
			out.println("<hr>");
		}

		public void exit() {}
	}
	
	public static class SummaryPersonDrawer implements PersonWalker {
		private final User user;
		private final PrintWriter out;
//		private final Database db;
		private int line = 0;
		
		public SummaryPersonDrawer(PrintWriter out, User user, Database db) {
			this.out = out;
			this.user = user;
//			this.db = db;
		}
		
		public void init() {
			out.println("<table width=\"100%\" border=\"0\">");
		}
		
		public void walk(Person person) throws Exception {
			if(!person.visible) return;

			if((line % 2) == 1) out.print("<tr>");
			else out.print("<tr bgcolor=\"#F0F0F0\">");
			
			out.print("<td><a name=\"" +  person.personId + "\"></a>");
			out.print(IdlNvr.getCountryString(person.country) + "</td>");
			out.print("<td><b><a href=\"./detail?id=" + person.name + "&c=" + person.country + "\"><font color=black>" + person.name + "</font></a></b></td>");

			out.print("<td>");
			out.println(person.shortHtml);

			if(!user.isGuest && user.allowEdit && (user.name.equals(person.editorName) || person.freeEdit)) {
				out.println(" <a href=\"./edit?id=" + person.name + "&c=" + person.country + "\">ҏW</a>");
			}

			out.println("</td></tr>");
			line++;
		}
		
		public void exit() {
			out.println("</table>");
			out.println("<hr>");
		}
	}

	protected void dispContents(HttpServletRequest request, HttpServletResponse response, PrintWriter out, Database db, User user) throws Exception {
		int boardId = IdlNvr.getInt(request.getParameter("b"));
		int country = IdlNvr.getInt(request.getParameter("c"));
		String summaryParam = request.getParameter("s");
		if(summaryParam == null) summaryParam = request.getParameter(" s");
		
		boolean summary = (IdlNvr.getInt(summaryParam) <= 0);

		PersonWalker walker;

		if(summary) walker = new SummaryPersonDrawer(out, user, db);
		else walker = new AllPersonDrawer(out, user, db);

		db.walkPersons(boardId, country, walker, summary);

		out.println("Powered by idlNVR " + IdlNvr.VERSION);
	}
}
