#ifndef	   MMAP_FILE_H_INCLUDED
#define	   MMAP_FILE_H_INCLUDED

// Author:		H. Shimora
// Created:		Jan  8 2004
// Version:		0.00

///-----------------------------------------------
/// Change Log:
///-----------------------------------------------
// version 0.00  Jan  8 2004    base version.
//
//

// for mmap, msync, munmap
#include  <sys/types.h>
#include  <sys/mman.h>

// for open, close
#include  <sys/types.h>
#include  <sys/stat.h>
#include  <fcntl.h>

// for write, lseek
#include  <unistd.h>

// for stat
#include  <sys/stat.h>


#include  <string>
#include  <cstdio>


class  MMap_File
{
private:
	int	io_flag;

	int	fd;

	char *	mapped_memory;
	off_t	mapped_offset;
	size_t	mapped_size_;

public:
	MMap_File( const std::string &  path ,
		   int  io_flag /* O_RDONLY or O_WRONLY or O_RDWR */ )
		: io_flag( io_flag ) ,
		  fd( -1 ) ,
		  mapped_memory( static_cast<char *>( 0 ) ) ,
		  mapped_offset( 0 ) ,
		  mapped_size_( 0 )
	{
		int	flag = io_flag;

		if ( (io_flag & O_WRONLY) == O_WRONLY
		  || (io_flag & O_RDWR  ) == O_RDWR )
		{
			flag |= (O_CREAT | O_TRUNC);
		}

		if ( (this -> fd = ::open( path.c_str() ,
					   flag ,
					   0644 )) == -1 )
		{
			std::perror( "open" );
		}
	}

	~MMap_File()
	{
		this -> detach();

		this -> close();
	}

	bool	valid() const
	{
		return( fd != -1 );
	}

	volatile char *	memory() const
	{
		return( this -> mapped_memory );
	}

	size_t	mapped_size() const
	{
		return( this -> mapped_size_ );
	}

	volatile const char &	operator [] ( size_t  indx ) const
	{
		return( *(this -> mapped_memory + indx) );
	}


	bool	attach( off_t  offset ,  size_t  new_size )
	{
		int	flag = 0x0;

		if ( (io_flag & O_WRONLY) == O_WRONLY
		  || (io_flag & O_RDWR  ) == O_RDWR )
		{
			flag |= PROT_WRITE;
		}

		if ( (io_flag & O_RDONLY) == O_RDONLY
		  || (io_flag & O_RDWR  ) == O_RDWR )
		{
			flag |= PROT_READ;
		}

		if ( (this -> mapped_memory
		      = static_cast<char *>
			( ::mmap( static_cast<void *>(0) ,
				  new_size ,
				  flag ,
				  MAP_SHARED ,
				  this -> fd ,
				  offset ) ) ) == MAP_FAILED )
		{
			std::perror( "mmap" );

			this -> mapped_size_ = 0;

			return( false );
		}

		this -> mapped_size_  = new_size;
		this -> mapped_offset = offset;

		return( true );
	}

	bool	attach_whole_file()
	{
		struct stat	file_info;

		if ( ::fstat( fd , &file_info ) == -1 )
		{
			std::perror( "fstat" );

			return( false );
		}

		return( this -> attach( 0 , file_info.st_size ) );
	}


	void	detach( bool  do_sync = true )
	{
		if ( this -> mapped_memory == static_cast<char *>( 0 ) )
		{
			return;
		}

		if ( do_sync )
		{
			if ( ::msync( this -> mapped_memory ,
				      0 /* all */ , MS_SYNC ) == -1 )
			{
				std::perror( "msync" );
			}
		}

		if ( this -> mapped_size_ != 0 )
		{
			if ( ::munmap( this -> mapped_memory ,
				       this -> mapped_size_ ) == -1 )
			{
				std::perror( "munmap" );
			}
		}

		this -> mapped_memory = static_cast<char *>( 0 );
		this -> mapped_size_  = 0;
	}



	bool	guarantee_least_file_size( size_t  least_size )
	{
		if ( this -> fd == -1 )
		{
			return( false );
		}


		if ( ::lseek( this -> fd , least_size - 1 , SEEK_SET ) == -1 )
		{
			std::perror( "lseek" );

			return( false );
		}

		const char	zero = 0;

		if ( ::write( this -> fd , &zero , 1 ) == -1 )
		{
			std::perror( "write" );

			return( false );
		}

		return( true );
	}


	bool	close()
	{
		this -> detach();

		if ( this -> fd == -1 )
		{
			return( true );
		}

		if ( ::close( this -> fd ) == -1 )
		{
			std::perror( "close" );

			return( false );
		}

		return( true );
	}
};


#endif	/* MMAP_FILE_H_INCLUDED */
