#include  "d2_polygon.h"
#include  "d2_segment.h"
#include  "float_traits.h"
#include  <iostream>
#include  <cstdlib>

// XXX
#include  "d2_straight_line.h"


D2_Polygon::D2_Polygon()
	: vertex()
{
}

D2_Polygon::D2_Polygon( const std::vector<D2_Vector> &  v )
	: vertex( v )
{
}

D2_Polygon::~D2_Polygon()
{
}

void   D2_Polygon::set( const std::vector<D2_Vector> &  v )
{
	this -> vertex = v;
}

void   D2_Polygon::set_empty()
{
	this -> vertex.clear();
}

void   D2_Polygon::add( const D2_Vector &  p )
{
	this -> vertex.push_back( p );
}

const std::vector<D2_Vector> &  D2_Polygon::points() const
{
	return( this -> vertex );
}

std::vector<D2_Vector> &  D2_Polygon::points()
{
	return( this -> vertex );
}

void   D2_Polygon::get_bounding_box( D2_Rectangle_Region *  r ) const
{
	r -> clear();
	r -> extend_region( this -> vertex );
}

D2_Vector  D2_Polygon::xy_center() const
{
	FLOAT	x_min = float_traits<FLOAT>::max();
	FLOAT	x_max = float_traits<FLOAT>::min();
	FLOAT	y_min = float_traits<FLOAT>::max();
	FLOAT	y_max = float_traits<FLOAT>::min();

	for ( size_t  i = 0  ;  i < this -> vertex.size()  ;  ++ i )
	{
		const D2_Vector &	p = this -> vertex[i];

		if ( p.x() > x_max )
		{
			x_max = p.x();
		}

		if ( p.x() < x_min )
		{
			x_min = p.x();
		}

		if ( p.y() > y_max )
		{
			y_max = p.y();
		}

		if ( p.y() < y_min )
		{
			y_min = p.y();
		}
	}

	return( D2_Vector( (x_min + x_max) / FLOAT(2.0) ,
			   (y_min + y_max) / FLOAT(2.0) ) );
}


bool   D2_Polygon::intersect( const D2_Vector &  p ,
			      bool  allow_on_segment ) const
{
#if 0
#define	D2_POLYGON_DEBUG_PRINT
#endif

	if ( this -> vertex.empty() )
	{
		return( false );
	}
	else if ( this -> vertex.size() == 1 )
	{
		return( allow_on_segment && (this -> vertex[0] == p) );
	}


	D2_Rectangle_Region	r;
	this -> get_bounding_box( &r );

	if ( ! r.in_region( p ) )
	{
		return( false );
	}


	//
	// make virtual half line
	//
	D2_Segment	line( p ,
			      D2_Vector( p.x()
					 + ((r.max_x() - r.min_x()
					     + r.max_y() - r.min_y())
					    + (this -> vertex[0] - p).r())
							      * FLOAT(3.0) ,
					 p.y() ) );

#ifdef D2_POLYGON_DEBUG_PRINT
	std::cerr << "line = [" << line.point_0() << "]"
		  <<    " to [" << line.point_1() << "]" << std::endl;
#endif


	//
	// check intersection with all segments
	//
	bool	inside = false;
	FLOAT	min_line_x = r.max_x() + FLOAT(1.0);

	for ( size_t  i = 0  ;  i < this -> vertex.size()  ;  i ++ )
	{
		size_t	p1_index = i + 1;

		if ( p1_index >= this -> vertex.size() )
		{
			p1_index = 0;
		}

		const D2_Vector	p0 = this -> vertex[i];
		const D2_Vector	p1 = this -> vertex[p1_index];

#ifdef D2_POLYGON_DEBUG_PRINT
		std::cerr << "checking [" << p0 << "] to [" << p1 << "]"
			  << std::endl;
#endif
		if ( ! allow_on_segment )
		{
			if ( D2_Segment( p0 , p1 ).on_segment( p ) )
			{
				return( false );
			}
		}

		if ( allow_on_segment && p == p0 )
		{
#ifdef D2_POLYGON_DEBUG_PRINT
			std::cerr << "  equals to vertex" << std::endl;
#endif
			return( true );
		}

		if ( line.intersect( D2_Segment( p0 , p1 ) ) )
		{
		#ifdef D2_POLYGON_DEBUG_PRINT
			std::cerr << "  intersect" << std::endl;
		#endif

			if ( p0.y() == p.y()
			  || p1.y() == p.y() )
			{
				if ( p0.y() == p.y() )
				{
					if ( p0.x() < min_line_x )
					{
						min_line_x = p0.x();
					}
				}

				if ( p1.y() == p.y() )
				{
					if ( p1.x() < min_line_x )
					{
						min_line_x = p1.x();
					}
				}


				if ( p0.y() == p1.y() )
				{
				#ifdef D2_POLYGON_DEBUG_PRINT
					std::cerr
					  << "  2 points on the line, skip"
					  << std::endl;
				#endif

					continue;
				}
				else if ( p0.y() < p.y()
				       || p1.y() < p.y() )
				{
				#ifdef D2_POLYGON_DEBUG_PRINT
					std::cerr
					  << "  top point on the line, skip"
					  << std::endl;
				#endif

					continue;
				}
				else // bottom point on the line
				{
				#ifdef D2_POLYGON_DEBUG_PRINT
					std::cerr
					  << "  bottom point on the line"
					  << std::endl;
				#endif

					// no operation, don't skip
				}
			}

		#ifdef D2_POLYGON_DEBUG_PRINT
			std::cerr << "  ! changed" << std::endl;
		#endif

			inside = (! inside);
		}
		else
		{
		#ifdef D2_POLYGON_DEBUG_PRINT
			std::cerr << "  not intersect" << std::endl;
		#endif
		}
	}

#ifdef D2_POLYGON_DEBUG_PRINT
	std::cerr << "inside = " << (inside ? "true" : "false") << std::endl;
#endif

	return( inside );
}

FLOAT  D2_Polygon::get_distance( const D2_Vector &  p ,
				 bool  check_as_plane ) const
{
	if ( this -> points().size() == 1 )
	{
		return( (this -> points()[0] - p).r() );
	}

	if ( check_as_plane && this -> intersect( p ) )
	{
		return( 0.0 );
	}

	FLOAT	min_dist = + float_traits<FLOAT>::max();

	for ( size_t  i = 0  ;  i + 1 < this -> points().size()  ;  i ++ )
	{
		D2_Segment	seg( this -> points()[i] ,
				     this -> points()[i + 1] );

		FLOAT	d = seg.distance( p );

		if ( d < min_dist )
		{
			min_dist = d;
		}
	}

	if ( this -> points().size() >= 3 )
	{
		D2_Segment	seg( *(this -> points().rbegin()) ,
				     *(this -> points().begin()) );

		FLOAT	d = seg.distance( p );

		if ( d < min_dist )
		{
			min_dist = d;
		}
	}

	// if this -> size() == 0, returns huge value

	return( min_dist );
}

FLOAT  D2_Polygon::area() const
{
	return( float_traits<FLOAT>::fabs( this -> signed_area_2()
                                           / FLOAT(2.0) ) );
}

FLOAT  D2_Polygon::signed_area_2() const
{
	if ( vertex.size() < 3 )
	{
		return( 0.0 );
	}

	FLOAT	sum = 0.0;
	for ( size_t  i = 0  ;  i < vertex.size()  ;  i ++ )
	{
		size_t	n = i + 1;
		if ( n == vertex.size() )
		{
			n = 0;
		}

		sum += (vertex[i].x() * vertex[n].y()
			- vertex[n].x() * vertex[i].y());
	}

	return( sum );
}

bool   D2_Polygon::is_reverse_clockwise() const
{
	return( this -> signed_area_2() > 0.0 );
}

bool   D2_Polygon::is_clockwise() const
{
	return( this -> signed_area_2() < 0.0 );
}


template<class Predicate>
void    scissor_with_line( const Predicate &  in_region ,
			   const std::vector<D2_Vector> &  points ,
			   std::vector<D2_Vector> *  new_points ,
			   const D2_Straight_Line &  line )
{
	new_points -> clear();

	std::vector<bool>	in_rectangle( points.size() );

	for ( size_t  i = 0  ;  i < points.size()  ;  i ++ )
	{
		in_rectangle[i] = in_region( points[i] );
	}

	for ( size_t  i = 0  ;  i < points.size()  ;  i ++ )
	{
		size_t	index_0 = i;
		size_t	index_1 = i + 1;

		if ( index_1 >= points.size() )
		{
			index_1 = 0;
		}

		const D2_Vector &	p0 = points[index_0];
		const D2_Vector &	p1 = points[index_1];

		if ( in_rectangle[index_0] )
		{
			if ( in_rectangle[index_1] )
			{
				new_points -> push_back( p1 );
			}
			else
			{
				try
				{
					D2_Vector
					  c = line.cross_point
					      ( D2_Straight_Line( p0 , p1 ) );

					new_points -> push_back( c );
				}
				catch( const D2_Region::No_Region_Error & )

				{
					std::cerr << "internal error"
						  << std::endl;

					std::abort();
				}
			}
		}
		else
		{
			if ( in_rectangle[index_1] )
			{
				try
				{
					D2_Vector
					  c = line.cross_point
					      ( D2_Straight_Line( p0 , p1 ) );

					new_points -> push_back( c );
				}
				catch( const D2_Region::No_Region_Error & )
				{
					std::cerr << "internal error"
						  << std::endl;

					std::abort();
				}

				new_points -> push_back( p1 );
			}
			else
			{
				// noting to do
			}
		}
	}
}

class  X_Less_Equal
{
private:
	FLOAT	threshold;

public:
	X_Less_Equal( FLOAT  threshold )
		: threshold( threshold )
	{
	}

	bool	operator() ( const D2_Vector &  p ) const
	{
		return( p.x() <= threshold );
	}
};

class  X_More_Equal
{
private:
	FLOAT	threshold;

public:
	X_More_Equal( FLOAT  threshold )
		: threshold( threshold )
	{
	}

	bool	operator() ( const D2_Vector &  p ) const
	{
		return( p.x() >= threshold );
	}
};

class  Y_Less_Equal
{
private:
	FLOAT	threshold;

public:
	Y_Less_Equal( FLOAT  threshold )
		: threshold( threshold )
	{
	}

	bool	operator() ( const D2_Vector &  p ) const
	{
		return( p.y() <= threshold );
	}
};

class  Y_More_Equal
{
private:
	FLOAT	threshold;

public:
	Y_More_Equal( FLOAT  threshold )
		: threshold( threshold )
	{
	}

	bool	operator() ( const D2_Vector &  p ) const
	{
		return( p.y() >= threshold );
	}
};


void   D2_Polygon::get_scissored_connected_polygon
		   ( D2_Polygon *  result ,
		     const D2_Rectangle_Region &  r ) const
{
	if ( this -> vertex.empty() )
	{
		result -> set_empty();

		return;
	}

	std::vector<D2_Vector>	p = this -> vertex;
	std::vector<D2_Vector>	new_p;

	scissor_with_line<X_Less_Equal>( X_Less_Equal( r.max_x() ) ,
					 p , &new_p ,
					 D2_Straight_Line
					 ( D2_Vector( r.max_x() , 0.0 ) ,
					   Angle::quarter_circle() ) );

	p = new_p;
	scissor_with_line<Y_Less_Equal>( Y_Less_Equal( r.max_y() ) ,
					 p , &new_p ,
					 D2_Straight_Line
					 ( D2_Vector( 0.0 , r.max_y() ) ,
					   Angle::origin() ) );

	p = new_p;
	scissor_with_line<X_More_Equal>( X_More_Equal( r.min_x() ) ,
					 p , &new_p ,
					 D2_Straight_Line
					 ( D2_Vector( r.min_x() , 0.0 ) ,
					   Angle::quarter_circle() ) );

	p = new_p;
	scissor_with_line<Y_More_Equal>( Y_More_Equal( r.min_y() ) ,
					 p , &new_p ,
					 D2_Straight_Line
					 ( D2_Vector( 0.0 , r.min_y() ) ,
					   Angle::origin() ) );

	result -> set( new_p );
}


#if 0
bool   D2_Polygon::get_overlap_polygon( std::vector<D2_Polygon> *  result ,
					const D2_Polygon &  p ) const
{
	// XXX: not completed yet
	(void)result;
	(void)p;

	return( false );
}
#endif

bool   D2_Polygon::operator == ( const D2_Polygon &  p ) const
{
	return( this -> vertex == p.vertex );
}

bool   D2_Polygon::operator != ( const D2_Polygon &  p ) const
{
	return( ! this -> operator ==(p) );
}
