#ifndef	   COMMANDBUF_H_INCLUDED
#define	   COMMANDBUF_H_INCLUDED

#ifdef HAVE_CONFIG_H
# include  "config.h"
#endif

#include  "streambuf_extension.h"
#include  <cstdio>
#include  <string>
#include  <vector>
#include  <csignal>
#include  <cstdlib>
#include  <cstdarg>
#include  <iostream>
#include  <unistd.h>
#include  <fcntl.h>
#include  <errno.h>
#include  <sys/types.h>
#include  <sys/wait.h>


class  commandbuf : public streambuf_extension
{
public:
	enum { BAD_PROCESSS_ID = -1 };
	enum { BAD_FD = -1 };

private:
	struct  Child_Process_Command
	{
	public:
		bool	external_command;

		const char *			file;
		const char * const * const	argv;

		int	(*func)(void);

	public:
		Child_Process_Command()
			: external_command( false ) ,
			  file( static_cast<const char *>(0) ) ,
			  argv( static_cast<const char * const * const>(0) ) ,
			  func( static_cast<int (*)(void)>(0) )
		{
		}

		Child_Process_Command( const char *  file ,
				       const char * const * const  argv )
			: external_command( true ) ,
			  file( file ) , argv( argv ) ,
			  func( static_cast<int (*)(void)>(0) )
		{
		}

		Child_Process_Command( int  (*func)(void) )
			: external_command( false ) ,
			  file( static_cast<const char *>(0) ) ,
			  argv( static_cast<const char * const * const>(0) ) ,
			  func( func )
		{
		}
	};


protected:
	int	to_com_fd;
	int	from_com_fd;
	pid_t	command_pid;
	Child_Process_Command command;

private:
	char	buffer[BUFSIZ];

	void	initialize_buffer()
	{
		this -> setg( buffer , buffer , buffer );
	}

private:
	int	process_exit_status;


protected:
	void	invoke( const Child_Process_Command &  command ,
			const char *  redirect_input_file
					 = static_cast<const char *>(0) ,
			const char *  redirect_output_file
					 = static_cast<const char *>(0) );
public:
	void	invoke( const char *  file ,
			const char * const * const  argv ,
			const char *  redirect_input_file
					= static_cast<const char *>(0) ,
			const char *  redirect_output_file
					= static_cast<const char *>(0) );

	void	invoke( const std::string &  file ,
			const std::vector<std::string> &  argv ,
			const char *  redirect_input_file
					= static_cast<const char *>(0) ,
			const char *  redirect_output_file
					= static_cast<const char *>(0) );

	void	invoke( int  (*func)(void) ,
			const char *  redirect_input_file
					 = static_cast<const char *>(0) ,
			const char *  redirect_output_file
					 = static_cast<const char *>(0) );

private:
	 commandbuf( const commandbuf & );		  // not allowed.
	 commandbuf &	operator= ( const commandbuf & ); // not allowed.

public:
	 commandbuf();

	 commandbuf( const std::string &  file ,
		     const std::vector<std::string> &  argv ,
		     const char *  redirect_input_file
				     = static_cast<const char *>(0) ,
		     const char *  redirect_output_file
				     = static_cast<const char *>(0) );

	 commandbuf( const std::vector<std::string> &  argv ,
		     const char *  redirect_input_file
				     = static_cast<const char *>(0) ,
		     const char *  redirect_output_file
				     = static_cast<const char *>(0) );

	 commandbuf( const char *  file ,  const char * const * const  argv ,
		     const char *  redirect_input_file
				     = static_cast<const char *>(0) ,
		     const char *  redirect_output_file
				     = static_cast<const char *>(0) );
	 commandbuf( const char * const * const  argv ,
		     const char *  redirect_input_file
				     = static_cast<const char *>(0) ,
		     const char *  redirect_output_file
				     = static_cast<const char *>(0) );
	 commandbuf( const char *  argv0 , ... );

	 commandbuf( int  (*func)(void) ,
		     const char *  redirect_input_file
				     = static_cast<const char *>(0) ,
		     const char *  redirect_output_file
				     = static_cast<const char *>(0) );

	~commandbuf();

protected:
	int  overflow( int  ch = EOF )
	{
		if ( ch == EOF || to_com_fd == BAD_FD )
		{
			return( EOF );
		}

		unsigned char	c = static_cast<unsigned char>(ch);

		if ( ::write( to_com_fd , &c , static_cast<size_t>(1) ) != 1 )
		{
			this -> close( std::ios::in );

			return( EOF );
		}
		else
		{
			return( ch );
		}
	}

	int  underflow()
	{
		int	ch = this -> uflow();

		this -> pbackfail( ch );

		return( ch );
	}

	int  uflow()
	{
		if ( from_com_fd == BAD_FD )
		{
			return( EOF );
		}

		if ( this -> input_buffer_current()
		     >= this -> buffer + sizeof(this -> buffer) )
		{
			this -> setg( this -> input_buffer_begin() ,
				      this -> input_buffer_begin() ,
				      this -> input_buffer_begin() );
		}

		const size_t	free_buf_size
				= (this -> buffer + sizeof(this -> buffer))
				  - this -> input_buffer_current();

		int	n;

		switch( (n = ::read( from_com_fd ,
				     this -> input_buffer_current() ,
				     free_buf_size )) )
		{
		case -1:
			std::perror( "read" );

			this -> close( std::ios::out );

			return( EOF );
			break;

		case  0:
			this -> close( std::ios::out );

			return( EOF );
			break;

		default:
		      {
			unsigned char	ch;
			ch = *(this -> input_buffer_current());

			this -> setg( this -> input_buffer_begin() ,
				      this -> input_buffer_current() ,
				      this -> input_buffer_current() + n );
			this -> input_buffer_move_current( 1 );

			return( ch );
			break;
		      }
		}
	}

	int	pbackfail( int  ch )
	{
		if ( this -> input_buffer_current()
		     > this -> input_buffer_begin() )
		{
			this -> input_buffer_move_current( -1 );

			return( ch );
		}
		else
		{
			return( EOF );
		}
	}


public:
	void  close( int  in_out = std::ios::in | std::ios::out )
	{
		if ( in_out & std::ios::in )
		{
			if ( to_com_fd != BAD_FD
			  && ::close( to_com_fd ) == -1 )
			{
				std::perror( "close" );
			}

			to_com_fd = BAD_FD;

			this -> initialize_buffer();
		}

		if ( in_out & std::ios::out )
		{
			if ( from_com_fd != BAD_FD
			  && ::close( from_com_fd ) == -1 )
			{
				std::perror( "close" );
			}

			from_com_fd = BAD_FD;

			this -> initialize_buffer();
		}
	}

	int	fd( int  in_out ) const
	{
		if ( in_out == std::ios::in )
		{
			return( to_com_fd );
		}
		else if ( in_out == std::ios::out )
		{
			return( from_com_fd );
		}
		else
		{
			return( BAD_FD );
		}
	}

	pid_t	getpid() const { return( command_pid ); }

	int	kill( int  sig = SIGTERM )
	{
		if ( command_pid == BAD_PROCESSS_ID )
		{
			return( -1 );
		}
		else
		{
			int	ret;

			if ( (ret = ::kill( command_pid , sig )) == -1
			  && errno != ESRCH /* No such process */ )
			{
				std::perror( "kill" );
			}

			return( ret );
		}
	}

	int	poll() const;

	void	wait();

	int	exit_status() const { return( process_exit_status ); }
};



inline
void   commandbuf::invoke( const Child_Process_Command &  command ,
			   const char *  redirect_input_file ,
			   const char *  redirect_output_file )
{
	this -> close( std::ios::in | std::ios::out );
	this -> kill();

	to_com_fd   = BAD_FD;
	from_com_fd = BAD_FD;
	command_pid = BAD_PROCESSS_ID;

	int	to  [2];
	int	from[2];

	if ( ::pipe( to ) == -1 )
	{
		std::perror( "pipe" );

		return;
	}

	if ( ::pipe( from ) == -1 )
	{
		std::perror( "pipe" );

		if ( ::close( to[0] ) == -1 ) { std::perror( "close" ); }
		if ( ::close( to[1] ) == -1 ) { std::perror( "close" ); }

		return;
	}


	switch( (command_pid = ::fork()) )
	{
	case  -1:
		// error
		std::perror( "fork" );

		if ( ::close(   to[0] ) == -1 )	{ std::perror( "close" ); }
		if ( ::close(   to[1] ) == -1 ) { std::perror( "close" ); }
		if ( ::close( from[0] ) == -1 ) { std::perror( "close" ); }
		if ( ::close( from[1] ) == -1 ) { std::perror( "close" ); }

		return;
		break;

	case  0:
		//
		// child process
		//

		// close unused pipe file descripters
		if ( ::close( to[1] ) == -1 )
		{
			std::perror( "close" );
			::_exit( 1 );
		}

		if ( ::close( from[0] ) == -1 )
		{
			std::perror( "close" );
			::_exit( 1 );
		}

		// set input
		if ( redirect_input_file )
		{
			int fd;
			if ( (fd = ::open( redirect_input_file ,
					   O_RDONLY )) == -1 )
			{
				std::perror( "open" );
				::_exit( 1 );
			}

			if ( ::dup2( fd , STDIN_FILENO ) == -1 )
			{
				std::perror( "dup2" );
				::_exit( 1 );
			}

			if ( ::close( fd ) == -1 )
			{
				std::perror( "close" );
				::_exit( 1 );
			}
		}
		else
		{
			if ( ::dup2( to[0] , STDIN_FILENO  ) == -1 )
			{
				std::perror( "dup2" );
				::_exit( 1 );
			}
		}

		if ( redirect_output_file )
		{
			int fd;
			if ( (fd = ::open( redirect_output_file ,
					   O_WRONLY
					   | O_CREAT
					   | O_TRUNC ,
					   0644 )) == -1 )
			{
				std::perror( "open" );
				::_exit( 1 );
			}

			if ( ::dup2( fd , STDOUT_FILENO ) == -1 )
			{
				std::perror( "dup2" );
				::_exit( 1 );
			}

			if ( ::close( fd ) == -1 )
			{
				std::perror( "close" );
				::_exit( 1 );
			}
		}
		else
		{
			if ( ::dup2( from[1] , STDOUT_FILENO ) == -1 )
			{
				std::perror( "dup2" );
				::_exit( 1 );
			}
		}


		if ( ::close( to[0] ) == -1 )
		{
			std::perror( "close" );
			::_exit( 1 );
		}

		if ( ::close( from[1] ) == -1 )
		{
			std::perror( "close" );
			::_exit( 1 );
		}

		if ( command.external_command )
		{
			if ( ::execvp( command.file ,
				       const_cast<char * const *>
						  (command.argv) ) == -1 )
			{
				if ( errno == ENOENT )
				{
					std::cerr << command.argv[0]
						  << " not found" << std::endl;
				}
				else
				{
					std::perror( "execvp" );
				}
			}

			::_exit( 1 );
		}
		else
		{
			::_exit( command.func() );
		}

		break;

	default:
		// parent process
		if ( ::close(   to[0] ) == -1 ) { std::perror( "close" ); }
		if ( ::close( from[1] ) == -1 ) { std::perror( "close" ); }

		to_com_fd   = to  [1];
		from_com_fd = from[0];

		break;
	}
}

inline
void   commandbuf::invoke( const char *  file ,
			   const char * const * const  argv ,
			   const char *  redirect_input_file ,
			   const char *  redirect_output_file )
{
	this -> invoke( Child_Process_Command( file , argv ) ,
			redirect_input_file , redirect_output_file );
}

inline
void   commandbuf::invoke( const std::string &  file ,
			   const std::vector<std::string> &  argv ,
			   const char *  redirect_input_file ,
			   const char *  redirect_output_file )
{
	const char **	v = static_cast<const char **>(0);

	try
	{
		v = new const char * [argv.size() + 1];

		size_t  i;
		for ( i = 0  ;  i < argv.size()  ;  i ++ )
		{
			v[i] = argv[i].c_str();
		}

		v[i] = static_cast<char *>(0);

		this -> invoke( file.c_str() , v ,
				redirect_input_file ,
				redirect_output_file );
	}
	catch(...)
	{
		delete[] v;

		throw;
	}

	delete[] v;
}

inline
void   commandbuf::invoke( int  (*func)(void) ,
			   const char *  redirect_input_file ,
			   const char *  redirect_output_file )
{
	this -> invoke( Child_Process_Command( func ) ,
			redirect_input_file , redirect_output_file );
}

inline
commandbuf::commandbuf()
	: to_com_fd( BAD_FD ) , from_com_fd( BAD_FD ) ,
	  command_pid( BAD_PROCESSS_ID ) , process_exit_status( EXIT_SUCCESS )
{
	this -> initialize_buffer();
}

inline
commandbuf::commandbuf( const std::string &  file ,
			const std::vector<std::string> &  argv ,
			const char *  redirect_input_file ,
			const char *  redirect_output_file )
	: to_com_fd( BAD_FD ) , from_com_fd( BAD_FD ) ,
	  command_pid( BAD_PROCESSS_ID ) , process_exit_status( EXIT_SUCCESS )
{
	this -> initialize_buffer();

	this -> invoke( file , argv ,
			redirect_input_file , redirect_output_file );
}

inline
commandbuf::commandbuf( const std::vector<std::string> &  argv ,
			const char *  redirect_input_file ,
			const char *  redirect_output_file )
	: to_com_fd( BAD_FD ) , from_com_fd( BAD_FD ) ,
	  command_pid( BAD_PROCESSS_ID ) , process_exit_status( EXIT_SUCCESS )
{
	this -> initialize_buffer();

	this -> invoke( argv[0] , argv ,
			redirect_input_file , redirect_output_file );
}

inline
commandbuf::commandbuf( const char *  file ,
			const char * const * const  argv ,
			const char *  redirect_input_file ,
			const char *  redirect_output_file )
	: to_com_fd( BAD_FD ) , from_com_fd( BAD_FD ) ,
	  command_pid( BAD_PROCESSS_ID ) , process_exit_status( EXIT_SUCCESS )
{
	this -> initialize_buffer();

	this -> invoke( file , argv ,
			redirect_input_file , redirect_output_file );
}

inline
commandbuf::commandbuf( const char * const * const  argv ,
			const char *  redirect_input_file ,
			const char *  redirect_output_file )
	: to_com_fd( BAD_FD ) , from_com_fd( BAD_FD ) ,
	  command_pid( BAD_PROCESSS_ID ) , process_exit_status( EXIT_SUCCESS )
{
	this -> initialize_buffer();

	this -> invoke( argv[0] , argv ,
			redirect_input_file , redirect_output_file );
}

inline
commandbuf::commandbuf( const char *  argv0 , ... )
	: to_com_fd( BAD_FD ) , from_com_fd( BAD_FD ) ,
	  command_pid( BAD_PROCESSS_ID ) , process_exit_status( EXIT_SUCCESS )
{
	std::vector<std::string>	argv;
	argv.push_back( argv0 );

	va_list	ap;
	va_start( ap , argv0 );

	for(;;)
	{
		const char *	p = va_arg( ap , const char * );

		if ( ! p )
		{
			break;
		}

		argv.push_back( p );
	}

	va_end( ap );

	this -> initialize_buffer();

	this -> invoke( argv[0] , argv );
}

inline
commandbuf::commandbuf( int  (*func)(void) ,
			const char *  redirect_input_file ,
			const char *  redirect_output_file )
	: to_com_fd( BAD_FD ) , from_com_fd( BAD_FD ) ,
	  command_pid( BAD_PROCESSS_ID ) , process_exit_status( EXIT_SUCCESS )
{
	this -> initialize_buffer();
	this -> invoke( func , redirect_input_file , redirect_output_file );
}


inline
void   commandbuf::wait()
{
	if ( command_pid != BAD_PROCESSS_ID
	  && command_pid != 0 )
	{
		int	status;
		if ( ::waitpid( command_pid , &status , 0 ) == -1
		  && errno != ECHILD )
		{
			std::perror( "waitpid" );
		}

		if ( WIFEXITED( status ) )
		{
			this -> process_exit_status = WEXITSTATUS( status );
		}
		else
		{
			this -> process_exit_status = EXIT_FAILURE;
		}
	}
}

inline
commandbuf::~commandbuf()
{
	this -> close( std::ios::in | std::ios::out );
	this -> wait();
}



#include  "system_call_wrapper.h"

inline
int    commandbuf::poll() const
{
	if ( (this -> input_buffer_end()
	      - this -> input_buffer_current()) > 0 )
	{
		return( 1 );
	}

	if ( from_com_fd == BAD_FD )
	{
		return( 0 );
	}

	return( System_Call_Wrapper::poll( this -> from_com_fd ) );
}


#endif	/* COMMANDBUF_H_INCLUDED */
