#ifndef	   ANGLE_H_INCLUDED
#define	   ANGLE_H_INCLUDED

#ifdef HAVE_CONFIG_H
# include  "config.h"
#endif

#include  <cmath>
#include  "math_extension.h"
#include  "float_traits.h"

//
// Angle
//
class  Angle
{
public:
	enum Unit { Radian , Degree };

protected:
	FLOAT	ang;

public:
	static	FLOAT	deg_to_rad( FLOAT  x )
		{
			return( static_cast<FLOAT>
				( x * float_traits<FLOAT>::PI() / 180.0 ) );
		}

	static	FLOAT	rad_to_deg( FLOAT  x )
		{
			return( static_cast<FLOAT>
				( x * 180.0 / float_traits<FLOAT>::PI() ) );
		}

	static	FLOAT  normalize_angle( FLOAT  dir ,
					bool  use_radian = true )
		{
			FLOAT	half;

			if ( use_radian )
			{
				half = float_traits<FLOAT>::PI();
			}
			else
			{
				half = 180.0;
			}

			if ( dir >= half )
			{
				do
				{
					dir -= static_cast<FLOAT>(2.0) * half;

				} while( dir >= half );
			}
			else if ( dir < (-half) )
			{
				do
				{
					dir += static_cast<FLOAT>(2.0) * half;

				} while( dir < (-half) );
			}

			return( dir );
		}

	static	Angle	atan2( FLOAT  y ,  FLOAT  x )
		{
			return( Angle( Angle::Radian , std::atan2( y , x ) ) );
		}

public:
		Angle()
			:ang( 0.0 )
		{
		}

		Angle( const Angle &  a )
			: ang( a.ang )
		{
		}

		Angle( Unit  u ,  FLOAT  value )
			: ang( (u == Radian) ?
			       value : Angle::deg_to_rad( value ) )
		{
		}

		FLOAT	radian() const
		{
			return( this -> ang );
		}

		FLOAT	degree() const
		{
			return( Angle::rad_to_deg( this -> ang ) );
		}

		Angle	normalize() const
		{
			return( Angle( Angle::Radian ,
				       Angle::normalize_angle(this -> ang) ) );
		}

		bool	operator <  ( const Angle &  a ) const
					{ return( this -> ang <  a.ang ); }
		bool	operator >  ( const Angle &  a ) const
					{ return( this -> ang >  a.ang ); }
		bool	operator <= ( const Angle &  a ) const
					{ return( this -> ang <= a.ang ); }
		bool	operator >= ( const Angle &  a ) const
					{ return( this -> ang >= a.ang ); }
		bool	operator == ( const Angle &  a ) const
					{ return( this -> ang == a.ang ); }
		bool	operator != ( const Angle &  a ) const
					{ return( this -> ang != a.ang ); }

		Angle	operator + ( const Angle &  a ) const
		{
			return( Angle( Angle::Radian , this -> ang + a.ang ) );
		}

		Angle	operator - ( const Angle &  a ) const
		{
			return( Angle( Angle::Radian , this -> ang - a.ang ) );
		}

		FLOAT	operator / ( const Angle &  a ) const
		{
			return( this -> ang / a.ang );
		}

		Angle	operator * ( FLOAT  x ) const
		{
			return( Angle( Angle::Radian , this -> ang * x ) );
		}

		Angle	operator / ( FLOAT  x ) const
		{
			return( Angle( Angle::Radian , this -> ang / x ) );
		}

		Angle &	operator += ( const Angle &  a )
		{
			this -> ang += a.ang;
			return( *this );
		}

		Angle &	operator -= ( const Angle &  a )
		{
			this -> ang -= a.ang;
			return( *this );
		}

		Angle &	operator *= ( FLOAT  x )
		{
			this -> ang *= x;
			return( *this );
		}

		Angle &	operator /= ( FLOAT  x )
		{
			this -> ang /= x;
			return( *this );
		}

		Angle	abs() const
		{
			return( Angle( Angle::Radian ,
				       float_traits<FLOAT>::fabs
				       ( this -> ang ) ) );
		}

		FLOAT	sin() const
		{
			return( float_traits<FLOAT>::sin( this -> ang ) );
		}

		FLOAT	cos() const
		{
			return( float_traits<FLOAT>::cos( this -> ang ) );
		}

		FLOAT	tan() const
		{
			return( std::tan( static_cast<FLOAT>
					  ( this -> ang ) ) );
		}

		FLOAT	sinh() const
		{
			return( std::sinh( static_cast<FLOAT>
					   ( this -> ang ) ) );
		}

		FLOAT	cosh() const
		{
			return( std::cosh( static_cast<FLOAT>
					   ( this -> ang ) ) );
		}

		FLOAT	tanh() const
		{
			return( std::tanh( static_cast<FLOAT>
					   ( this -> ang ) ) );
		}

	static	Angle	origin()
		{
			return( Angle( Angle::Radian , 0.0 ) );
		}

	static	Angle	asin( FLOAT  d )
		{
			return( Angle( Angle::Radian ,
				       float_traits<FLOAT>::asin( d ) ) );
		}

	static	Angle	acos( FLOAT  d )
		{
			return( Angle( Angle::Radian ,
				       float_traits<FLOAT>::acos( d ) ) );
		}


	static	Angle	full_circle()
		{
			return( Angle::half_circle() * 2.0 );
		}

	static	Angle	half_circle()
		{
			return( Angle( Angle::Radian ,
				       float_traits<FLOAT>::PI() ) );
		}

	static	Angle	quarter_circle()
		{
			return( Angle::half_circle() / 2.0 );
		}
};

inline
static
Angle	operator + ( const Angle &  a )
{
	return( a );
}

inline
static
Angle	operator - ( const Angle &  a )
{
	return( a * (-1.0) );
}

inline
static
Angle	operator * ( FLOAT  x ,  const Angle &  a )
{
	return( a * x );
}


//
// Radian
//
class  Radian : public Angle
{
public:
	 Radian( FLOAT  x ) : Angle( Angle::Radian , x ) {}
	 Radian( const Angle &  a ) : Angle( a ) {}
};

//
// Degree
//
class  Degree : public Angle
{
public:
	 Degree( FLOAT  x ) : Angle( Angle::Degree , x ) {}
	 Degree( const Angle &  a ) : Angle( a ) {}
};


#endif	/* ANGLE_H_INCLUDED */
