#include "fuse_xml.h"
#include "universal_name.h"
#include "path_converter.h"
#include "string_extension.h"

#include <sys/stat.h>
#include <cstring>
#include <cerrno>
#include <sstream>

#include <iostream>
#include <algorithm>

#include "ref_count_ptr.h"
#include "xml_data_tree.h"

extern ref_count_ptr<fuse_xml::XMLDataTree> xml_data_tree;

namespace fuse_xml {

int FuseXML::getattr( const char * path,
                      struct stat * stat_buf )
{
    std::cerr << "getattr [" << path << "]" << std::endl;

    std::memset( stat_buf, 0, sizeof(struct stat) );

#if 0
    if ( std::strcmp( path, "/" ) == 0 )
    {
        stat_buf -> st_mode = S_IFDIR | 0755;
        stat_buf -> st_nlink = 2;
    }
    else if ( std::strcmp( path, "/GetCapabilities.xml" ) == 0 )
    {
        stat_buf -> st_mode = S_IFREG | 0444;
        stat_buf -> st_nlink = 1;

        std::string output;
        if ( ! misp_client::MispProxy::instance().getCapabilities( &output ) )
        {
            stat_buf -> st_size = -1;
        }
        else
        {
            stat_buf -> st_size = output.size();
        }
    }
    else if ( std::strcmp( path, "/Features" ) == 0 )
    {
        stat_buf -> st_mode = S_IFREG | 0444;
        stat_buf -> st_nlink = 1;

        std::vector<Universal_Name> feature_list;

        if ( ! misp_client::MispProxy::instance()
               .getFeatureList( &feature_list ) )
        {
            std::cerr << "Can't get feature list." << std::endl;

            stat_buf -> st_size = -1;
        }
        else
        {
            std::cerr << "number of features = " << feature_list.size()
                      << std::endl;

            size_t sum = 0;
            for ( size_t i = 0; i < feature_list.size(); ++ i )
            {
                sum += (feature_list[i].getFQN().length() + 1);

                std::cerr << i << " [" << feature_list[i].getFQN() << "]"
                          << std::endl;
            }

            stat_buf -> st_size = sum;
        }
    }
    else if ( std::strcmp( path, "/FeatureCollection" ) == 0 )
    {
        stat_buf -> st_mode = S_IFDIR | 0755;
        stat_buf -> st_nlink = 2;
    }
    else if ( std::strncmp( path, "/FeatureCollection/",
                            std::strlen( "/FeatureCollection/" ) ) == 0 )
    {
        std::vector<std::string> hier;
        FuseXML::getPathInfo( &hier,
                                 path + std::strlen( "/FeatureCollection/" ) );

        if ( hier.empty() )
        {
            stat_buf -> st_mode = S_IFDIR | 0755;
            stat_buf -> st_nlink = 2;
        }
        else if ( hier.size() == 1 )
        {
            // /FeatureCollection/TYPE
            stat_buf -> st_mode = S_IFDIR | 0755;
            stat_buf -> st_nlink = 2;
        }
        else if ( hier.size() == 2 )
        {
            // /FeatureCollection/TYPE/{1,2,...}
            stat_buf -> st_mode = S_IFDIR | 0755;
            stat_buf -> st_nlink = 2;
        }
        else if ( hier.size() == 3 )
        {
            // /FeatureCollection/TYPE/{1,2,...}
            stat_buf -> st_mode = S_IFREG | 0444;
            stat_buf -> st_nlink = 1;

            std::string output;

            const Universal_Name feature_name( Path_Converter::unescape_slash
                                               ( hier[0] ) );

            long num;
            if ( ! String_Extension::string_to_long( &num, hier[1] ) )
            {
                return -ENOENT;
            }

            misp_client::MispProxy::instance()
                .getFeatureString( &output, feature_name, num );

            stat_buf -> st_size = output.size();
        }
        else
        {
            return -ENOENT;
        }
    }
    else
    {
        return -ENOENT;
    }

    return 0;
#else
    if ( std::strcmp( path, "/" ) == 0 )
    {
        stat_buf -> st_mode = S_IFDIR | 0755;
        stat_buf -> st_nlink = 2;
        return 0;
    }

    std::map<std::string, ref_count_ptr<XMLDataNode> >::const_iterator
        it = xml_data_tree -> getFiles().find( path );

    if ( it == xml_data_tree -> getFiles().end() )
    {
        return -ENOENT;
    }

    ref_count_ptr<XMLDataNode> n = (*it).second;

    if ( n -> isText() )
    {
        stat_buf -> st_mode = S_IFREG | 0444;
        stat_buf -> st_nlink = 1;
        stat_buf -> st_size = n -> getValue().size();
    }
    else
    {
        stat_buf -> st_mode = S_IFDIR | 0755;
        stat_buf -> st_nlink = 2;
    }

    return 0;
#endif
}

int FuseXML::open( const char * path,
                   struct fuse_file_info * file_info )
{
    std::cerr << "open: path = [" << path << "]" << std::endl;

#if 0
    if ( std::strcmp( path, "/" ) == 0 )
    {
        if ( file_info -> flags & 0x11 != O_RDONLY )
        {
            return -EACCES;
        }

        return 0;
    }
#endif

    std::map<std::string, ref_count_ptr<XMLDataNode> >::const_iterator
        it = xml_data_tree -> getFiles().find( path );

    if ( it == xml_data_tree -> getFiles().end() )
    {
        return -ENOENT;
    }

    if ( (*it).second -> isText() )
    {
        return 0;
    }
    else
    {
        if ( file_info -> flags & 0x11 != O_RDONLY )
        {
            return -EACCES;
        }
        else
        {
            return 0;
        }
    }
}

int FuseXML::readdir( const char * path,
                         void * buf,
                         fuse_fill_dir_t filler,
                         off_t offset,
                         struct fuse_file_info * file_info )
{
    std::cerr << "readdir: path = [" << path << "]" << std::endl;

    static_cast<void>(offset);
    static_cast<void>(file_info);

    static_cast<void>(path);
    static_cast<void>(buf);
    static_cast<void>(filler);

#if 0
    if ( std::strcmp( path, "/" ) == 0 )
    {
        filler( buf, ".", NULL, 0 );
        filler( buf, "..", NULL, 0 );

        return 0;
    }
#endif

    std::map<std::string, ref_count_ptr<XMLDataNode> >::const_iterator
        it = xml_data_tree -> getFiles().find( path );

    if ( it == xml_data_tree -> getFiles().end() )
    {
        return -ENOENT;
    }

    ref_count_ptr<XMLDataNode> n = (*it).second;

    if ( n -> isElement() )
    {
        filler( buf, ".", NULL, 0 );
        filler( buf, "..", NULL, 0 );

        for ( size_t i = 0 ; i < n -> getChildElements().size() ; ++ i )
        {
            filler( buf,
                    Path_Converter::escape_slash
                    ( n -> getChildElements()[i].toPathString() ).c_str(),
                    NULL, 0 );
        }
    }

    return 0;


#if 0
    else if ( std::strcmp( path, "/FeatureCollection" ) == 0 )
    {
        filler( buf, ".", NULL, 0 );
        filler( buf, "..", NULL, 0 );

        std::vector<Universal_Name> feature_list;

        if ( ! misp_client::MispProxy::instance().getFeatureList( &feature_list ) )
        {
            std::cerr << "Can't get feature list." << std::endl;

            return -EIO;
        }

        std::cerr << "number of features = " << feature_list.size()
                  << std::endl;

        for ( size_t i = 0; i < feature_list.size(); ++ i )
        {
            filler( buf, 
                    Path_Converter::escape_slash
                      ( feature_list[i].getFQN() ).c_str(),
                    NULL, 0 );
        }

        return 0;
    }
    else if ( std::strncmp( path, "/FeatureCollection/",
                            std::strlen( "/FeatureCollection/" ) ) == 0 )
    {
        filler( buf, ".", NULL, 0 );
        filler( buf, "..", NULL, 0 );

        std::string sub_path( path + std::strlen( "/FeatureCollection/" ) );
        std::cerr << "sub_path = [" << sub_path << "]" << std::endl;
        String_Extension::chomp( &sub_path, '/' );

        std::vector<std::string> hier;
        String_Extension::split_by_char_set( &hier, sub_path, "/" );

        for ( size_t i = 0; i < hier.size(); ++ i )
        {
            std::cerr << "path[" << i << "] = [" << hier[i] << "]"
                      << std::endl;
        }

        if ( hier.empty() )
        {
            return 0;
        }

        const Universal_Name feature_name( Path_Converter::unescape_slash
                                           ( hier[0] ) );

        long count = misp_client::MispProxy::instance()
                     .getFeatureCount( feature_name );

        std::cerr << "feature_name = [" << feature_name.getNamespace() << "], ["
                  << feature_name.getLocalName() << "]" << std::endl;
        std::cerr << "count = " << count << std::endl;

        if ( count == -1 )
        {
            return -ENOENT;
        }

        if ( hier.size() == 1 )
        {
            // /FeatureCollection/*

            for ( long i = 1; i <= count; ++ i )
            {
                std::stringstream num;
                num << i;

                filler( buf, num.str().c_str(), NULL, 0 );
            }
        }
        else if ( hier.size() == 2 )
        {
            // /FeatureCollection/{1,2,...}

            unsigned long num;

            if ( ! String_Extension::string_to_unsigned_long( &num, hier[1] ) )
            {
                return -ENOENT;
            }

            std::cerr << "num = " << num << std::endl;

            filler( buf, feature_name.getLocalName().c_str(), NULL, 0 );
        }
        else
        {
            return -ENOENT;
        }

        return 0;
    }
    else
    {
        return -ENOENT;
    }
#else
    return -ENOENT;
#endif
}

int FuseXML::read( const char * path,
                   char * buf, size_t size,
                   off_t offset,
                   struct fuse_file_info * file_info )
{
    std::cerr << "read: path = [" << path << "], "
              << "size = " << size << ", "
              << "offset = " << offset << std::endl;

    static_cast<void>(file_info);

    static_cast<void>(buf);

    std::map<std::string, ref_count_ptr<XMLDataNode> >::const_iterator
        it = xml_data_tree -> getFiles().find( path );

    if ( it != xml_data_tree -> getFiles().end() )
    {
        ref_count_ptr<XMLDataNode> n = (*it).second;

        const std::string & output = n -> getValue();
        size_t write_size = std::min( static_cast<size_t>(output.size() - offset),
                                      size );
        std::memcpy( buf, output.data() + offset, write_size );

        return write_size;
    }

#if 0
    if ( std::strcmp( path, "/GetCapabilities.xml" ) == 0 )
    {
        misp_client::MispProxy::instance().getCapabilities( &output );
    }
    else if ( std::strcmp( path, "/Features" ) == 0 )
    {
        std::vector<Universal_Name> feature_list;

        if ( ! misp_client::MispProxy::instance().getFeatureList( &feature_list ) )
        {
            return -EIO;
        }

        for ( size_t i = 0; i < feature_list.size(); ++ i )
        {
            output += feature_list[i].getFQN();
            output += '\n';
        }
    }
    else if ( std::strncmp( path, "/FeatureCollection/",
                            std::strlen( "/FeatureCollection/" ) ) == 0 )
    {
        std::vector<std::string> hier;
        FuseXML::getPathInfo( &hier,
                                 path + std::strlen( "/FeatureCollection/" ) );

        if ( hier.size() != 3 )
        {
            // ! /FeatureCollection/TYPE/{1,2,...}/FEATURE
            return -ENOENT;
        }

        const Universal_Name feature_name( Path_Converter::unescape_slash
                                           ( hier[0] ) );

        long num;
        if ( ! String_Extension::string_to_long( &num, hier[1] ) )
        {
            return -ENOENT;
        }

        misp_client::MispProxy::instance()
            .getFeatureString( &output, feature_name, num );
    }
    else
    {
        return -ENOENT;
    }

    size_t write_size = std::min( static_cast<size_t>(output.size() - offset),
                                  size );
    std::memcpy( buf, output.data() + offset, write_size );

    return write_size;
#else
    return -ENOENT;
#endif
}

void FuseXML::getPathInfo( std::vector<std::string> * hier,
                           const std::string & path )
{
    std::string p = path;
    String_Extension::chomp( &p, '/' );

    String_Extension::split_by_char_set( hier, p, "/" );

    for ( size_t i = 0; i < hier -> size(); ++ i )
    {
        std::cerr << "path[" << i << "] = [" << (*hier)[i] << "]"
                  << std::endl;
    }
}

struct fuse_operations FuseXML::getFuseOperations()
{
    struct fuse_operations operations;
    memset( &operations, 0, sizeof(operations) );

    operations.read    = FuseXML::read;
    operations.open    = FuseXML::open;
    operations.readdir = FuseXML::readdir;
    operations.getattr = FuseXML::getattr;

    return operations;
}


} // end of namespace fuse_xml
