<?php
/**
 * 
 * Daomancy
 * 
 * 　WheremancyとFromancyでSQLを組み立て、SELECT文を発行し、
 * 結果からDaomancyRecord（を継承したDAO）を生成します。
 * 　
 * 
 * PHP version 5
 * @package Daomancy
 * @version Release: 0.1.0
 * @author y.terrashima
 */


require_once 'Fromancy.php';
require_once 'Wheremancy.php';
require_once 'WheremancyLiteral.php';
require_once 'DaomancyRecordRack.php';

class Daomancy
{
	//DBハンドル（PDO）
	protected $dbh;
	//PDOステートメントハンドル
	protected $sth;
	
	//Fromancyオブジェクト
	protected $from;
	//Wheremancyオブジェクト
	protected $where;
	//ORDER BY句用（文字列）
	protected $orderBy;
	//LIMIT句用（数値）
	protected $limit;
	//OFFSET句用（数値）
	protected $offset;
	//RecordRackクラス名
	protected $recordRackClass = 'DaomancyRecordRack';
	
	
	//////////////////////////////////////////////////////////////////////
	//	コンストラクタ・ファクトリ
	//////////////////////////////////////////////////////////////////////
	public function __construct(&$dbh)
	{
		$this->dbh = &$dbh;
	}
	
	public static function _(&$dbh)
	{
		return new self($dbh);
	}
	
	public static function select(&$dbh)
	{
		return new self($dbh);
	}
	
	
	//////////////////////////////////////////////////////////////////////
	//	FROM
	//////////////////////////////////////////////////////////////////////
	
	/**
	 * DAOを指定
	 * @param DAOクラス名
	 * @param エイリアス名（省略時はクラス名）
	 * @param データを取得するかどうか
	 * 		（省略時はtrue。条件のみに使用する場合はfalseを指定）
	 */
	public function & from($class , $as = null , $fetch = true)
	{
		$this->from = & Fromancy::_($class , $as , $fetch);
		return $this;
	}
	
	/**
	 * INNNER JOIN
	 * @param DAOクラス名
	 * @param エイリアス名（省略時はクラス名）
	 * @param データを取得するかどうか
	 * 		（省略時はtrue。条件のみに使用する場合はfalseを指定）
	 */
	public function & join($class , $as = null , $fetch = true)
	{
		$this->from->join($class , $as , $fetch);
		return $this;
	}
	
	/**
	 * LEFT JOIN
	 * @param DAOクラス名
	 * @param エイリアス名（省略時はクラス名）
	 * @param データを取得するかどうか
	 * 		（省略時はtrue。条件のみに使用する場合はfalseを指定）
	 */
	public function & leftJoin($class , $as = null , $fetch = true)
	{
		$this->from->leftJoin($class , $as , $fetch);
		return $this;
	}
	
	/**
	 * RIGHT JOIN
	 * @param DAOクラス名
	 * @param エイリアス名（省略時はクラス名）
	 * @param データを取得するかどうか
	 * 		（省略時はtrue。条件のみに使用する場合はfalseを指定）
	 */
	public function & rightJoin($class , $as = null , $fetch = true)
	{
		$this->from->rightJoin($class , $as , $fetch);
		return $this;
	}
	
	/**
	 * ON
	 * @param ON句。最近のJOINで指定したDAOに適用されます
	 */
	public function & on($on)
	{
		$this->from->on($on);
		return $this;
	}
	
	//////////////////////////////////////////////////////////////////////
	//	WHERE
	//////////////////////////////////////////////////////////////////////
	
	/**
	 * WHERE
	 * @param SQL または Wheremancyオブジェクト
	 * @param SQL文字列の場合のみ。プレースホルダ?に渡す引数
	 */
	public function & where($where , $params = array())
	{
		$this->where = Wheremancy::_($where , $params);
		return $this;
	}
	
	/**
	 * AND
	 * 		whereがまだ指定されていない場合、最新のon句に適用されます。
	 * @param SQL または Wheremancyオブジェクト
	 * @param SQL文字列の場合のみ。プレースホルダ?に渡す引数
	 */
	public function & and_($where , $params = array())
	{
		if(isset($this->where)) {
			$this->where->and_($where , $params);
		} else {
			$this->from->and_($where);
		}
		return $this;
	}
	
	/**
	 * OR
	 * @param SQL または Wheremancyオブジェクト
	 * @param SQL文字列の場合のみ。プレースホルダ?に渡す引数
	 */
	public function & or_($where , $params = array())
	{
		if(isset($this->where)) {
			$this->where->or_($where , $params);
		}
		return $this;
	}
	
	//////////////////////////////////////////////////////////////////////
	//	ORDER BY,LIMIT,OFFSET
	//////////////////////////////////////////////////////////////////////
	
	/**
	 * ORDER BY
	 * @param ORDER BY句
	 */
	public function & orderBy($orderBy)
	{
		$this->orderBy = $orderBy;
		return $this;
	}
	
	/**
	 * LIMIT
	 * @param LIMIT（数値のみ）
	 */
	public function &limit($limit)
	{
		$this->limit = intval($limit);
		return $this;
	}
	
	/**
	 * OFFSET
	 * @param OFFSET（数値のみ）
	 */
	public function &offset($offset)
	{
		$this->offset = intval($offset);
		return $this;
	}
	
	
	//////////////////////////////////////////////////////////////////////
	//	SQL生成、実行
	//////////////////////////////////////////////////////////////////////
	
	/**
	 * SQL実行（ステートメント生成のみ）
	 */
	public function & execute()
	{
		if(!$this->sth) {
			$this->sth = $this->dbh->query($this->toSQL());
			$this->sth->setFetchMode(PDO::FETCH_NUM);
			$this->sth->execute($this->getParams());
		}
		return $this;
	}
	
	/**
	 * 結果セットから次の１行を取得
	 * @return DaomancyRecordRack
	 */
	public function & fetch()
	{
		$ret = & $this->_fetchRecordRack();
		return $ret;
	}
	
	/**
	 * 全行取得
	 * @return array(DaomancyRecordRack)
	 */
	public function & fetchAll()
	{
		$ret = array();
		while($rr = & $this->_fetchRecordRack($sth)) {
			$ret[] = & $rr;
		}
		$this->release();
		return $ret;
	}
	
	/**
	 * ステートメント破棄
	 */
	public function & release()
	{
		unset($this->sth);
		return $this;
	}
	
	/**
	 * SQL生成
	 */
	public function toSQL()
	{
		$sql = $this->from->toSQL();
		if(isset($this->where)) {
			$sql .= $this->where->toSQL();
		}
		$sql .= $this->_getOrderBySQL()
			. $this->_getLimitSQL()
			. $this->_getOffsetSQL()
		;
		return $sql;
	}
	
	/**
	 * 指定されたSQL引数を適用順にすべて取得
	 */
	public function getParams()
	{
		$params = array();
		if(isset($this->where)) {
			$params = $this->where->getParams();
		}
		return $params;
	}
	
	/*
	 * デバッグ用SQL生成
	 * 		そのまま流せるかたちのSQLを生成します。
	 * 		※PDO->quote()が実装されていないDBドライバでは動きません
	 */
	public function debug()
	{
		$sql = $this->toSQL();
		try {
			$params = array_map(array($this->dbh , 'quote')
				, $this->getParams());
		} catch(Exception $e) {
			throw $e;
		}
		$sql = str_replace('?' , '%s' , $sql);
		return vsprintf($sql , $params);
	}
	
	
	//////////////////////////////////////////////////////////////////////
	//	内部メソッド
	//////////////////////////////////////////////////////////////////////
	protected function _getOrderBySQL()
	{
		if(!isset($this->orderBy)) {
			return '';
		}
		return " ORDER BY {$this->orderBy} ";
	}
	
	protected function _getLimitSQL()
	{
		if(!isset($this->limit)) {
			return '';
		}
		return " LIMIT {$this->limit} ";
	}
	
	protected function _getOffsetSQL()
	{
		if(!isset($this->offset)) {
			return '';
		}
		return " OFFSET {$this->offset} ";
	}
	
	protected function &_fetchRecordRack()
	{
		$this->execute();
		$ret = null;
		if($row = $this->sth->fetch()) {
			$ret = $this->createRecordRack();
			foreach($this->from->from as &$from) {
				if(!$from->fetch) continue;
				$ret->{$from->as} = new $from->class($this->dbh);
				foreach($ret->{$from->as}->cols as $col) {
					$ret->{$from->as}->{$col} = array_shift($row);
				}
			}
		}
		return $ret;
	}
	
	protected function & createRecordRack()
	{
		$ret = new $this->recordRackClass();
		return $ret;
	}
}

?>