/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.property;

import java.util.LinkedList;

import org.trix.cuery.util.CSSUtil;
import org.trix.cuery.value.CSSValue;

import org.w3c.dom.Element;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: CascadableProperty.java,v 1.05 2005/09/14 9:56:41 Teletha Exp $
 */
public class CascadableProperty extends AbstractProperty {

    /** This is the UA's default stylesheet. */
    public static final int ORIGIN_AGENT = 1;

    /** This is the user's stylesheet. */
    public static final int ORIGIN_USER = 2;

    /** This is the author's stylesheet. */
    public static final int ORIGIN_AUTHOR = 3;

    /** The list of styles. */
    private LinkedList properties = new LinkedList();

    /** The position of a seperator between author and user. */
    private int author = 0;

    /** The position of a seperator between user and agent. */
    private int user = 0;

    /** The parent element. */
    private Element parent;

    /**
     * Create CascadableProperty instance.
     */
    public CascadableProperty() {
    }

    /**
     * Create CascadableProperty instance.
     * 
     * @param parent A parent element.
     */
    public CascadableProperty(Element parent) {
        this.parent = parent;
    }

    /**
     * @see org.trix.cuery.property.Property#getValue(java.lang.String, int)
     */
    public CSSValue getValue(String name, int state) {
        PropertyDefinition definition = PropertyRegistry.getDefinition(name);
        CSSValue value = getSpecifiedValue(definition, state);

        return definition.getComputedValue(value, this, CSSUtil.getProperty(parent));
    }

    /**
     * Assign a specified value to a property. See also <a
     * href="http://www.w3.org/TR/REC-CSS2/cascade.html#specified-value">the mechanisms</a>.
     * 
     * @param definition A property definition.
     * @param state A state type.
     * @return A specified value.
     */
    private CSSValue getSpecifiedValue(PropertyDefinition definition, int state) {
        // assert null
        if (definition == null) {
            
        }
        
        String name = definition.getName();
        CSSValue result;

        // If the cascade results in a value, use it.
        // search in user stylesheet with important priority
        result = searchValue(name, state, author, user, true);

        if (result != null) {
            return result;
        }

        // search in author stylesheet with important priority
        result = searchValue(name, state, 0, author, true);

        if (result != null) {
            return result;
        }

        // search in author stylesheet
        result = searchValue(name, state, 0, author, false);

        if (result != null) {
            return result;
        }

        // search in user stylesheet
        result = searchValue(name, state, author, user, false);

        if (result != null) {
            return result;
        }

        // search in agent stylesheet
        result = searchValue(name, state, user, properties.size(), false);

        if (result != null) {
            return result;
        }

        // Otherwise, if the property is inherited, use the value of the parent element, generally
        // the computed value.
        if (parent != null && definition.isInheritable()) {
            return CSSUtil.getProperty(parent).getValue(name);
        }

        // Otherwise use the property's initial value. The initial value of each property is
        // indicated in the property's definition.
        return definition.getInitialValue();
    }

    /**
     * Search value by various conditions.
     * 
     * @param name A property name.
     * @param state A state type.
     * @param start A start point.
     * @param end A end point.
     * @param important A important flag.
     * @return A found value.
     */
    private CSSValue searchValue(String name, int state, int start, int end, boolean important) {
        for (int i = start; i < end; i++) {
            PropertyWeightSet set = (PropertyWeightSet) properties.get(i);
            Property property = set.property;

            // check important
            if (important && !property.isImportant(name)) {
                continue;
            }

            // try to retrive value
            CSSValue value = property.getValue(name);

            if (value == null) {
                continue;
            }

            // check state pseudo classes
            if (state == NORMAL) {
                if (set.state == NORMAL) {
                    return value;
                }
            } else {
                if ((state & set.state) == set.state || set.state == NORMAL) {
                    return value;
                }
            }
        }
        return null;
    }

    /**
     * @see org.trix.cuery.property.Property#isImportant(java.lang.String)
     */
    public boolean isImportant(String name) {
        return false;
    }

    /**
     * Add property.
     * 
     * @param property A target property.
     * @param origin A origin type.
     * @param specificity A specificity of this property.
     * @param position A position of this property.
     */
    public void addProperty(Property property, int origin, int specificity, int position) {
        addProperty(property, origin, specificity, position, NORMAL);
    }

    /**
     * Add property.
     * 
     * @param property A target property.
     * @param origin A origin type.
     * @param specificity A specificity of this property.
     * @param state A state type.
     * @param position A position of this property.
     */
    public void addProperty(Property property, int origin, int specificity, int position, int state) {
        PropertyWeightSet set = new PropertyWeightSet();
        set.property = property;
        set.position = position;
        set.specificity = specificity;
        set.state = state;

        switch (origin) {
        case ORIGIN_AUTHOR:
            addProperty(set, 0, author);
            author++;
            user++;
            break;

        case ORIGIN_USER:
            addProperty(set, author, user);
            user++;
            break;

        case ORIGIN_AGENT:
            addProperty(set, user, properties.size());
            break;

        default:
            break;
        }
    }

    /**
     * Add property to a suitable position in list of properties.
     * 
     * @param set A property weight set.
     * @param start A start position for this property origin.
     * @param end A end position for this property origin.
     */
    private void addProperty(PropertyWeightSet set, int start, int end) {
        for (int i = start; i < end; i++) {
            PropertyWeightSet target = (PropertyWeightSet) properties.get(i);

            // check specificity
            if (set.specificity > target.specificity) {
                properties.add(i, set);
                return;
            }

            // check position
            if (set.specificity == target.specificity && set.position > target.position) {
                properties.add(i, set);
                return;
            }
        }
        // lowest weight
        properties.add(end, set);
    }

    /**
     * Clear cached style information.
     */
    public void clear() {
        properties.clear();
        author = 0;
        user = 0;
    }

    /**
     * DOCUMENT.
     * 
     * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
     * @version $ Id: PropertyWeightSet.java,v 1.0 2005/08/25 16:42:35 Teletha Exp $
     */
    private static class PropertyWeightSet {

        /** The actual property. */
        private Property property;

        /** The position of this property. */
        private int position;

        /** The specificity of this property. */
        private int specificity;

        /** The state of this property. */
        private int state;
    }
}
