/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.filter;

import org.trix.cuery.util.CSSUtil;
import org.trix.cuery.util.DOMUtil;

import org.w3c.css.sac.Selector;
import org.w3c.css.sac.SiblingSelector;
import org.w3c.css.sac.SimpleSelector;
import org.w3c.dom.Element;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: DirectAdjacentFilter.java,v 1.11 2005/11/17 06:35:32 Teletha Exp $
 */
public class DirectAdjacentFilter implements Filter, SiblingSelector {

    /** The node type. */
    protected short nodeType;

    /** The parent selector. */
    protected Filter parent;

    /** The current selector. */
    protected Filter filter;

    /**
     * Create DirectAdjacentFilter instance.
     * 
     * @param nodeType A node type.
     * @param parent A parent selector.
     * @param current A current selector.
     */
    public DirectAdjacentFilter(short nodeType, Selector parent, SimpleSelector current) {
        this.nodeType = nodeType;
        this.parent = CSSUtil.convert(parent);
        this.filter = CSSUtil.convert(current);
    }

    /**
     * @see org.trix.cuery.filter.Filter#accept(org.w3c.dom.Element)
     */
    public boolean accept(Element target) {
        if (!filter.accept(target)) {
            return false;
        }

        target = DOMUtil.getPreviousElement(target);

        // assert null
        if (target == null) {
            return false;
        }
        return parent.accept(target);
    }

    /**
     * @see org.trix.cuery.filter.Filter#getSpecificity()
     */
    public int getSpecificity() {
        return parent.getSpecificity() + filter.getSpecificity();
    }

    /**
     * @see org.trix.cuery.filter.Filter#getState()
     */
    public int getState() {
        return 0;
    }

    /**
     * @see org.w3c.css.sac.Condition#getConditionType()
     */
    public short getConditionType() {
        return SAC_AND_CONDITION;
    }

    /**
     * @see org.w3c.css.sac.SiblingSelector#getNodeType()
     */
    public short getNodeType() {
        return nodeType;
    }

    /**
     * @see org.w3c.css.sac.SiblingSelector#getSelector()
     */
    public Selector getSelector() {
        return parent;
    }

    /**
     * @see org.w3c.css.sac.SiblingSelector#getSiblingSelector()
     */
    public SimpleSelector getSiblingSelector() {
        return (SimpleSelector) filter;
    }

    /**
     * @see org.w3c.css.sac.Selector#getSelectorType()
     */
    public short getSelectorType() {
        return SAC_DIRECT_ADJACENT_SELECTOR;
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return parent + " + " + filter;
    }
}
