/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.trix.cuery.css.AbstractCSSConsumer;
import org.trix.cuery.css.CSSConsumer;
import org.trix.cuery.css.CSSPipe;
import org.trix.cuery.filter.Filter;
import org.trix.cuery.parser.CueryParser;
import org.trix.cuery.property.CascadableProperty;
import org.trix.cuery.property.Property;
import org.trix.cuery.property.SimpleProperty;
import org.trix.cuery.util.DOMUtil;
import org.trix.cuery.value.CSSValue;

import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.InputSource;
import org.w3c.css.sac.LexicalUnit;
import org.w3c.css.sac.SelectorList;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.events.Event;
import org.w3c.dom.events.EventListener;
import org.w3c.dom.events.EventTarget;
import org.w3c.dom.stylesheets.StyleSheet;
import org.w3c.dom.stylesheets.StyleSheetList;

import org.xml.sax.SAXException;

/**
 * <p>
 * This class weaves information on the property sepecified with CSS for DOM document.
 * </p>
 * <p>
 * Once a user agent has parsed a document and constructed a document tree, it must assign, for
 * every element in the tree, a value to every property that applies to the target media type. The
 * final value of a property is the result of a three-step calculation: the value is determined
 * through specification (the "specified value"), then resolved into an absolute value if necessary
 * (the "computed value"), and finally transformed according to the limitations of the local
 * environment (the "actual value").
 * </p>
 * <p>
 * CSSWeaver calculates not the actual value but the computed value. So you should decide the actual
 * value even for the runtime environment.
 * </p>
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: CSSWeaver.java,v 1.07 2005/11/15 3:58:08 Teletha Exp $
 */
public class CSSWeaver {

    /** The css parser. */
    private static CueryParser parser = new CueryParser();

    /** The style processor. */
    private PropertyProcessor processor = new PropertyProcessor();

    /** The inline style processor. */
    private PropertyProcessor inlineProcessor = null;

    /** The uri resolver. */
    private URIResolver resolver = new DefaultURIResolver();

    /** The flag to aware xml-stylesheet pi. */
    private boolean awareXMLStylesheet = false;

    /** The flag to aware DOM node mutation. */
    private boolean awareDOMMutarion = false;

    /** The cache for css cuery. */
    private CSSQuery queryCache = null;

    /**
     * Add styles as an agent's stylesheet.
     * 
     * @param source A stylesheet source.
     * @throws IOException If this uri is invalid.
     */
    public void addAgentStylesheet(InputSource source) throws IOException {
        parseStylesheet(source, processor, CascadableProperty.ORIGIN_AGENT);
    }

    /**
     * Add styles as an agent's stylesheet with the css pipe for events hooking.
     * 
     * @param source A stylesheet source.
     * @param pipe A css pipe to hook.
     * @throws IOException If this uri is invalid.
     */
    public void addAgentStylesheet(InputSource source, CSSPipe pipe) throws IOException {
        pipe.setConsumer(processor);
        parseStylesheet(source, pipe, CascadableProperty.ORIGIN_AGENT);
    }

    /**
     * Add styles as an author's stylesheet.
     * 
     * @param source A stylesheet source.
     * @throws IOException If this uri is invalid.
     */
    public void addAuthorStylesheet(InputSource source) throws IOException {
        parseStylesheet(source, processor, CascadableProperty.ORIGIN_AUTHOR);
    }

    /**
     * Add styles as an author's stylesheet with the css pipe for events hooking.
     * 
     * @param source A stylesheet source.
     * @param pipe A css pipe to hook.
     * @throws IOException If this uri is invalid.
     */
    public void addAuthorStylesheet(InputSource source, CSSPipe pipe) throws IOException {
        pipe.setConsumer(processor);
        parseStylesheet(source, pipe, CascadableProperty.ORIGIN_AUTHOR);
    }

    /**
     * Add styles as an user's stylesheet.
     * 
     * @param source A stylesheet source.
     * @throws IOException If this uri is invalid.
     */
    public void addUserStylesheet(InputSource source) throws IOException {
        parseStylesheet(source, processor, CascadableProperty.ORIGIN_USER);
    }

    /**
     * Add styles as an user's stylesheet with the css pipe for events hooking.
     * 
     * @param source A stylesheet source.
     * @param pipe A css pipe to hook.
     * @throws IOException If this uri is invalid.
     */
    public void addUserStylesheet(InputSource source, CSSPipe pipe) throws IOException {
        pipe.setConsumer(processor);
        parseStylesheet(source, pipe, CascadableProperty.ORIGIN_USER);
    }

    /**
     * Helper method to parse stylesheet.
     * 
     * @param stylesheet A target stylesheet to parse.
     * @param consumer A css consumer.
     * @param origin A stylesheet origin.
     * @throws IOException If this stylesheet is invalid.
     */
    private synchronized void parseStylesheet(InputSource stylesheet, CSSConsumer consumer, int origin)
            throws IOException {
        processor.origin = origin;
        parser.setDocumentHandler(consumer);
        parser.parseStyleSheet(stylesheet);
    }

    /**
     * Apply styles to the document file.
     * 
     * @param path A path to the xml document file.
     * @return A styled document.
     * @throws IOException If this document has I/O error.
     */
    public Document apply(String path) throws IOException {
        return apply(new File(path));
    }

    /**
     * Apply styles to the document file.
     * 
     * @param file A xml document file.
     * @return A styled document.
     * @throws IOException If this document has I/O error.
     */
    public Document apply(File file) throws IOException {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setNamespaceAware(true);

        try {
            DocumentBuilder builder = factory.newDocumentBuilder();

            return apply(builder.parse(file));
        } catch (ParserConfigurationException e) {
            throw new IOException(e.getMessage());
        } catch (SAXException e) {
            throw new IOException(e.getMessage());
        }
    }

    /**
     * Apply styles to the document.
     * 
     * @param document A target document.
     * @return A styled document.
     */
    public Document apply(Document document) {
        // create query for this document
        CSSQuery query = new CSSQuery(document);

        // weave registered styles
        processor.weave(query, false);

        // aware linked inline styles
        if (awareXMLStylesheet) {
            inlineProcessor = new PropertyProcessor(processor);
            StyleSheetList stylesheets = DOMUtil.getStylesheets(document);

            for (int i = 0; i < stylesheets.getLength(); i++) {
                StyleSheet stylesheet = stylesheets.item(i);

                // check type
                if (!stylesheet.getType().equals("text/css")) {
                    continue;
                }

                try {
                    // resolve uri and parse it
                    InputSource source = resolver.resolve(stylesheet.getHref(), document.getDocumentURI());
                    parseStylesheet(source, inlineProcessor, CascadableProperty.ORIGIN_AUTHOR);
                } catch (IOException e) {
                    System.out.println(e);
                    // do nothing
                } catch (CSSException e) {
                    System.out.println(e);
                    // do nothing
                }
            }

            // weave linked inline styles
            inlineProcessor.weave(query, false);
        }

        // register listener
        if (awareDOMMutarion && document instanceof EventTarget) {
            // store cache
            queryCache = query;

            EventTarget target = (EventTarget) document;
            target.addEventListener("DOMNodeInserted", new MutationListener(), true);
        }
        return document;
    }

    /**
     * Set a flag whether xml-stylesheet processing instruction is aware or not.
     * 
     * @param aware A flag.
     */
    public void awareXmlStylesheet(boolean aware) {
        this.awareXMLStylesheet = aware;
    }

    /**
     * Set a flag whether dom node mutation is aware or not. If this setting is true, whole document
     * is reparsed at each DOM node modifications. So you shouldn't use this method when the
     * processing speed is important.
     * 
     * @param aware A flag.
     */
    public void awareDOMMutation(boolean aware) {
        this.awareDOMMutarion = aware;
    }

    /**
     * Set URIResolver.
     * 
     * @param resolver A uri resolver.
     */
    public void setURIResolver(URIResolver resolver) {
        // assert null
        if (resolver == null) {
            return;
        }
        this.resolver = resolver;
    }

    /**
     * DOCUMENT.
     * 
     * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
     * @version $ Id: PropertyProcessor.java,v 1.03 2005/09/11 5:50:17 Teletha Exp $
     */
    private static final class PropertyProcessor extends AbstractCSSConsumer {

        /** The style rule list. */
        private List container = new ArrayList();

        /** The origin. */
        private int origin = CascadableProperty.ORIGIN_AUTHOR;

        /** The current position. */
        private int position;

        /** The current poperty. */
        private SimpleProperty property;

        /**
         * Create PropertyProcessor instance.
         */
        public PropertyProcessor() {
            this.position = -1;
        }

        /**
         * Create PropertyProcessor instance.
         * 
         * @param parent A parent.
         */
        public PropertyProcessor(PropertyProcessor parent) {
            this.position = parent.position;
        }

        /**
         * @see org.trix.cuery.css.AbstractCSSConsumer#property(java.lang.String,
         *      org.w3c.css.sac.LexicalUnit, boolean)
         */
        public void property(String name, LexicalUnit value, boolean important) throws CSSException {
            property.setProperty(name, (CSSValue) value, important);
        }

        /**
         * @see org.trix.cuery.css.AbstractCSSConsumer#startSelector(org.w3c.css.sac.SelectorList)
         */
        public void startSelector(SelectorList selectorList) throws CSSException {
            property = new SimpleProperty();
            position++;
        }

        /**
         * @see org.trix.cuery.css.AbstractCSSConsumer#endSelector(org.w3c.css.sac.SelectorList)
         */
        public void endSelector(SelectorList selectorList) throws CSSException {
            container.add(new Object[] {selectorList, property, Integer.valueOf(position), Integer.valueOf(origin)});
        }

        /**
         * Actual style weaver.
         * 
         * @param query A css query.
         * @param reparse A flag whether this weaving is reparsing or not.
         */
        private void weave(CSSQuery query, boolean reparse) {
            // check null
            if (query == null) {
                return;
            }

            // avoid dupulicate operation
            Set set = new HashSet();

            for (int i = 0; i < container.size(); i++) {
                Object[] objects = (Object[]) container.get(i);
                SelectorList list = (SelectorList) objects[0];
                Property property = (Property) objects[1];
                int position = ((Integer) objects[2]).intValue();
                int origin = ((Integer) objects[3]).intValue();

                for (int j = 0; j < list.getLength(); j++) {
                    Filter filter = (Filter) list.item(j);
                    Set elements = query.select(filter);
                    Iterator iterator = elements.iterator();

                    // weave style
                    while (iterator.hasNext()) {
                        Element element = (Element) iterator.next();
                        CascadableProperty cascadable = (CascadableProperty) element.getUserData(Property.KEY);

                        if (cascadable == null) {
                            // check parent element
                            Element parent = DOMUtil.getParentElement(element);

                            if (parent == null) {
                                cascadable = new CascadableProperty();
                            } else {
                                cascadable = new CascadableProperty(parent);
                            }
                        }

                        if (reparse && !set.contains(element)) {
                            set.add(element);
                            cascadable.clear();
                        }

                        cascadable.addProperty(property, origin, filter.getSpecificity(), position, filter.getState());
                        element.setUserData(Property.KEY, cascadable, null);
                    }
                }
            }
        }
    }

    /**
     * DOCUMENT.
     * 
     * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
     * @version $ Id: MutationListener.java,v 1.0 2005/11/15 3:41:09 Teletha Exp $
     */
    private final class MutationListener implements EventListener {

        /**
         * @see org.w3c.dom.events.EventListener#handleEvent(org.w3c.dom.events.Event)
         */
        public void handleEvent(Event event) {
            String type = event.getType();

            if (type.equals("DOMNodeInserted")) {
                Node node = (Node) event.getTarget();

                if (node.getNodeType() != Node.ELEMENT_NODE) {
                    return;
                }

                if (inlineProcessor == null) {
                    processor.weave(queryCache, true);
                } else {
                    inlineProcessor.weave(queryCache, true);
                }
            }
        }
    }
}
