package jp.sourceforge.csvparser;

import java.io.BufferedReader;
import java.io.Reader;
import java.io.StringReader;

import junit.framework.TestCase;

public class CSVParserImplTest extends TestCase {

    private CSVParserImpl parser;

    public CSVParserImplTest(String name) {

        super(name);
    }

    protected void setUp() throws Exception {

        super.setUp();
        parser = new CSVParserImpl();
    }

    private CSVHandler createCSVHandler() {

        return new MockCSVHandler();
    }

    private Reader openReader(String s) {

        return new BufferedReader(new StringReader(s));
    }

    private void assertEquals(String[] expected, String[] actual) {

        assertEquals(expected.length, actual.length);

        for (int i = 0; i < expected.length; i++) {
            assertEquals(String.valueOf(i), expected[i], actual[i]);
        }
    }

    public void testEmpty() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testDelim() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader(",");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "endValue",
                "startValue",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testQuot() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\"");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testCRLF() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\r\n");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testLF() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\n");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testChar() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("a");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "character:a",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testCharDelim() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("a,");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "character:a",
                "endValue",
                "startValue",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testCharQuot() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("a\"");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "character:a",
                "character:\"",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testCharCRLF() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("a\r\n");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "character:a",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testCharLF() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("a\n");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "character:a",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testCharChar() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("aa");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "character:a",
                "character:a",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testDelimDelim() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader(",,");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "endValue",
                "startValue",
                "endValue",
                "startValue",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testDelimQuot() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader(",\"");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "endValue",
                "startValue",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testDelimLF() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader(",\n");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "endValue",
                "startValue",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testDelimChar() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader(",a");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "endValue",
                "startValue",
                "character:a",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testQuotDelim() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\",");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "character:,",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testQuotQuot() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\"\"");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testQuotCRLF() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\"\r\n");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "character:\r",
                "character:\n",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testQuotLF() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\"\n");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "character:\n",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testQuotChar() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\"a");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "character:a",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testQuotQuotDelim() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\"\",");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "endValue",
                "startValue",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testQuotQuotQuot() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\"\"\"");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "character:\"",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testQuotQuotCRLF() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\"\"\r\n");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testQuotQuotLF() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\"\"\n");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testQuotQuotChar() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\"\"a");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "character:a",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testTab() throws Exception {

        parser.setDelimiter('\t');
        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\t");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "endValue",
                "startValue",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testQuotTab() throws Exception {

        parser.setDelimiter('\t');
        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("\"\t");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "character:\t",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }

    public void testSimple() throws Exception {

        MockCSVHandler handler = new MockCSVHandler();
        Reader reader = openReader("1,One,First\r\n2,Two,Second\r\n3,Three,Third\r\n");
        
        try {
            parser.parse(reader, handler);
        } finally {
            reader.close();
        }

        String[] actual = handler.getCalls();
        String[] expected = {
                "startDocument",
                "startRecord",
                "startValue",
                "character:1",
                "endValue",
                "startValue",
                "character:O",
                "character:n",
                "character:e",
                "endValue",
                "startValue",
                "character:F",
                "character:i",
                "character:r",
                "character:s",
                "character:t",
                "endValue",
                "endRecord",
                "startRecord",
                "startValue",
                "character:2",
                "endValue",
                "startValue",
                "character:T",
                "character:w",
                "character:o",
                "endValue",
                "startValue",
                "character:S",
                "character:e",
                "character:c",
                "character:o",
                "character:n",
                "character:d",
                "endValue",
                "endRecord",
                "startRecord",
                "startValue",
                "character:3",
                "endValue",
                "startValue",
                "character:T",
                "character:h",
                "character:r",
                "character:e",
                "character:e",
                "endValue",
                "startValue",
                "character:T",
                "character:h",
                "character:i",
                "character:r",
                "character:d",
                "endValue",
                "endRecord",
                "endDocument"
        };
        assertEquals(expected, actual);
    }
}