package jp.sourceforge.csvparser;

import java.io.IOException;
import java.io.Reader;

public class CSVParserImpl implements CSVParser {

    private static final int END = 0;

    private static final int NEW_RECORD = 1;

    private static final int DELIMITED = 2;

    private static final int VALUE = 3;

    private static final int QUOTED = 4;

    private static final int ESCAPED = 5;

    private char delimiter = ',';

    public void parse(Reader reader, CSVHandler handler) throws IOException {

        handler.startDocument();
        int state = NEW_RECORD;
        int ch = 0;

        while ((ch = reader.read()) != -1) {
            state = handle(state, ch, handler);
        }

        state = handle(state, ch, handler);
        handler.endDocument();
    }

    private int handle(int state, int ch, CSVHandler handler) {

        if (state == NEW_RECORD) {
            return handleOnNewRecord(ch, handler);

        } else if (state == VALUE) {
            return handleOnValue(ch, handler);

        } else if (state == DELIMITED) {
            return handleOnDelimited(ch, handler);

        } else if (state == QUOTED) {
            return handleOnQuoted(ch, handler);

        } else if (state == ESCAPED) {
            return handleOnEscaped(ch, handler);

        } else {
            /* ignore */
        }

        return state;
    }

    private int handleOnNewRecord(int ch, CSVHandler handler) {

        int state = NEW_RECORD;

        if (ch == delimiter) {
            handler.startRecord();
            handler.startValue();
            handler.endValue();
            state = DELIMITED;

        } else if (ch == '"') {
            handler.startRecord();
            handler.startValue();
            state = QUOTED;

        } else if (ch == '\r') {
            /* ignore */

        } else if (ch == '\n') {
            handler.startRecord();
            handler.endRecord();

        } else if (ch == -1) {
            state = END;

        } else {
            handler.startRecord();
            handler.startValue();
            handler.character((char) ch);
            state = VALUE;
        }

        return state;
    }

    private int handleOnValue(int ch, CSVHandler handler) {

        int state = VALUE;

        if (ch == delimiter) {
            handler.endValue();
            state = DELIMITED;

        } else if (ch == '"') {
            handler.character((char) ch);

        } else if (ch == '\r') {
            /* ignore */

        } else if (ch == '\n') {
            handler.endValue();
            handler.endRecord();
            state = NEW_RECORD;

        } else if (ch == -1) {
            handler.endValue();
            handler.endRecord();
            state = END;

        } else {
            handler.character((char) ch);
        }

        return state;
    }

    private int handleOnDelimited(int ch, CSVHandler handler) {

        int state = DELIMITED;

        if (ch == delimiter) {
            handler.startValue();
            handler.endValue();

        } else if (ch == '"') {
            handler.startValue();
            state = QUOTED;

        } else if (ch == '\r') {
            /* ignore */

        } else if (ch == '\n') {
            handler.startValue();
            handler.endValue();
            handler.endRecord();
            state = NEW_RECORD;

        } else if (ch == -1) {
            handler.startValue();
            handler.endValue();
            handler.endRecord();
            state = END;

        } else {
            handler.startValue();
            handler.character((char) ch);
            state = VALUE;
        }

        return state;
    }

    private int handleOnQuoted(int ch, CSVHandler handler) {

        int state = QUOTED;

        if (ch == delimiter) {
            handler.character((char) ch);

        } else if (ch == '"') {
            state = ESCAPED;

        } else if (ch == '\r') {
            handler.character((char) ch);

        } else if (ch == '\n') {
            handler.character((char) ch);

        } else if (ch == -1) {
            handler.endValue();
            handler.endRecord();
            state = END;

        } else {
            handler.character((char) ch);
        }

        return state;
    }

    private int handleOnEscaped(int ch, CSVHandler handler) {

        int state = ESCAPED;

        if (ch == delimiter) {
            handler.endValue();
            state = DELIMITED;

        } else if (ch == '"') {
            handler.character((char) ch);
            state = QUOTED;

        } else if (ch == '\r') {
            /* ignore */

        } else if (ch == '\n') {
            handler.endValue();
            handler.endRecord();
            state = NEW_RECORD;

        } else if (ch == -1) {
            handler.endValue();
            handler.endRecord();
            state = END;

        } else {
            handler.character((char) ch);
            state = VALUE;
        }

        return state;
    }

    public char getDelimiter() {

        return delimiter;
    }

    public void setDelimiter(char delimiter) {

        this.delimiter = delimiter;
    }
}