(*
 * Copyright (C) 2002 Jun Kikuchi <kikuchi@bonnou.com>
 *
 * This file is part of ÍobN.
 *
 * ÍobN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ÍobN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *)
unit PacketPipeUnit;

interface

uses
  Classes, gcryptUnit, KeyUnit, PacketUnit, PacketIOUnit, zlibUnit;

const
  ELITERAL_FILEMODE = 'epPbg̃t@C[hُł';
  EENCMDC_VER       = 'Sym. Encrypted Integrity Protected DatapPbg̃o[Wُł';
  ESESSKEY          = 'ZbVL[ُł';
  EZLIB_DEFLATEINIT = 'kCȕɎs܂';
  EZLIB_DEFLATE     = 'kɎs܂';
  ECOMP_ALGO        = 'sȈkASYł';

type
  TPipe = class(TStream)
  private
    FStream: TStream;
    FOnProgress: TNotifyEvent;
    procedure DoProgress;
  public
    constructor Create(Stream: TStream);
    function Write(const Buffer; Count: Longint): Longint; override;
    function Read(var Buffer; Count: Longint): Longint; override;
    procedure EncodeFrom(S: TStream);
    procedure DecodeTo(S: TStream);
    function DecodeToNull(DecodeSize: Integer): Integer;
    property OnProgress: TNotifyEvent read FOnProgress write FOnProgress;
  end;

  TPartialPacketPipe = class(TPipe)
  private
    FPacketType: PACKET_TYPES;
    FBufStream: TMemoryStream;
  public
    constructor Create(Stream: TStream; PT: PACKET_TYPES);
    destructor Destroy; override;
  end;

  TPartialPacketEncodePipe = class(TPartialPacketPipe)
  private
    FWriteHeader: Boolean;
  protected
    procedure WriteHeader;
    procedure FlushBuffer;
  public
    constructor Create(Stream: TStream; PT: PACKET_TYPES);
    destructor Destroy; override;
    function Write(const Buffer; Count: Longint): Longint; override;
  end;

  TPartialPacketDecodePipe = class(TPartialPacketPipe)
  private
    FReadHeader: Boolean;
    FBodyLen: Int64;
  public
    constructor Create(Stream: TStream; PT: PACKET_TYPES);
    function Read(var Buffer; Count: Longint): Longint; override;
  end;

  TLiteralPacketPipe = class(TPipe)
  private
    FHeader: TLiteralPacketHeader;
  public
    constructor Create(Stream: TStream; LPH: TLiteralPacketHeader);
    property Header: TLiteralPacketHeader read FHeader;
  end;

  TLiteralPacketEncodePipe = class(TLiteralPacketPipe)
  private
    FWriteHeader: Boolean;
  public
    constructor Create(Stream: TStream; LPH: TLiteralPacketHeader);
    function Write(const Buffer; Count: Longint): Longint; override;
  end;

  TLiteralPacketDecodePipe = class(TLiteralPacketPipe)
  private
    FReadHeader: Boolean;
  public
    constructor Create(Stream: TStream; LPH: TLiteralPacketHeader);
    function Read(var Buffer; Count: Longint): Longint; override;
    property ReadHeader: Boolean read FReadHeader;
  end;

  TEncryptPacketsPipe = class(TPipe)
  private
    FSessKey: TSessionKey;
    FCH: GCRY_CIPHER_HD;
    FMD: GCRY_MD_HD;
  public
    constructor Create(Stream: TStream; SessKey: TSessionKey);
    destructor Destroy; override;
  end;

  TEncryptPacketsEncodePipe = class(TEncryptPacketsPipe)
  private
    FWriteHeader: Boolean;
  protected
    procedure WriteHeader;
    procedure FlushBuffer;
  public
    constructor Create(Stream: TStream; SessKey: TSessionKey);
    destructor Destroy; override;
    function Write(const Buffer; Count: Longint): Longint; override;
  end;

  TEncryptPacketsDecodePipe = class(TEncryptPacketsPipe)
  private
    FReadHeader: Boolean;
    FBufStream: TMemoryStream;
    FCheckDigest: Boolean;
  protected
    procedure ReadHeader;
    procedure ReadDigest;
  public
    constructor Create(Stream: TStream; SessKey: TSessionKey);
    destructor Destroy; override;
    function Read(var Buffer; Count: Longint): Longint; override;
    property CheckDigest: Boolean read FCheckDigest;
  end;

  TCompressPacketPipe = class(TPipe)
  private
    FCompAlgo: COMPRESS_ALGOS;
    FZStream: z_stream;
    FBuf: PChar;
    FBufLen: Cardinal;
  public
    constructor Create(Stream: TStream);
    property CompAlgo: COMPRESS_ALGOS read FCompAlgo;
  end;

  TCompressPacketEncodePipe = class(TCompressPacketPipe)
  private
    FWriteHeader: Boolean;
  protected
    procedure FlushBuffer;
  public
    constructor Create(Stream: TStream; CompAlgo: COMPRESS_ALGOS);
    destructor Destroy; override;
    function Write(const Buffer; Count: Longint): Longint; override;
  end;

  TCompressPacketDecodePipe = class(TCompressPacketPipe)
  private
    FReadHeader: Boolean;
  public
    constructor Create(Stream: TStream);
    function Read(var Buffer; Count: Longint): Longint; override;
  end;

implementation

uses
  SysUtils, UTCUnit, Dialogs;

procedure TPipe.DoProgress;
begin
  if Assigned(FOnProgress) then
    FOnProgress(Self);
end;

constructor TPipe.Create(Stream: TStream);
begin
  inherited Create;
  FStream := Stream;
  FOnProgress := nil;
end;

function TPipe.Write(const Buffer; Count: Longint): Longint;
begin
  Result := 0;
  Assert(False);
end;

function TPipe.Read(var Buffer; Count: Longint): Longint;
begin
  Result := 0;
  Assert(False);
end;

procedure TPipe.EncodeFrom(S: TStream);
const
  MaxBufSize = $F000;
var
  Buffer: PByte;
  ReadSize: Integer;
begin
  GetMem(Buffer, MaxBufSize);
  try
    ReadSize := S.Read(Buffer^, MaxBufSize);
    while ReadSize = MaxBufSize do
    begin
      Write(Buffer^, ReadSize);
      ReadSize := S.Read(Buffer^, MaxBufSize);
    end;
    Write(Buffer^, ReadSize);
  finally
    FreeMem(Buffer, MaxBufSize);
  end;
  DoProgress;
end;

procedure TPipe.DecodeTo(S: TStream);
const
  MaxBufSize = $F000;
var
  Buffer: PByte;
  ReadSize: Integer;
begin
  GetMem(Buffer, MaxBufSize);
  try
    ReadSize := Read(Buffer^, MaxBufSize);
    while ReadSize = MaxBufSize do
    begin
      S.WriteBuffer(Buffer^, ReadSize);
      ReadSize := Read(Buffer^, MaxBufSize);
    end;
    S.WriteBuffer(Buffer^, ReadSize);
  finally
    FreeMem(Buffer, MaxBufSize);
  end;
  DoProgress;
end;

function TPipe.DecodeToNull(DecodeSize: Integer): Integer;
var
  Buffer: PByte;
begin
  GetMem(Buffer, DecodeSize);
  try
    Result := Read(Buffer^, DecodeSize);
  finally
    FreeMem(Buffer, DecodeSize);
  end;
end;

constructor TPartialPacketPipe.Create(Stream: TStream; PT: PACKET_TYPES);
begin
  inherited Create(Stream);
  FPacketType := PT;
  FBufStream := TMemoryStream.Create;
end;

destructor TPartialPacketPipe.Destroy;
begin
  FBufStream.Free;
  inherited;
end;

procedure TPartialPacketEncodePipe.WriteHeader;
var
  B: Byte;
begin
  FWriteHeader := True;
  B := $C0 or Byte(FPacketType);
  FStream.WriteBuffer(B, 1);
end;

constructor TPartialPacketEncodePipe.Create(Stream: TStream; PT: PACKET_TYPES);
begin
  inherited Create(Stream, PT);
  FWriteHeader := False;
end;

destructor TPartialPacketEncodePipe.Destroy;
begin
  FlushBuffer;
  inherited;
end;

function TPartialPacketEncodePipe.Write(const Buffer; Count: Longint): Longint;
var
  I, J, K: Integer;
  B: Byte;
  MS: TMemoryStream;
begin
  if not FWriteHeader then
    WriteHeader;

  Result := FBufStream.Write(Buffer, Count);
  
  if FBufStream.Position >= 8192 then
  begin
    J := FBufStream.Position div 8192;
    K := FBufStream.Position mod 8192;

    FBufStream.Position := 0;
    for I := 1 to J do
    begin
      B := $E0 or $0D; (* Partial Body Lengths 8192 oct *)
      FStream.WriteBuffer(B, 1);

      FStream.CopyFrom(FBufStream, 8192); 
    end;

    if K = 0 then
      FBufStream.Clear
    else
    begin
      MS := TMemoryStream.Create;
      try
        MS.CopyFrom(FBufStream, K);
        FBufStream.Clear;
        MS.SaveToStream(FBufStream);
      finally
        MS.Free;
      end;
    end;
  end;

  DoProgress;
end;

procedure TPartialPacketEncodePipe.FlushBuffer;
var
  B: Byte;
  Len: Integer;
begin
  if not FWriteHeader then
    WriteHeader;
    
  Len := FBufStream.Position;
  case Len of
  0..191:
    begin
      B := Len;
      FStream.WriteBuffer(B, 1);
    end;
  192..8383:
    begin
      Len := Len - 192;
      B := (Len div 256) + 192;
      FStream.WriteBuffer(B, 1);
      B := Len mod 256;
      FStream.WriteBuffer(B, 1);
    end;
  else
    Assert(False);
  end;

  FBufStream.SaveToStream(FStream);
end;

constructor TPartialPacketDecodePipe.Create(Stream: TStream; PT: PACKET_TYPES);
begin
  inherited Create(Stream, PT);
  FReadHeader := False;
  FBodyLen := 0;
end;

function TPartialPacketDecodePipe.Read(var Buffer; Count: Longint): Longint;
var
  B, B1, B2: Byte;
  N: Longint;
begin
  Assert(Count <> 0);

  if not FReadHeader then
  begin
    FReadHeader := True;

    FStream.ReadBuffer(B, 1);
    B := B and $3F;
    if not IsPacketType(B) then
      raise EPacketRead.Create(EPACKET_TYPE);
  end;

  while FBufStream.Position < Count do
  begin
    if FBodyLen = 0 then
    begin
      if FStream.Read(B1, 1) <> 1 then
      begin
        Break;
      end;
      case B1 of
      0..191:
        FBodyLen := B1;
      192..223:
        begin
          FStream.ReadBuffer(B2, 1);
          FBodyLen := ((B1 - 192) shl 8) + B2 + 192;
        end;
      224..254:
        FBodyLen := 1 shl (B1 and $1F);
      else
        raise EPacketRead.Create(EPACKET_LEN);
      end;
    end;

    N := FBufStream.Position + FBodyLen;
    if N <= Count then
      N := FBodyLen
    else
      N := Count - FBufStream.Position;

    if N <> 0 then
      FBufStream.CopyFrom(FStream, N);
    FBodyLen := FBodyLen - N;
  end;
  
  FBufStream.Position := 0;
  Result := FBufStream.Read(Buffer, FBufStream.Size);
  FBufStream.Clear;

  DoProgress;
end;

constructor TLiteralPacketPipe.Create(Stream: TStream; LPH: TLiteralPacketHeader);
begin
  inherited Create(Stream);
  FHeader := LPH;
end;

constructor TLiteralPacketEncodePipe.Create(Stream: TStream; LPH: TLiteralPacketHeader);
begin
  inherited Create(Stream, LPH);
  FWriteHeader := False;
end;

function TLiteralPacketEncodePipe.Write(const Buffer; Count: Longint): Longint;
var
  B: Byte;
  I: Integer;
  UTF8: UTF8String;
  UTC: Longword;
begin
  if not FWriteHeader then
  begin
    FWriteHeader := True;

    B := Byte(FHeader.Mode);
    FStream.WriteBuffer(B, 1);

    UTF8 := UTF8Encode(FHeader.FileName);
    B := Length(UTF8);
    FStream.WriteBuffer(B, 1);
    for I := 1 to Length(UTF8) do
    begin
      B := Byte(UTF8[I]);
      FStream.WriteBuffer(B, 1);
    end;

    UTC := DateTimeToUTC(FHeader.Date);
    B := (UTC shr 24) and $FF;
    FStream.WriteBuffer(B, 1);
    B := (UTC shr 16) and $FF;
    FStream.WriteBuffer(B, 1);
    B := (UTC shr  8) and $FF;
    FStream.WriteBuffer(B, 1);
    B := UTC and $FF;
    FStream.WriteBuffer(B, 1);
  end;

  Result := FStream.Write(Buffer, Count);

  DoProgress;
end;

constructor TLiteralPacketDecodePipe.Create(Stream: TStream; LPH: TLiteralPacketHeader);
begin
  inherited Create(Stream, LPH);
  FReadHeader := False;
end;

function TLiteralPacketDecodePipe.Read(var Buffer; Count: Longint): Longint;
var
  B, B1, B2, B3, B4: Byte;
  I: Integer;
  UTC: Longword;
begin
  Assert(Count <> 0);

  if not FReadHeader then
  begin
    FReadHeader := True;

    FStream.ReadBuffer(B, 1);
    if not (B in [$62,$74]) then
      raise EPacketRead.Create(ELITERAL_FILEMODE);
    FHeader.Mode := FILE_MODES(B);

    FStream.ReadBuffer(B, 1);
    FHeader.FileName := '';
    for I := 1 to B do
    begin
      FStream.ReadBuffer(B, 1);
      FHeader.FileName := FHeader.FileName + Char(B);
    end;
    FHeader.FileName := UTF8Decode(FHeader.FileName);

    FStream.ReadBuffer(B1, 1);
    FStream.ReadBuffer(B2, 1);
    FStream.ReadBuffer(B3, 1);
    FStream.ReadBuffer(B4, 1);
    UTC := (B1 shl 24) or (B2 shl 16) or (B3 shl 8) or B4;
    FHeader.Date := UTCToDateTime(UTC);
  end;

  Result := FStream.Read(Buffer, Count);

  DoProgress;
end;

constructor TEncryptPacketsPipe.Create(Stream: TStream; SessKey: TSessionKey);
begin
  inherited Create(Stream);
  FSessKey := SessKey;
  FCH := gcry_cipher_open(FSessKey.CipherAlgo, GCRY_CIPHER_MODE_CFB, GCRY_CIPHER_SECURE);
  FMD := gcry_md_open(GCRY_MD_SHA1, GCRY_MD_FLAG_SECURE);
end;

destructor TEncryptPacketsPipe.Destroy;
begin
  gcry_md_close(FMD);
  gcry_cipher_close(FCH);
  inherited;
end;

procedure TEncryptPacketsEncodePipe.WriteHeader;
var
  B: Byte;
  Buf: array of Byte;
  BufLen, I, RC: Integer;
begin
  FWriteHeader := True;
  
  B := 1;
  FStream.WriteBuffer(B, 1);

  BufLen := gcry_cipher_get_algo_blklen(FSessKey.CipherAlgo);
  BufLen := BufLen + 2;
  SetLength(Buf, BufLen);
  gcry_randomize(PByte(Buf), BufLen -2, GCRY_STRONG_RANDOM);
  Buf[BufLen -2] := Buf[BufLen -4];
  Buf[BufLen -1] := Buf[BufLen -3];
  
  gcry_md_write(FMD, PByte(Buf), BufLen);

  RC := gcry_cipher_setkey(FCH, PByte(FSessKey.KeyData), Length(FSessKey.KeyData));
  Assert(RC = 0);
  RC := gcry_cipher_setiv(FCH, nil, 0);
  Assert(RC = 0);
  RC := gcry_cipher_encrypt(FCH, PByte(Buf), BufLen, nil, 0);
  Assert(RC = 0);
  RC := gcry_cipher_sync(FCH);
  Assert(RC = 0);

  for I := 0 to High(Buf) do
  begin
    B := Buf[I];
    FStream.Write(B, 1);
  end;
end;

procedure TEncryptPacketsEncodePipe.FlushBuffer;
var
  B: Byte;
  PDigest: PByteArray;
  I: Integer;
begin
  if not FWriteHeader then
    WriteHeader;

  B := $D3;
  Write(B, 1);
  B := $14;
  Write(B, 1);

  PDigest := gcry_md_read(FMD, GCRY_MD_SHA1);
  for I := 0 to 19 do
  begin
    B := PDigest[I];
    Write(B, 1);
  end;
end;

constructor TEncryptPacketsEncodePipe.Create(Stream: TStream; SessKey: TSessionKey);
begin
  inherited Create(Stream, SessKey);
  FWriteHeader := False;
end;

destructor TEncryptPacketsEncodePipe.Destroy;
begin
  FlushBuffer;
  inherited;
end;

function TEncryptPacketsEncodePipe.Write(const Buffer; Count: Longint): Longint;
var
  RC: Integer;
begin
  if not FWriteHeader then
    WriteHeader;

  gcry_md_write(FMD, PByte(@Buffer), Count);
  RC := gcry_cipher_encrypt(FCH, PByte(@Buffer), Count, nil, 0);
  Assert(RC = 0);

  Result := FStream.Write(Buffer, Count);

  DoProgress;
end;

procedure TEncryptPacketsDecodePipe.ReadHeader;
var
  B: Byte;
  Buf: array of Byte;
  BufLen, I, RC: Integer;
begin
  FReadHeader := True;

  FStream.ReadBuffer(B, 1);
  if B <> 1 then
    raise EPacketRead.Create(EENCMDC_VER);

  BufLen := gcry_cipher_get_algo_blklen(FSessKey.CipherAlgo);
  BufLen := BufLen + 2;
  SetLength(Buf, BufLen);
  for I := 0 to High(Buf) do
  begin
    FStream.ReadBuffer(B, 1);
    Buf[I] := B;
  end;

  RC := gcry_cipher_setkey(FCH, PByte(FSessKey.KeyData), Length(FSessKey.KeyData));
  Assert(RC = 0);
  RC := gcry_cipher_setiv(FCH, nil, 0);
  Assert(RC = 0);
  RC := gcry_cipher_decrypt(FCH, PByte(Buf), BufLen, nil, 0);
  Assert(RC = 0);
  RC := gcry_cipher_sync(FCH);
  Assert(RC = 0);

  if (Buf[BufLen -2] <> Buf[BufLen -4]) or
     (Buf[BufLen -1] <> Buf[BufLen -3]) then
    raise EPacketRead.Create(ESESSKEY);

  gcry_md_write(FMD, PByte(Buf), BufLen);

  FBufStream.CopyFrom(FStream, 22);
end;

procedure TEncryptPacketsDecodePipe.ReadDigest;
var
  B: Byte;
  Buf: array of Byte;
  RC, I: Integer;
  PDigest: PByteArray;
begin
  FCheckDigest := False;

  SetLength(Buf, 22);
  FBufStream.Position := 0;
  for I := 0 to High(Buf) do
  begin
    FBufStream.ReadBuffer(B, 1);
    Buf[I] := B;
  end;

  RC := gcry_cipher_decrypt(FCH, PByte(Buf), 22, nil, 0);
  Assert(RC = 0);
  if Buf[0] <> $D3 then
    Exit;
  if Buf[1] <> $14 then
    Exit;

  gcry_md_write(FMD, PByte(Buf), 2);
  PDigest := gcry_md_read(FMD, GCRY_MD_SHA1);
  for I := 0 to 19 do
    if PDigest[I] <> Buf[I+2] then
      Exit;

  FCheckDigest := True;
end;

constructor TEncryptPacketsDecodePipe.Create(Stream: TStream; SessKey: TSessionKey);
begin
  inherited Create(Stream, SessKey);
  FReadHeader := False;
  FBufStream := TMemoryStream.Create;
  FCheckDigest := False;
end;

destructor TEncryptPacketsDecodePipe.Destroy;
begin
  FBufStream.Free;
  inherited;
end;

function TEncryptPacketsDecodePipe.Read(var Buffer; Count: Longint): Longint;
var
  ReadLen: Int64;
  RC: Integer;
  MS: TMemoryStream;
begin
  Assert(Count <> 0);

  if not FReadHeader then
    ReadHeader;

  ReadLen := FStream.Read(Buffer, Count);
  FBufStream.WriteBuffer(Buffer, ReadLen);
  FBufStream.Position := 0;

  Result := FBufStream.Read(Buffer, FBufStream.Size - 22);
  RC := gcry_cipher_decrypt(FCH, PByte(@Buffer), Result, nil, 0);
  Assert(RC = 0);
  gcry_md_write(FMD, PByte(@Buffer), Result);

  MS := TMemoryStream.Create;
  try
    MS.CopyFrom(FBufStream, 22);
    FBufStream.Clear;
    MS.SaveToStream(FBufStream);
  finally
    MS.Free;
  end;

  if ReadLen <> Count then
    ReadDigest;

  DoProgress;
end;

constructor TCompressPacketPipe.Create(Stream: TStream);
begin
  inherited Create(Stream);
  FCompAlgo := COMP_NONE;
  FBuf := nil;
  FBufLen := 0;
end;

constructor TCompressPacketEncodePipe.Create(Stream: TStream; CompAlgo: COMPRESS_ALGOS);
var
  RC: Integer;
begin
  inherited Create(Stream);
  FCompAlgo := CompAlgo;
  FWriteHeader := False;

  FBufLen := 8192;
  GetMem(FBuf, FBufLen);

  FZStream.zalloc    := nil;
  FZStream.zfree     := nil;
  FZStream.opaque    := nil;
  FZStream.next_out  := FBuf;
  FZStream.avail_out := FBufLen;

  RC := Z_OK;
  case FCompAlgo of
  COMP_NONE:
    RC := deflateinit(@FZStream, Z_NO_COMPRESSION);
  COMP_ZIP:
    RC := deflateinit2(@FZStream, Z_DEFAULT_COMPRESSION, Z_DEFLATED, -13, 8, Z_DEFAULT_STRATEGY);
  COMP_ZLIB:
    RC := deflateinit(@FZStream, Z_DEFAULT_COMPRESSION);
  else
    Assert(False);
  end;

  if RC <> Z_OK then
    raise EPacketWrite.Create(EZLIB_DEFLATEINIT + '#13ZLIB:' + FZStream.msg + '');
end;

destructor TCompressPacketEncodePipe.Destroy;
begin
  try
    FlushBuffer;
  finally
    deflateend(@FZStream);
    FreeMem(FBuf);
  end;
  inherited;
end;

function TCompressPacketEncodePipe.Write(const Buffer; Count: Longint): Longint;
var
  B: Byte;
  RC: Integer;
begin
  if not FWriteHeader then
  begin
    FWriteHeader := True;
    B := Byte(FCompAlgo);
    FStream.WriteBuffer(B, 1);
  end;

  FZStream.next_in   := PChar(@Buffer);
  FZStream.avail_in  := Count;
  while FZStream.avail_in > 0 do
  begin
    RC := deflate(@FZStream, Z_NO_FLUSH);
    if (RC <> Z_OK) and (RC <> Z_STREAM_END) then
      raise EPacketWrite.Create(EZLIB_DEFLATE + #13 + 'ZLIB:' + FZStream.msg + '');

    if FZStream.avail_out = 0 then
    begin
      FStream.WriteBuffer(FBuf^, FBufLen - FZStream.avail_out);

      FZStream.next_out  := FBuf;
      FZStream.avail_out := FBufLen;
    end;
  end;

  Result := Count;
end;

procedure TCompressPacketEncodePipe.FlushBuffer;
var
  RC: Integer;
begin
  FZStream.next_in := nil;
  FZStream.avail_in := 0;

  repeat
    RC := deflate(@FZStream, Z_FINISH);
    if (RC <> Z_OK) and (RC <> Z_STREAM_END) then
      raise EPacketWrite.Create(EZLIB_DEFLATE + #13 + 'ZLIB:' + FZStream.msg + '');

    FStream.WriteBuffer(FBuf^, FBufLen - FZStream.avail_out);
    
    FZStream.next_out  := FBuf;
    FZStream.avail_out := FBufLen;
  until RC = Z_STREAM_END;
end;

constructor TCompressPacketDecodePipe.Create(Stream: TStream);
begin
  inherited Create(Stream);
  FCompAlgo   := COMP_NONE;
  FReadHeader := False;

  FBufLen := 2048;
  GetMem(FBuf, FBufLen);

  FZStream.zalloc    := nil;
  FZStream.zfree     := nil;
  FZStream.opaque    := nil;
end;

function TCompressPacketDecodePipe.Read(var Buffer; Count: Longint): Longint;
var
  B: Byte;
  RC: Integer;
begin
  RC := Z_OK;

  if not FReadHeader then
  begin
    FReadHeader := True;
    FStream.ReadBuffer(B, 1);
    if not IsCompAlgo(B) then
      raise EPacketRead.Create(ECOMP_ALGO);

    FCompAlgo := COMPRESS_ALGOS(B);

    RC := Z_OK;
    case FCompAlgo of
    COMP_NONE:
      RC := inflateinit(@FZStream);
    COMP_ZIP:
      RC := inflateinit2(@FZStream, -13);
    COMP_ZLIB:
      RC := inflateinit(@FZStream);
    else
      Assert(False);
    end;

    if RC <> Z_OK then
      raise EPacketWrite.Create(EZLIB_DEFLATEINIT + '#13ZLIB:' + FZStream.msg + '');
  end;

  FZStream.next_out  := PChar(@Buffer);
  FZStream.avail_out := Count;

  while (FZStream.avail_out > 0) and (RC <> Z_STREAM_END) do
  begin
    if FZStream.avail_in = 0 then
    begin
      FZStream.avail_in := FStream.Read(FBuf^, FBufLen);
      FZStream.next_in  := FBuf;

      if FZStream.avail_in = 0 then
        Break;
    end;

    RC := inflate(@FZStream, Z_NO_FLUSH);
  end;

  if (RC = Z_STREAM_END) and (FZStream.avail_in > 0) then
    FZStream.avail_in := 0;

  Result := Count - Integer(FZStream.avail_out);
end;

end.
